<?php
class Easy_Custom_Admin_Notification {
    protected $version = '1.1.0';
    public function run() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_notices', array($this, 'display_admin_notice'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_easy_custom_admin_notification_dismiss', array($this, 'handle_dismiss'));
    }
    public function add_admin_menu() {
        add_options_page(
            __('Easy Custom Admin Notification', 'easy-custom-admin-notification'),
            __('Admin Notification', 'easy-custom-admin-notification'),
            'manage_options',
            'easy-custom-admin-notification',
            array($this, 'render_settings_page')
        );
    }
    public function register_settings() {
        register_setting(
            'easy_custom_admin_notification_options',
            'easy_custom_admin_notification_settings',
            array($this, 'sanitize_settings')
        );
        add_settings_section(
            'easy_custom_admin_notification_main',
            __('Notification Settings', 'easy-custom-admin-notification'),
            null,
            'easy-custom-admin-notification'
        );
        add_settings_field(
            'notifications',
            __('Notifications', 'easy-custom-admin-notification'),
            array($this, 'render_notifications_field'),
            'easy-custom-admin-notification',
            'easy_custom_admin_notification_main'
        );
    }
    public function sanitize_settings($input) {
        $sanitized = [];
        $sanitized['notifications'] = [];
        if (isset($input['notifications']) && is_array($input['notifications'])) {
            foreach ($input['notifications'] as $key => $notification) {
                if (empty($notification['title']) && empty($notification['message'])) {
                    continue;
                }
                $sanitized['notifications'][$key] = [
                    'title' => sanitize_text_field($notification['title'] ?? ''),
                    'enabled' => isset($notification['enabled']) ? 1 : 0,
                    'message' => wp_kses_post($notification['message'] ?? ''),
                    'gradient_color_1' => sanitize_hex_color($notification['gradient_color_1'] ?? '#ff6b6b'),
                    'gradient_color_2' => sanitize_hex_color($notification['gradient_color_2'] ?? '#ff8e53'),
                    'dismiss_behavior' => in_array($notification['dismiss_behavior'] ?? 'persistent', ['persistent', 'default', 'always_visible']) ? $notification['dismiss_behavior'] : 'persistent',
                    'roles' => isset($notification['roles']) && is_array($notification['roles']) ? array_map('sanitize_key', $notification['roles']) : [],
                ];
            }
        }
        return $sanitized;
    }
    public function render_notifications_field() {
        $settings = get_option('easy_custom_admin_notification_settings', []);
        $notifications = $settings['notifications'] ?? [];
        $roles = wp_roles()->get_names();
        $max_index = count($notifications) > 0 ? max(array_keys($notifications)) + 1 : 1;
        ?>
        <div id="notifications-wrapper" class="space-y-6" data-max-index="<?php echo esc_attr($max_index); ?>">
            <?php if (!empty($notifications)) : ?>
                <?php foreach ($notifications as $index => $notification) : ?>
                    <div class="notification-item" data-id="<?php echo esc_attr($index); ?>">
                        <details class="bg-white shadow-lg rounded-xl overflow-hidden">
                            <?php
                            // translators: %d is the notification number (e.g., 1, 2, 3).
                            $default_title = sprintf(__('Notification %d (Click to expand/collapse)', 'easy-custom-admin-notification'), esc_html($index + 1));
                            ?>
                            <summary class="p-4 bg-gray-50 hover:bg-gray-100 cursor-pointer font-semibold text-gray-800 flex justify-between items-center">
                                <?php echo esc_html($notification['title'] ?: $default_title); ?>
                                <svg class="w-5 h-5 transform transition-all duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path></svg>
                            </summary>
                            <div class="p-6 space-y-4">
                                <?php $this->render_notification_fields($index, $notification, $roles); ?>
                                <div class="mt-4">
                                    <button type="button" class="remove-notification button" data-id="<?php echo esc_attr($index); ?>"><?php esc_html_e('Remove Notification', 'easy-custom-admin-notification'); ?></button>
                                </div>
                            </div>
                        </details>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
            <button type="button" id="add-notification" class="mt-6 inline-flex items-center px-4 py-2 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"><?php esc_html_e('Add New Notification', 'easy-custom-admin-notification'); ?></button>
        </div>
        <script>
            jQuery(document).ready(function($) {
                console.log('Notification Management Script Loaded');
                $('#notifications-wrapper').on('click', '.remove-notification', function(e) {
                    e.preventDefault();
                    var itemId = $(this).data('id');
                    console.log('Remove Notification clicked for item:', itemId);
                    // translators: Confirmation message for removing a notification.
                    if (confirm('<?php echo esc_js(__('Are you sure you want to remove this notification?', 'easy-custom-admin-notification')); ?>')) {
                        var $item = $(this).closest('.notification-item');
                        $item.animate({ opacity: 0, height: 0 }, 300, function() {
                            $item.remove();
                        });
                    }
                });
                $('#notifications-wrapper').on('toggle', 'details', function() {
                    $(this).find('svg').toggleClass('rotate-180');
                });
                $('#add-notification').on('click', function() {
                    console.log('Add New Notification clicked');
                    var maxIndex = parseInt($('#notifications-wrapper').data('max-index')) || 0;
                    var newIndex = maxIndex;
                    // translators: %d represents the number of the new notification (e.g., 1, 2, 3).
                    var title = '<?php echo esc_js(__('New Notification %d (Click to expand/collapse)', 'easy-custom-admin-notification')); ?>'.replace('%d', newIndex + 1);
                    var template = `
                        <div class="notification-item opacity-0 translate-y-4 transition-all duration-100" data-id="${newIndex}">
                            <details class="bg-white shadow-lg rounded-xl overflow-hidden" open>
                                <summary class="p-4 bg-gray-50 hover:bg-gray-100 cursor-pointer font-semibold text-gray-800 flex justify-between items-center">
                                    ${title}
                                    <svg class="w-5 h-5 transform transition-all duration-200 rotate-180" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path></svg>
                                </summary>
                                <div class="p-6 space-y-4">
                                    <div class="space-y-4">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Notification Title', 'easy-custom-admin-notification'); ?></label>
                                            <input type="text" name="easy_custom_admin_notification_settings[notifications][${newIndex}][title]" class="mt-2 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" placeholder="<?php esc_attr_e('Enter a title for this notification', 'easy-custom-admin-notification'); ?>" />
                                            <p class="mt-1 text-xs text-gray-500"><?php esc_html_e('This title is used to identify the notification in the settings panel.', 'easy-custom-admin-notification'); ?></p>
                                        </div>
                                        <hr class="my-4 border-gray-200" />
                                        <div class="flex items-center">
                                            <input type="checkbox" name="easy_custom_admin_notification_settings[notifications][${newIndex}][enabled]" value="1" checked class="h-4 w-4 text-blue-600 border border-gray-300 rounded-sm focus:ring-2 focus:ring-blue-500" />
                                            <label class="ml-2 text-sm font-medium text-gray-700"><?php esc_html_e('Enable Notification', 'easy-custom-admin-notification'); ?></label>
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Notification Message', 'easy-custom-admin-notification'); ?></label>
                                            <textarea name="easy_custom_admin_notification_settings[notifications][${newIndex}][message]" id="notification_message_${newIndex}" class="mt-2 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" rows="8"></textarea>
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Gradient Color 1', 'easy-custom-admin-notification'); ?></label>
                                            <input type="text" name="easy_custom_admin_notification_settings[notifications][${newIndex}][gradient_color_1]" value="#ff6b6b" class="color-field mt-2 block w-full max-w-xs px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" data-default-color="#ff6b6b" />
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Gradient Color 2', 'easy-custom-admin-notification'); ?></label>
                                            <input type="text" name="easy_custom_admin_notification_settings[notifications][${newIndex}][gradient_color_2]" value="#ff8e53" class="color-field mt-2 block w-full max-w-xs px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" data-default-color="#ff8e53" />
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Dismiss Behavior', 'easy-custom-admin-notification'); ?></label>
                                            <select name="easy_custom_admin_notification_settings[notifications][${newIndex}][dismiss_behavior]" class="mt-2 block w-full max-w-sm px-3 py-2 border border-gray-300 rounded-md shadow-sm text-gray-900 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm">
                                                <option value="persistent"><?php esc_html_e('Persistent (Dismiss once, stays hidden)', 'easy-custom-admin-notification'); ?></option>
                                                <option value="default"><?php esc_html_e('Default (Reappears on new sessions)', 'easy-custom-admin-notification'); ?></option>
                                                <option value="always_visible"><?php esc_html_e('Always Visible (Cannot be dismissed)', 'easy-custom-admin-notification'); ?></option>
                                            </select>
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Show Notification to Roles', 'easy-custom-admin-notification'); ?></label>
                                            <select multiple="multiple" name="easy_custom_admin_notification_settings[notifications][${newIndex}][roles][]" class="mt-2 block w-full max-w-sm px-3 py-2 border border-gray-300 rounded-md shadow-sm text-gray-900 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm">
                                                <?php foreach ($roles as $role_value => $role_name) : ?>
                                                    <option value="<?php echo esc_attr($role_value); ?>">
                                                        <?php echo esc_html($role_name); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                            <p class="mt-1 text-xs text-gray-500"><?php esc_html_e('Select the user roles that will see this notification. Leave empty to show to all roles.', 'easy-custom-admin-notification'); ?></p>
                                        </div>
                                    </div>
                                    <div class="mt-4">
                                        <button type="button" class="remove-notification button" data-id="${newIndex}"><?php esc_html_e('Remove Notification', 'easy-custom-admin-notification'); ?></button>
                                    </div>
                                </div>
                            </details>
                        </div>`;
                    $('#notifications-wrapper').append(template);
                    var $newItem = $(`[data-id="${newIndex}"]`);
                    $newItem.animate({ opacity: 1, transform: 'translateY(0)' }, 300);
                    $newItem.find('.color-field').wpColorPicker();
                    console.log('Attempting to initialize TinyMCE for textarea:', `#notification_message_${newIndex}`);
                    setTimeout(() => {
                        if (typeof tinymce !== 'undefined' && typeof wp !== 'undefined' && wp.editor && wp.editor.initialize) {
                            wp.editor.initialize(`notification_message_${newIndex}`, {
                                tinymce: {
                                    wpautop: true,
                                    plugins: 'charmap colorpicker compat3x directionality fullscreen hr image link lists media paste tabfocus textcolor wordpress wpautoresize wpdialogs wpeditimage wpemoji wpgallery wplink wptextpattern wpview',
                                    toolbar1: 'formatselect bold italic bullist numlist blockquote alignleft aligncenter alignright link unlink wp_more wp_adv',
                                    toolbar2: 'strikethrough hr forecolor backcolor emoticons',
                                    media_buttons: false,
                                    teeny: false,
                                    textarea_rows: 8
                                },
                                quicktags: true
                            });
                            console.log('TinyMCE successfully initialized for:', `notification_message_${newIndex}`);
                        } else {
                            console.error('TinyMCE initialization failed. wp.editor:', !!wp.editor, 'tinymce:', !!tinymce);
                        }
                    }, 100);
                    $('#notifications-wrapper').data('max-index', newIndex + 1);
                    console.log('New notification added with index:', newIndex);
                });
            });
        </script>
        <?php
    }
    protected function render_notification_fields($index, $notification, $roles) {
        $title = $notification['title'] ?? '';
        $enabled = $notification['enabled'] ?? 0;
        $message = $notification['message'] ?? '';
        $color1 = $notification['gradient_color_1'] ?? '#ff6b6b';
        $color2 = $notification['gradient_color_2'] ?? '#ff8e53';
        $dismiss_behavior = $notification['dismiss_behavior'] ?? 'persistent';
        $selected_roles = $notification['roles'] ?? [];
        ?>
        <div class="space-y-4">
            <div>
                <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Notification Title', 'easy-custom-admin-notification'); ?></label>
                <input type="text" name="easy_custom_admin_notification_settings[notifications][<?php echo esc_attr($index); ?>][title]" value="<?php echo esc_attr($title); ?>" class="mt-2 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" placeholder="<?php esc_attr_e('Enter a title for this notification', 'easy-custom-admin-notification'); ?>" />
                <p class="mt-1 text-xs text-gray-500"><?php esc_html_e('This title is used to identify the notification in the settings panel.', 'easy-custom-admin-notification'); ?></p>
            </div>
            <hr class="my-4 border-gray-200" />
            <div class="flex items-center">
                <input type="checkbox" name="easy_custom_admin_notification_settings[notifications][<?php echo esc_attr($index); ?>][enabled]" value="1" <?php checked(1, $enabled); ?> class="h-4 w-4 text-blue-600 border border-gray-300 rounded-sm focus:ring-2 focus:ring-blue-500" />
                <label class="ml-2 text-sm font-medium text-gray-700"><?php esc_html_e('Enable Notification', 'easy-custom-admin-notification'); ?></label>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Notification Message', 'easy-custom-admin-notification'); ?></label>
                <?php
                wp_editor($message, 'notification_message_' . $index, [
                    'textarea_name' => 'easy_custom_admin_notification_settings[notifications][' . $index . '][message]',
                    'media_buttons' => false,
                    'textarea_rows' => 8,
                    'teeny' => false,
                    'editor_class' => 'mt-2 block w-full border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500'
                ]);
                ?>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Gradient Color 1', 'easy-custom-admin-notification'); ?></label>
                <input type="text" name="easy_custom_admin_notification_settings[notifications][<?php echo esc_attr($index); ?>][gradient_color_1]" value="<?php echo esc_attr($color1); ?>" class="color-field mt-2 block w-full max-w-xs px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" data-default-color="#ff6b6b" />
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Gradient Color 2', 'easy-custom-admin-notification'); ?></label>
                <input type="text" name="easy_custom_admin_notification_settings[notifications][<?php echo esc_attr($index); ?>][gradient_color_2]" value="<?php echo esc_attr($color2); ?>" class="color-field mt-2 block w-full max-w-xs px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" data-default-color="#ff8e53" />
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Dismiss Behavior', 'easy-custom-admin-notification'); ?></label>
                <select name="easy_custom_admin_notification_settings[notifications][<?php echo esc_attr($index); ?>][dismiss_behavior]" class="mt-2 block w-full max-w-sm px-3 py-2 border border-gray-300 rounded-md shadow-sm text-gray-900 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm">
                    <option value="persistent" <?php selected('persistent', $dismiss_behavior); ?>><?php esc_html_e('Persistent (Dismiss once, stays hidden)', 'easy-custom-admin-notification'); ?></option>
                    <option value="default" <?php selected('default', $dismiss_behavior); ?>><?php esc_html_e('Default (Reappears on new sessions)', 'easy-custom-admin-notification'); ?></option>
                    <option value="always_visible" <?php selected('always_visible', $dismiss_behavior); ?>><?php esc_html_e('Always Visible (Cannot be dismissed)', 'easy-custom-admin-notification'); ?></option>
                </select>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700"><?php esc_html_e('Show Notification to Roles', 'easy-custom-admin-notification'); ?></label>
                <select multiple="multiple" name="easy_custom_admin_notification_settings[notifications][<?php echo esc_attr($index); ?>][roles][]" class="mt-2 block w-full max-w-sm px-3 py-2 border border-gray-300 rounded-md shadow-sm text-gray-900 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 sm:text-sm">
                    <?php foreach ($roles as $role_value => $role_name) : ?>
                        <option value="<?php echo esc_attr($role_value); ?>" <?php selected(in_array($role_value, $selected_roles)); ?>>
                            <?php echo esc_html($role_name); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <p class="mt-1 text-xs text-gray-500"><?php esc_html_e('Select the user roles that will see this notification. Leave empty to show to all roles.', 'easy-custom-admin-notification'); ?></p>
            </div>
        </div>
        <?php
    }
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <header class="ecan-header bg-gradient-to-r from-blue-600 to-blue-800 text-white rounded-xl shadow-lg p-6 mb-8 flex items-center space-x-4">
                <svg class="w-10 h-10" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"></path>
                </svg>
                <h1 class="text-3xl font-semibold"><?php echo esc_html(get_admin_page_title()); ?></h1>
            </header>
            <div class="settings-section bg-white rounded-xl shadow-md p-6 mb-8">
                <form method="post" action="options.php">
                    <?php
                    settings_fields('easy_custom_admin_notification_options');
                    do_settings_sections('easy-custom-admin-notification');
                    submit_button(__('Save Changes', 'easy-custom-admin-notification'), 'primary inline-flex items-center px-4 py-2 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2');
                    ?>
                </form>
            </div>
        </div>
        <?php
    }
    public function display_admin_notice() {
        $screen = get_current_screen();
        if ($screen->id !== 'dashboard') {
            return;
        }
        $settings = get_option('easy_custom_admin_notification_settings', []);
        $notifications = $settings['notifications'] ?? [];
        if (empty($notifications)) {
            return;
        }
        $user = wp_get_current_user();
        $user_roles = (array) $user->roles;
        foreach ($notifications as $index => $notification) {
            if (empty($notification['enabled'])) {
                continue;
            }
            $message = $notification['message'] ?? '';
            if (empty($message)) {
                continue;
            }
            $selected_roles = $notification['roles'] ?? [];
            if (!empty($selected_roles) && empty(array_intersect($user_roles, $selected_roles))) {
                continue;
            }
            $dismiss_behavior = $notification['dismiss_behavior'] ?? 'persistent';
            if ($dismiss_behavior !== 'always_visible') {
                $user_id = get_current_user_id();
                $dismissed = get_user_meta($user_id, 'easy-custom-admin-notification_dismissed_' . $index, true);
                if ($dismiss_behavior === 'persistent' && $dismissed) {
                    continue;
                }
                if ($dismiss_behavior === 'default' && $dismissed === wp_get_session_token()) {
                    continue;
                }
            }
            $dismiss_class = $dismiss_behavior !== 'always_visible' ? 'is-dismissible' : '';
            $color1 = esc_attr($notification['gradient_color_1'] ?? '#ff6b6b');
            $color2 = esc_attr($notification['gradient_color_2'] ?? '#ff8e53');
            ?>
            <div class="custom-notice notice <?php echo esc_attr($dismiss_class); ?> rounded-md shadow-lg text-white" id="easy-custom-admin-notification-<?php echo esc_attr($index); ?>" style="background: linear-gradient(135deg, <?php echo esc_attr($color1); ?>, <?php echo esc_attr($color2); ?>);">
                <div class="flex-grow"><?php echo wp_kses_post($message); ?></div>
                <?php if ($dismiss_behavior !== 'always_visible') : ?>
                    <button type="button" class="notice-dismiss"><span class="screen-reader-text"><?php esc_html_e('Dismiss this notice', 'easy-custom-admin-notification'); ?></span></button>
                <?php endif; ?>
            </div>
            <?php
        }
    }
    public function handle_dismiss() {
        check_ajax_referer('easy_custom_admin_notification_dismiss', 'nonce');
        $notification_id = isset($_POST['notification_id']) ? sanitize_key($_POST['notification_id']) : '';
        if ($notification_id === '') {
            wp_send_json_error();
        }
        $user_id = get_current_user_id();
        $settings = get_option('easy_custom_admin_notification_settings', []);
        $notifications = $settings['notifications'] ?? [];
        if (!isset($notifications[$notification_id])) {
            wp_send_json_error();
        }
        $dismiss_behavior = $notifications[$notification_id]['dismiss_behavior'] ?? 'persistent';
        if ($dismiss_behavior === 'persistent') {
            update_user_meta($user_id, 'easy-custom-admin-notification_dismissed_' . $notification_id, '1');
        } elseif ($dismiss_behavior === 'default') {
            update_user_meta($user_id, 'easy-custom-admin-notification_dismissed_' . $notification_id, wp_get_session_token());
        }
        wp_send_json_success();
    }
    public function enqueue_scripts($hook) {
        if ($hook === 'index.php' || $hook === 'settings_page_easy-custom-admin-notification') {
            wp_enqueue_style(
                'easy-custom-admin-notification',
                plugin_dir_url(__FILE__) . '../assets/css/admin-notification.css',
                ['wp-color-picker'],
                $this->version
            );
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_editor();
            wp_enqueue_script(
                'easy-custom-admin-notification',
                plugin_dir_url(__FILE__) . '../assets/js/admin-notification.js',
                ['wp-color-picker', 'jquery', 'wp-tinymce', 'editor'],
                $this->version,
                true
            );
            if ($hook === 'index.php') {
                wp_localize_script(
                    'easy-custom-admin-notification',
                    'easyCustomAdminNotification',
                    [
                        'ajax_url' => admin_url('admin-ajax.php'),
                        'nonce' => wp_create_nonce('easy_custom_admin_notification_dismiss'),
                    ]
                );
            }
        }
    }
}
?>