<?php 

/**
 * The public-facing functionality of the plugin.
 */
class Easyavup_Public {
    /**
     * Initialize plugin.
     */
    public function init() {
        // Hook into get_avatar filter to use custom avatar
        add_filter( 'get_avatar', array( $this, 'easyavup_get_custom_avatar' ), 9999999999, 6 );
    }

    /**
     * Get custom avatar if user has uploaded one.
     *
     * @param string            $avatar      Image tag for the user's avatar.
     * @param int|WP_User|string $id_or_email The Gravatar to retrieve.
     * @param int               $size        Square avatar width and height in pixels to retrieve.
     * @param string            $default     URL for the default image or a default type.
     * @param string            $alt         Alternative text to use in the avatar image tag.
     * @return string Modified avatar HTML.
     */
    public function easyavup_get_custom_avatar( $avatar, $id_or_email, $size, $default, $alt, $args ) {
        $defaults = array(
            // get_avatar_data() args.
            'size'          => 96,
            'height'        => null,
            'width'         => null,
            'default'       => get_option( 'avatar_default', 'mystery' ),
            'force_default' => false,
            'rating'        => get_option( 'avatar_rating' ),
            'scheme'        => null,
            'alt'           => '',
            'class'         => null,
            'force_display' => false,
            'loading'       => null,
            'fetchpriority' => null,
            'decoding'      => null,
            'extra_attr'    => '',
        );

        if ( empty( $args ) ) {
            $args = array();
        }

        $user_id = 0;

        // Get user ID from various input types
        if ( is_numeric( $id_or_email ) ) {
            $user_id = absint( $id_or_email );
        } elseif ( is_object( $id_or_email ) && ! empty( $id_or_email->user_id ) ) {
            $user_id = absint( $id_or_email->user_id );
        } elseif ( is_object( $id_or_email ) && ! empty( $id_or_email->ID ) ) {
            $user_id = absint( $id_or_email->ID );
        } elseif ( is_string( $id_or_email ) && is_email( $id_or_email ) ) {
            $user = get_user_by( 'email', $id_or_email );
            $user_id = $user ? $user->ID : 0;
        }

        if ( ! $user_id ) {
            return $avatar;
        }

        $easyavup_avatar_id = get_user_meta( $user_id, 'easyavup_avatar_id', true );

        if ( ! $easyavup_avatar_id ) {
            return $avatar;
        }

        // Verify attachment exists and is an image
        if ( ! wp_attachment_is_image( $easyavup_avatar_id ) ) {
            delete_user_meta( $user_id, 'easyavup_avatar_id' );
            return $avatar;
        }

        $image_url = wp_get_attachment_image_url( $easyavup_avatar_id, array( $size, $size ) );

        if ( ! $image_url ) {
            delete_user_meta( $user_id, 'easyavup_avatar_id' );
            return $avatar;
        }
        

        $args['size']    = (int) $size;
        $args = wp_parse_args( $args, $defaults );
        
        if ( empty( $args['height'] ) ) {
            $args['height'] = $args['size'];
        }
        if ( empty( $args['width'] ) ) {
            $args['width'] = $args['size'];
        }

        $url2x = wp_get_attachment_image_url( $easyavup_avatar_id, array( $size * 2, $size * 2 ) );

        $class = array( 'avatar', 'avatar-' . $size, 'photo', 'easyavup_avatar' );

        if ( $args['class'] ) {
            if ( is_array( $args['class'] ) ) {
                $class = array_merge( $class, $args['class'] );
            } else {
                $class[] = $args['class'];
            }
        }

        $extra_attr = $args['extra_attr'];

        if ( in_array( $args['loading'], array( 'lazy', 'eager' ), true )
            && ! preg_match( '/\bloading\s*=/', $extra_attr )
        ) {
            if ( ! empty( $extra_attr ) ) {
                $extra_attr .= ' ';
            }

            $extra_attr .= "loading='{$args['loading']}'";
        }

        if ( in_array( $args['fetchpriority'], array( 'high', 'low', 'auto' ), true )
            && ! preg_match( '/\bfetchpriority\s*=/', $extra_attr )
        ) {
            if ( ! empty( $extra_attr ) ) {
                $extra_attr .= ' ';
            }

            $extra_attr .= "fetchpriority='{$args['fetchpriority']}'";
        }

        if ( in_array( $args['decoding'], array( 'async', 'sync', 'auto' ), true )
            && ! preg_match( '/\bdecoding\s*=/', $extra_attr )
        ) {
            if ( ! empty( $extra_attr ) ) {
                $extra_attr .= ' ';
            }

            $extra_attr .= "decoding='{$args['decoding']}'";
        }

        return sprintf(
            "<img alt='%s' src='%s' srcset='%s' class='%s' height='%d' width='%d' %s/>",
            esc_attr( $args['alt'] ),
            esc_url( $image_url ),
            esc_url( $url2x ) . ' 2x',
            esc_attr( implode( ' ', $class ) ),
            (int) $args['height'],
            (int) $args['width'],
            $extra_attr
        );
    }

    public function easyavup_enqueue_scripts() {
        wp_enqueue_style( 'easyavup-style', EASYAVUP__URL . '/public/assets/css/easyavup.css', [], EASYAVUP__VERSION, 'all' );

        if ( is_user_logged_in() ) {
            $dependency = ['jquery'];
            // Only enqueue if user can upload files
            if ( current_user_can( 'upload_files' ) ) {
                wp_enqueue_media();
                $dependency[] = 'media-upload';
                $dependency[] = 'media-views';
            }
            wp_enqueue_script( 'easyavup-script', EASYAVUP__URL . '/public/assets/js/easyavup.js', $dependency, EASYAVUP__VERSION, true );

            wp_localize_script('easyavup-script', 'easyavup_frontend_localize', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'easyavup_strings' => array(
                    'easyavup_selectImage' => esc_html__( 'Select Profile Picture', 'easy-avatar-upload' ),
                    'easyavup_useImage' => esc_html__( 'Use this image', 'easy-avatar-upload' ),
                    'easyavup_updating' => esc_html__( 'Updating...', 'easy-avatar-upload' ),
                    'easyavup_success' => esc_html__( 'Profile picture updated successfully!', 'easy-avatar-upload' ),
                    'easyavup_removed' => esc_html__( 'Profile picture removed successfully!', 'easy-avatar-upload' ),
                    'easyavup_removed_btn' => esc_html__( 'Removed', 'easy-avatar-upload' ),
                    'easyavup_removing_btn' => esc_html__( 'Removing...', 'easy-avatar-upload' ),
                    'easyavup_updated' => esc_html__( 'Updated', 'easy-avatar-upload' ),
                ),
                'easyavup_upload_limit' => wp_max_upload_size() ?? ( 2 * 1024 * 1024 )
            ]);
        }
    }


    /**
     * Display user avatar with optional edit functionality
     * 
     * @param array $atts Shortcode attributes
     * @return string HTML output or empty string if user not logged in
     */
    public function easyavup_easyavup_avatar( $atts ) {
 
       // Early return if user not logged in
        if ( ! is_user_logged_in() ) {
            return '';
        }

        // Parse and validate attributes with proper defaults
        $attributes = shortcode_atts( 
            array(
                'size'  => 45,
                'mode'  => 'view',
                'shape' => 50,
            ), 
            $atts, 
            'easyavup_avatar' 
        );
 
        $size = absint( $attributes['size'] );

        $mode = sanitize_key( $attributes['mode'] );
        $allowed_modes = array( 'edit', 'view', '' );
        if ( ! in_array( $mode, $allowed_modes, true ) ) {
            $mode = '';
        }

        $shape = absint( $attributes['shape'] );
        $shape = max( 0, min( 100, $shape ) ); 

        $css_class = 'easyavup_front';
        $data_action = '';
        
        if ( ! empty( $mode ) && 'edit' == $mode ) {
            $css_class .= ' easyavup_open_action';
            $data_action = esc_attr( $mode );
        }

        // Get current user ID once
        $user_id = get_current_user_id();

        // Build avatar with proper error handling
        $avatar = get_avatar( $user_id, $size, '', '', array( 'loading' => 'lazy' ) );

        $parent_styles = [
            'height : ' . sprintf('%dpx', intval( $size )),
            'width : ' . sprintf('%dpx', intval( $size ))
        ];

        $implode_styles = implode(';', $parent_styles);

        ob_start();
    ?>
        <div class="<?php echo esc_attr( $css_class ); ?>" data-action="<?php echo esc_attr( $data_action ); ?>" style="<?php echo esc_attr( $implode_styles ); ?>">
            <div class="easyavup_img_wrap" style="border-radius: <?php echo esc_attr( intval( $shape ) ); ?>%">
                <?php echo wp_kses_post( $avatar ); ?>
            </div>

            <?php if ( 'edit' === $mode ): ?>
                <div class="easyavup_upload">
                    <svg fill="none" stroke="white" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"></path>
                    </svg>              
                </div>               
            <?php endif ?>
        </div>

        <?php 
        return ob_get_clean();
    }

    public function add_poupup_html() {

        if ( !is_user_logged_in() ) {
            return;
        }

        $nonce = wp_create_nonce( 'easyavup_profile_picture_nonce_' . get_current_user_id() );
        $easyavup_avatar_id = get_user_meta( get_current_user_id(), 'easyavup_avatar_id', true );
        $easyavup_avatar_id = $easyavup_avatar_id ? absint( $easyavup_avatar_id ) : 0;

        // Verify attachment still exists
        if ( $easyavup_avatar_id && ! wp_attachment_is_image( $easyavup_avatar_id ) ) {
            delete_user_meta( get_current_user_id(), 'easyavup_avatar_id' );
            $easyavup_avatar_id = 0;
        }

        ?>

        <div class="easyavup_overlay"></div>
        
        <div class="easyavup_popup">
            <button class="easyavup_close-btn easyavup_closeBtn">×</button>
            
            <div class="easyavup_popup-header">
                <h2 class="easyavup_popup-title"><?php esc_html_e( 'Update Profile Picture', 'easy-avatar-upload' ); ?></h2>
                <p class="easyavup_popup-subtitle"><?php esc_html_e( 'Choose a photo that represents you well', 'easy-avatar-upload' ); ?></p>
            </div>
            
            <div class="easyavup_profile-section">
                <div class="easyavup_profile-image">
                    <?php echo get_avatar( get_current_user_id(), 120 ); ?>
                </div>
            </div>

            <input type="hidden" name="easyavup_nonce" id="easyavup-nonce" value="<?php echo esc_attr( $nonce ); ?>">

            <?php if ( !current_user_can( 'upload_files' ) ): ?>
                <input type="hidden" name="easyavup_avatar_id" id="easyavup-avatar-id" value="<?php echo esc_attr( $easyavup_avatar_id ); ?>">
                <input type="file" id="profile-picture-upload_manual" accept="image/*" style="display: none;">
            <?php endif ?>
            <div class="easyavup_success-message" id="easyavup_successMessage"></div>
            <div class="easyavup_action-buttons">
                <button class="easyavup_btn easyavup_btn-primary easyavup_changePictureBtn">
                    <svg fill="none" stroke="white" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"></path>
                    </svg>   
                    <?php esc_html_e( 'Change Picture', 'easy-avatar-upload' ); ?>
                </button>
                
                <button class="easyavup_btn easyavup_btn-danger easyavup_deleteBtn" data-id="<?php echo esc_attr( $easyavup_avatar_id ); ?>">
                    <svg class="easyavup_icon" viewBox="0 0 24 24">
                        <path d="M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z"/>
                    </svg>
                    <?php esc_html_e( 'Delete Picture', 'easy-avatar-upload' ); ?>
                </button>
                
                <button class="easyavup_btn easyavup_btn-secondary easyavup_cancelBtn">
                <?php esc_html_e( 'Cancel', 'easy-avatar-upload' ); ?>
                </button>
            </div>
        </div>
    <?php 
    }
}

 ?>