<?php

/**
 * The AJAX functionality of the plugin.
 */
class Easyavup_Ajax {
	
    public function easyavup_save_profile_picture() {
        if (!wp_doing_ajax()) {
            wp_die(esc_html__('Invalid request.', 'easy-avatar-upload'));
        }

        // Verify nonce
        $nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
        $user_id = get_current_user_id();

        if ( ! wp_verify_nonce( $nonce, 'easyavup_profile_picture_nonce_' . $user_id ) ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Security check failed.', 'easy-avatar-upload' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'edit_user', $user_id ) ) {
            wp_send_json_error( array( 'message' => esc_html__( 'You do not have permission to edit this user.', 'easy-avatar-upload' ) ) );
        }

        $attachment_id = 0;

        if ( isset($_FILES['directfile']) && is_array($_FILES['directfile']) ) {
            $file = $_FILES['directfile'];
        } else {
            $file = '';
        }

        if ( !empty($file) && isset($file['name']) && $file['name'] !== '' ) {

            // Handle file-specific errors.
            if ($file['error'] !== UPLOAD_ERR_OK) {
                wp_send_json_error( array( 'message' => 'File upload error: ' . $file['error'] ) );
            }

            if ( ! function_exists( 'wp_handle_upload' ) ) {
                require_once( ABSPATH . 'wp-admin/includes/file.php' );
            }

            if ( !function_exists( 'wp_generate_attachment_metadata' ) ) {
                require_once( ABSPATH . 'wp-admin/includes/image.php' );
            }

            if ( !function_exists( 'wp_insert_attachment' ) ) {
                require_once( ABSPATH . 'wp-admin/includes/media.php' );
            }

            $upload_overrides = array( 'test_form' => false );
            $uploaded_file = wp_handle_upload( $file, $upload_overrides );

            if ( isset( $uploaded_file['error'] ) ) {
                wp_send_json_error( array( 'message' => 'Upload failed: ' . $uploaded_file['error'] ) );
            }

            // Check file type to ensure it's an image.
            $wp_filetype = wp_check_filetype(basename($uploaded_file['file']), null );
            $allowed_types = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');

            if (!in_array($wp_filetype['type'], $allowed_types)) {
                wp_delete_file($uploaded_file['file']);
                wp_send_json_error(array('message' => 'Invalid file type. Please upload an image.'));
            }

            // Create the attachment post.
            $attachment = array(
                'post_mime_type' => $uploaded_file['type'],
                'post_title'     => sanitize_file_name( $file['name'] ),
                'post_content'   => '',
                'post_status'    => 'inherit',
                'post_author'    => $user_id
            );

            $attachment_id = wp_insert_attachment( $attachment, $uploaded_file['file'], 0 );

            if ( is_wp_error( $attachment_id ) ) {
                wp_delete_file( $uploaded_file['file'] );
                wp_send_json_error( array( 'message' => 'Failed to create attachment.' ) );
            }

            // Generate metadata and update attachment.
            $attachment_data = wp_generate_attachment_metadata( $attachment_id, $uploaded_file['file'] );
            wp_update_attachment_metadata( $attachment_id, $attachment_data );


        } elseif ( !empty($_POST['attachment_id'] ) ) {
            $attachment_id = absint( $_POST['attachment_id'] ?? 0 );
        }

        if ( $attachment_id <= 0 ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Invalid attachment ID.', 'easy-avatar-upload' ) ) );
        }

        // Verify attachment exists and is an image
        if ( ! wp_attachment_is_image( $attachment_id ) ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Please select a valid image file.', 'easy-avatar-upload' ) ) );
        }

        // Check if user owns the attachment or can edit others' media
        $attachment = get_post( $attachment_id );
        if ( ! $attachment ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Attachment not found.', 'easy-avatar-upload' ) ) );
        }            
        

        if (!current_user_can('edit_post', $attachment_id) && $attachment->post_author != $user_id) {
            wp_send_json_error(array('message' => esc_html__('You do not have permission to use this image.', 'easy-avatar-upload')));
        }

        delete_user_meta( $user_id, 'easyavup_avatar_id' );
        $updated = update_user_meta( $user_id, 'easyavup_avatar_id', $attachment_id );
        
        if ( false === $updated ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Failed to update profile picture.', 'easy-avatar-upload' ) ) );
        }

        $image_url = $attachment_id ? wp_get_attachment_image_url( $attachment_id, 'full' ) : get_avatar_url( $user_id, array( 'size' => 150 ) );

        wp_send_json_success([
            'url' => $image_url,
            'id' => $attachment_id
        ]);

    }


    public function easyavup_remove_profile_picture() {

        // Verify nonce
        $nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
        $user_id = get_current_user_id();

        if ( ! wp_verify_nonce( $nonce, 'easyavup_profile_picture_nonce_' . $user_id ) ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Security check failed.', 'easy-avatar-upload' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'edit_user', $user_id ) ) {
            wp_send_json_error( array( 'message' => esc_html__( 'You do not have permission to edit this user.', 'easy-avatar-upload' ) ) );
        }

        // Remove user meta
        $deleted = delete_user_meta( $user_id, 'easyavup_avatar_id' );

        if ( false === $deleted ) {
            wp_send_json_error( array( 'message' => esc_html__( 'Failed to remove profile picture.', 'easy-avatar-upload' ) ) );
        }

        wp_send_json_success([
            'default' => get_avatar_url( $user_id, [
                'size' => '150'
            ])
        ]);
    }
}

?>