<?php 

/**
 * The admin-specific functionality of the plugin.
 */
class Easyavup_Admin {
    /**
     * Adds the profile picture upload field to the user profile page.
     *
     * @param WP_User $user The user object.
     */
    public function easyavup_add_profile_field( $user ) {
        // Check if user can edit their profile or if current user can edit users
        if ( ! current_user_can( 'edit_user', $user->ID ) ) {
            return;
        }

        $current_user = wp_get_current_user();
        if ( !in_array('administrator', $current_user->roles) && $current_user->ID != $user->ID ) {
            return;
        }

        $easyavup_avatar_id = get_user_meta( $user->ID, 'easyavup_avatar_id', true );
        $easyavup_avatar_id = $easyavup_avatar_id ? absint( $easyavup_avatar_id ) : 0;

        // Verify attachment still exists
        if ( $easyavup_avatar_id && ! wp_attachment_is_image( $easyavup_avatar_id ) ) {
            delete_user_meta( $user->ID, 'easyavup_avatar_id' );
            $easyavup_avatar_id = 0;
        }

        $image_url = $easyavup_avatar_id ? wp_get_attachment_image_url( $easyavup_avatar_id, 'full' ) : get_avatar_url( $user->ID, array( 'size' => 150 ) );

        $file_size_formatted = size_format( wp_max_upload_size() );
        if ( ! $file_size_formatted ) {
            $file_size_formatted = '2MB';
        }

        $nonce = wp_create_nonce( 'easyavup_profile_picture_nonce_' . $user->ID );

        ?>

        <div class="easyavup_upload-container" id="easyavup_profile_back-wrap">
            <div class="easyavup_profile_heading">
                <h2><?php esc_html_e( 'Update Profile Picture', 'easy-avatar-upload' ); ?></h2>
                <p><?php esc_html_e( 'Choose a photo that represents you well', 'easy-avatar-upload' ); ?></p>
            </div>

            <div class="easyavup_current-avatar">
                <div class="easyavup_avatar-circle">
                    <img src="<?php echo esc_attr( $image_url ); ?>" alt="<?php esc_attr_e( 'Current profile picture', 'easy-avatar-upload' ); ?>">
                </div>
            </div>

            <div class="easyavup_upload-area">
                <div class="easyavup_upload-icon">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"/>
                    </svg>
                </div>
                <div class="easyavup_upload-text">
                    <h3><?php esc_html_e( 'Upload File Here', 'easy-avatar-upload' ); ?></h3>
                </div>
                <div class="easyavup_upload-hint">
                    <?php
                        printf(
                            /* translators: %s: Maximum file size */
                            esc_html__( 'PNG, JPG up to %s', 'easy-avatar-upload' ),
                            esc_html( $file_size_formatted )
                        );
                    ?>
                </div>
            </div>

            <input type="hidden" name="easyavup_avatar_id" id="easyavup-avatar-id" value="<?php echo esc_attr( $easyavup_avatar_id ); ?>">
            <input type="hidden" name="easyavup_nonce" id="easyavup-nonce" value="<?php echo esc_attr( $nonce ); ?>">

            <?php if ( !current_user_can( 'upload_files' ) ): ?>
                <input type="file" id="profile-picture-upload_manual" accept="image/*" style="display: none;">
            <?php endif ?>

            <div class="easyavup_action-buttons">
                <button class="easyavup_btn easyavup_btn-primary" id="easyavup_save_profile_picture">
                    <span><?php esc_html_e( 'Update Profile', 'easy-avatar-upload' ); ?></span>
                </button>
                <button class="easyavup_btn easyavup_btn-secondary" id="easyavup-remove-avatar-button" data-id="<?php echo esc_attr( $easyavup_avatar_id ); ?>">
                    <span><?php esc_html_e( 'Remove Picture', 'easy-avatar-upload' ); ?></span>
                </button>
                
            </div>
            <div class="easyavup_message" role="status" aria-live="polite"></div>

            <div class="easyavup_guidelines">
               <p><?php esc_html_e( 'For best results, use a square image that\'s at least 400×400 pixels', 'easy-avatar-upload' ); ?></p>
            </div>

        </div>

        <?php
    }


    /**
     * Enqueues the necessary scripts and styles for the media uploader on the admin profile page.
     *
     * @param string $hook The current admin page hook.
     */
    public function easyavup_enqueue_admin_scripts( $hook ) {

        if ( ! in_array( $hook, array( 'profile.php', 'user-edit.php' ), true ) ) {
            return;
        }
        

        $dependency = ['jquery'];
        // Only enqueue if user can upload files
        if ( current_user_can( 'upload_files' ) ) {
            wp_enqueue_media();
            $dependency[] = 'media-upload';
            $dependency[] = 'media-views';
        }

        wp_enqueue_script('easyavup-admin-script', EASYAVUP__URL . '/admin/assets/js/easyavup_admin.js', $dependency, EASYAVUP__VERSION, true);
       
        wp_localize_script('easyavup-admin-script', 'easyavup_localize', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'easyavup_strings' => array(
                'easyavup_selectImage' => esc_html__( 'Select Profile Picture', 'easy-avatar-upload' ),
                'easyavup_useImage' => esc_html__( 'Use this image', 'easy-avatar-upload' ),
                'easyavup_updating' => esc_html__( 'Updating...', 'easy-avatar-upload' ),
                'easyavup_success' => esc_html__( 'Profile picture updated successfully!', 'easy-avatar-upload' ),
                'easyavup_removed' => esc_html__( 'Profile picture removed successfully!', 'easy-avatar-upload' ),
                'easyavup_removed_btn' => esc_html__( 'Removed', 'easy-avatar-upload' ),
                'easyavup_removing_btn' => esc_html__( 'Removing...', 'easy-avatar-upload' ),
                'easyavup_updated' => esc_html__( 'Updated', 'easy-avatar-upload' ),
            ),
            'easyavup_upload_limit' => wp_max_upload_size() ?? ( 2 * 1024 * 1024 )
        ]);

        wp_enqueue_style( 'easyavup-admin-style', EASYAVUP__URL . '/admin/assets/css/easyavup_admin.css', [], EASYAVUP__VERSION, 'all' );
    }

    public function eastavyo_settings($links) {
        $profile_edit_url = get_edit_profile_url( get_current_user_id() );
        $settings_link = '<a href="'. $profile_edit_url .'#easyavup_profile_back-wrap">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

 ?>