<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Handles API functions for posts, custom posts, and users.
*/

class Easy_APIs_Simplify_API_Functions {

    public function __construct() {
        add_action('rest_api_init', array($this, 'register_api_routes'));
    }

    /**
     * Registers all API routes.
     */
    public function register_api_routes() {
        // Posts & Custom Posts APIs
        register_rest_route('easy-apis/v1', '/posts/', array(
            'methods'  => 'GET',
            'callback' => array($this, 'get_posts'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/posts/', array(
            'methods'  => 'POST',
            'callback' => array($this, 'create_post'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/posts/(?P<id>\d+)', array(
            'methods'  => 'PUT',
            'callback' => array($this, 'update_post'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/posts/(?P<id>\d+)', array(
            'methods'  => 'DELETE',
            'callback' => array($this, 'delete_post'),
            'permission_callback' => array($this, 'check_permission')
        ));

        // Users APIs
        register_rest_route('easy-apis/v1', '/users/', array(
            'methods'  => 'GET',
            'callback' => array($this, 'get_users'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/users/', array(
            'methods'  => 'POST',
            'callback' => array($this, 'create_user'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/users/(?P<id>\d+)', array(
            'methods'  => 'PUT',
            'callback' => array($this, 'update_user'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/users/(?P<id>\d+)', array(
            'methods'  => 'DELETE',
            'callback' => array($this, 'delete_user'),
            'permission_callback' => array($this, 'check_permission')
        ));

        register_rest_route('easy-apis/v1', '/products/', array(
            'methods'  => 'GET',
            'callback' => array($this, 'get_products'),
            'permission_callback' => array($this, 'check_permission')
        ));
    }

    /**
     * Retrieves all posts (including custom post types).
     */
    public function get_posts($request) {
        $args = array(
            'post_type'      => $request->get_param('type') ?: 'post',
            'posts_per_page' => $request->get_param('limit') ?: 10,
            'post_status'    => 'publish'
        );

        $posts = get_posts($args);
        return rest_ensure_response($posts);
    }

    /**
     * Creates a new post.
     */
    public function create_post($request) {
        $post_data = array(
            'post_title'   => sanitize_text_field($request->get_param('title')),
            'post_content' => sanitize_textarea_field($request->get_param('content')),
            'post_status'  => 'publish',
            'post_type'    => $request->get_param('type') ?: 'post'
        );

        $post_id = wp_insert_post($post_data);
        return rest_ensure_response(['post_id' => $post_id]);
    }

    /**
     * Updates an existing post.
     */
    public function update_post($request) {
        $post_id = $request['id'];
        $post_data = array(
            'ID'           => $post_id,
            'post_title'   => sanitize_text_field($request->get_param('title')),
            'post_content' => sanitize_textarea_field($request->get_param('content'))
        );

        wp_update_post($post_data);
        return rest_ensure_response(['message' => 'Post updated successfully']);
    }

    /**
     * Deletes a post.
     */
    public function delete_post($request) {
        $post_id = $request['id'];
        wp_delete_post($post_id, true);
        return rest_ensure_response(['message' => 'Post deleted successfully']);
    }

    /**
     * Retrieves all users.
     */
    public function get_users() {
        $users = get_users();
        return rest_ensure_response($users);
    }

    /**
     * Creates a new user.
     */
    public function create_user($request) {
        $user_data = array(
            'user_login' => sanitize_user($request->get_param('username')),
            'user_pass'  => sanitize_text_field($request->get_param('password')),
            'user_email' => sanitize_email($request->get_param('email')),
            'role'       => $request->get_param('role') ?: 'subscriber'
        );

        $user_id = wp_insert_user($user_data);
        return rest_ensure_response(['user_id' => $user_id]);
    }

    /**
     * Updates an existing user.
     */
    public function update_user($request) {
        $user_id = $request['id'];
        $user_data = array(
            'ID'         => $user_id,
            'user_email' => sanitize_email($request->get_param('email')),
            'user_pass'  => sanitize_text_field($request->get_param('password'))
        );

        wp_update_user($user_data);
        return rest_ensure_response(['message' => 'User updated successfully']);
    }

    /**
     * Deletes a user.
     */
    public function delete_user($request) {
        $user_id = $request['id'];
        wp_delete_user($user_id);
        return rest_ensure_response(['message' => 'User deleted successfully']);
    }

    /**
     * Checks if the current user has permission.
     */
    public function check_permission() {
        return current_user_can('manage_options');
    }

    /**
     * Retrieves WooCommerce products with full data and pagination.
     */
    public function get_products($request) {
        if (!class_exists('WooCommerce')) {
            return new WP_Error('woocommerce_missing', 'WooCommerce is not active.', array('status' => 404));
        }

        $per_page = $request->get_param('per_page') ?: 10;
        $page     = $request->get_param('page') ?: 1;

        $args = array(
            'post_type'      => 'product',
            'posts_per_page' => intval($per_page),
            'paged'          => intval($page),
            'post_status'    => 'publish'
        );

        $query = new WP_Query($args);
        $products = [];

        foreach ($query->posts as $post) {
            $product = wc_get_product($post->ID);

            if ($product) {
                $products[] = [
                    'id'            => $product->get_id(),
                    'name'          => $product->get_name(),
                    'slug'          => $product->get_slug(),
                    'price'         => $product->get_price(),
                    'regular_price' => $product->get_regular_price(),
                    'sale_price'    => $product->get_sale_price(),
                    'stock_status'  => $product->get_stock_status(),
                    'sku'           => $product->get_sku(),
                    'description'   => $product->get_description(),
                    'short_description' => $product->get_short_description(),
                    'categories'    => wp_get_post_terms($product->get_id(), 'product_cat', ['fields' => 'names']),
                    'images'        => array_map(function($img) {
                        return [
                            'src'    => $img['src'],
                            'alt'    => $img['alt'],
                            'name'   => $img['name']
                        ];
                    }, $product->get_gallery_image_ids() ? array_map(function($id) {
                        $image = wp_get_attachment_image_src($id, 'full');
                        return [
                            'src'  => $image[0],
                            'alt'  => get_post_meta($id, '_wp_attachment_image_alt', true),
                            'name' => get_the_title($id)
                        ];
                    }, $product->get_gallery_image_ids()) : []),
                    'featured_image' => wp_get_attachment_url($product->get_image_id()),
                    'permalink'     => get_permalink($product->get_id()),
                    'attributes'    => $product->get_attributes()
                ];
            }
        }

        return rest_ensure_response([
            'products'      => $products,
            'total'         => $query->found_posts,
            'per_page'      => intval($per_page),
            'current_page'  => intval($page),
            'total_pages'   => ceil($query->found_posts / $per_page)
        ]);
    }
}

// Initialize the class
new Easy_APIs_Simplify_API_Functions();
