<?php
/*
Plugin Name: Membership 7
Description: This is an easy to use plugin for membership sites.
Version: 1.4
Requires at least: 5.8
Requires PHP: 7.3
Author: Kunimi Hirano
Author URI: https://crasios.com/
License: GPL v2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Text Domain: membership-7
Domain Path: /languages
*/

add_action('init', 'Membership7::init');
register_activation_hook( __FILE__, ['Membership7', 'myplugin_activate'] );

class Membership7
{
    const VERSION           = '1.1';
    const PLUGIN_ID         = 'membership-7';
    const DOMAIN            = 'membership-7';
    const CREDENTIAL_ACTION = self::PLUGIN_ID . '-nonce-action';
    const CREDENTIAL_NAME   = self::PLUGIN_ID . '-nonce-key';
    const PLUGIN_DB_PREFIX  = self::PLUGIN_ID . '_';
    const CONFIG_MENU_SLUG  = self::PLUGIN_ID . '-config';
    protected $redirect_url_after_sign_up;
    protected $notify_after_sign_up;
    protected $email_body_after_sign_up;
    protected $view_authority_no_when_url;

    static function init()
    {
        return new self();
    }

    function __construct(){
        
        load_plugin_textdomain(self::DOMAIN, false, basename( dirname( __FILE__ ) ).'/languages' );

        if (is_admin() && is_user_logged_in()) {
            // メニュー追加
            add_action('admin_menu', [$this, 'set_plugin_menu']);
            add_action('admin_menu', [$this, 'set_plugin_sub_menu']);
             // コールバック関数定義
            add_action('admin_init', [$this, 'save_config']);
            add_action('admin_notices', [$this, 'notices']);
            $this->redirect_url_after_sign_up = get_option(self::PLUGIN_DB_PREFIX . "redirect_url_after_sign_up");
            $this->notify_after_sign_up = get_option(self::PLUGIN_DB_PREFIX . "notify_after_sign_up");
            $this->email_body_after_sign_up = get_option(self::PLUGIN_DB_PREFIX . "notify_after_sign_up");
            $this->view_authority_no_when_url = get_option(self::PLUGIN_DB_PREFIX . "view_authority_no_when_url");

        }
    }
    /** プラグイン有効化したら設定を初期化 **/
    function myplugin_activate() {
        $redirect_url_after_sign_up = get_option(Membership7::PLUGIN_DB_PREFIX . "redirect_url_after_sign_up");
        if(empty($redirect_url_after_sign_up)){
            update_option(self::PLUGIN_DB_PREFIX . 'redirect_url_after_sign_up', home_url());
        }
        $notify_after_sign_up = get_option(Membership7::PLUGIN_DB_PREFIX . "notify_after_sign_up");
        if(empty($notify_after_sign_up)){
            update_option(self::PLUGIN_DB_PREFIX . 'notify_after_sign_up', 'both');
        }
        $view_authority_no_when_url = get_option(Membership7::PLUGIN_DB_PREFIX . "view_authority_no_when_url");
        if(empty($view_authority_no_when_url)){
            update_option(self::PLUGIN_DB_PREFIX . 'view_authority_no_when_url', home_url());
        }
    }
    
    /** メニュー表示 **/
    function set_plugin_menu()
    {

        add_menu_page(
            
            __( 'Membership Site', self::DOMAIN), 
            __( 'Membership Site', self::DOMAIN), 

            'manage_options',
            'membership-7',
            [$this, 'show_about_plugin'],
            'dashicons-format-gallery',
            99
        );
    }

    function set_plugin_sub_menu() {
        add_submenu_page(
            'membership-7',
            __( 'Settings', self::DOMAIN), 
            __( 'Settings', self::DOMAIN), 
            'manage_options',
            'membership-7-config',
            [$this, 'show_config_form']);
    }
    function show_about_plugin() {
      $html = "<h1>" . __( 'Membership Site', self::DOMAIN) . "</h1>";
      $html .= "<p>" . __( 'How to use Membership 7', self::DOMAIN) . "</p>";
      $html .= "<h2>" . __( 'Step 1', self::DOMAIN) . "</h2>";
      $html .= __( 'Go to "Membership Site" > "Settings" and set the URL to be forwarded to after user registration.', self::DOMAIN);
      $html .= "<h2>" . __( 'Step 2', self::DOMAIN) . "</h2>";
      $html .= __( 'Create a membership registration page in the fixed page. Member registration shortcode[esm file=\'signup-form\']', self::DOMAIN);
      $html .= "<h2>" . __( 'Step 3', self::DOMAIN) . "</h2>";
      $html .= __( 'Select "Email notification for user registration" in "Membership Site" > "Settings', self::DOMAIN);
      $html .= "<h2>" . __( 'Step 4', self::DOMAIN) . "</h2>";
      $html .= __( 'Create a fixed page to be displayed when the user does not have permission to view the page. Example: You do not have permission to browse. Please register here.', self::DOMAIN);
      $html .= "<h2>" . __( 'Step 5', self::DOMAIN) . "</h2>";
      $html .= __( 'Set the URL of the page to be displayed when you do not have permission to view it.', self::DOMAIN);
      $html .= "<h1 style=\"margin-top:60px;\">" . __( 'Support Site', self::DOMAIN) . "</h1>";
      $html .= "<p>" . __( 'If you have any questions, please refer to <a href="https://membership7.crasios.com/">the support site.</a>', self::DOMAIN) . "</p>";
      echo $html;
    }
    
    
    /** 設定画面の表示 */
    function show_config_form() { ?>
        <?php if($_POST['posted'] == 'save') : ?><div class="updated"><p><strong><?php _e( 'Saved the configuration.', self::DOMAIN); ?></strong></p></div><?php endif; ?>
            <div class="wrap">
            <h1><?php _e( 'Membership Site Settings', self::DOMAIN); ?></h1>
            <form action="" method='post' id="my-submenu-form">
                <?php wp_nonce_field(self::CREDENTIAL_ACTION, self::CREDENTIAL_NAME) ?>
                <table>
                <tr>
                <td><label for="redirect_url_after_sign_up"><?php _e( 'Forwarding URL after user registration', self::DOMAIN); ?></label></td>
                <td><input type="text" name="redirect_url_after_sign_up" value="<?php echo esc_html($this->redirect_url_after_sign_up); ?>" style="width:500px;"/></td>
                </tr>
                <tr>
                    <td><label for="notify_after_sign_up"><?php _e( 'Email notification for user registration', self::DOMAIN); ?></label></td>
                    <td><select name="notify_after_sign_up">
                        <option value="both" <?php if($this->notify_after_sign_up === 'both'){ echo 'selected'; } ?>><?php _e( 'Notification to both users and administrators', self::DOMAIN); ?></option>
                        <option value="admin" <?php if($this->notify_after_sign_up === 'admin'){ echo 'selected'; } ?>><?php _e( 'Notification to administrator only', self::DOMAIN); ?></option>
                        <option value="user" <?php if($this->notify_after_sign_up === 'user'){ echo 'selected'; } ?>><?php _e( 'Notify users only', self::DOMAIN); ?></option>
                    </select>
                    </td>
                </tr>
                <tr>
                  <td><label for="view_authority_no_when_url"><?php _e( 'URL of the page to display when you do not have permission to view it', self::DOMAIN); ?></label></td>
                  <td><input type="text" name="view_authority_no_when_url" value="<?php echo esc_html($this->view_authority_no_when_url); ?>" style="width:500px;"/></td>
                </tr>
                </table>
                <!--<p>
                  <label for="email_body_after_sign_up">ユーザー登録時のメール通知（本文）</label>
                </p>
                  <textarea  name="email_body_after_sign_up" value="<?= $this->email_body_after_sign_up ?>" rows="10" cols="100" /></textarea>-->
                        
                <input type="hidden" name="posted" value="save">
                <p><input type='submit' value='<?php _e( 'Save', self::DOMAIN); ?>' class='button button-primary button-large'></p>
            </form>
        </div>
    <?php
    }
    
    
    /** 設定画面の項目データベースに保存する */
    function save_config()
    {
        // credentialの確認処理
        if (isset($_POST[self::CREDENTIAL_NAME]) && $_POST[self::CREDENTIAL_NAME]) {
            
            if (check_admin_referer(self::CREDENTIAL_ACTION, self::CREDENTIAL_NAME)) {
                $e = new WP_Error();
                if(filter_var($_POST['redirect_url_after_sign_up'], FILTER_VALIDATE_URL) ){
                    $this->redirect_url_after_sign_up = $_POST['redirect_url_after_sign_up'];
                    update_option(self::PLUGIN_DB_PREFIX . 'redirect_url_after_sign_up', $this->redirect_url_after_sign_up);
                }else{
                    $e -> add('Error', __('Please enter the URL in URL format for the forwarding destination URL after user registration.'), 'membership-7-config');
                    set_transient( 'membership-7-config-errors', $e->get_error_messages(), 5);
                    update_option(self::PLUGIN_DB_PREFIX . 'redirect_url_after_sign_up', '');
                }
                if(filter_var($_POST['view_authority_no_when_url'], FILTER_VALIDATE_URL) ){
                    $this->view_authority_no_when_url = $_POST['view_authority_no_when_url'];
                    update_option(self::PLUGIN_DB_PREFIX . 'view_authority_no_when_url', $this->view_authority_no_when_url);
                }else{
                    $e -> add('Error', __('Please enter the URL of the page to be displayed when you do not have permission to view it in URL format.'), 'membership-7-config');
                    set_transient( 'membership-7-config-errors', $e->get_error_messages(), 5);
                    update_option(self::PLUGIN_DB_PREFIX . 'view_authority_no_when_url', '');
                }
                if($_POST['notify_after_sign_up'] === 'both' || $_POST['notify_after_sign_up'] === 'admin' || $_POST['notify_after_sign_up'] === 'user'){
                    $this->notify_after_sign_up = $_POST['notify_after_sign_up'];
                    update_option(self::PLUGIN_DB_PREFIX . 'notify_after_sign_up', $this->notify_after_sign_up);
                }else{
                    $e -> add('Error', __('Please select the data in the pull-down menu for email notification upon user registration.'), 'membership-7-config');
                    set_transient( 'membership-7-config-errors', $e->get_error_messages(), 5);
                }

            }
        }
    }

    function notices(){
        if( $messages = get_transient('membership-7-config-errors')): ?>
        <div class="error">
            <ul>
                <?php foreach( $messages as $massage ): ?>
                    <li><?php echo esc_html($massage);?></li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php endif; ?>
    <?php
    }

    function get_view_authority_no_when_url(){
        return get_option(Membership7::PLUGIN_DB_PREFIX . "view_authority_no_when_url");
    }

    /** 会員登録の処理をまとめた関数 **/
    function my_user_signup() {
        $user_name  = isset( $_POST['user_name'] ) ? sanitize_text_field( $_POST['user_name'] ) : '';
        $user_pass  = isset( $_POST['user_pass'] ) ? sanitize_text_field( $_POST['user_pass'] ) : '';
        $user_email = isset( $_POST['user_email'] ) ? sanitize_text_field( $_POST['user_email'] ) : '';
    
        //空じゃないかチェック
        if ( empty( $user_name ) || empty( $user_pass ) || empty( $user_email ) ) {
            echo 'There is a lack of information.';
            exit;
        }
    
        //すでにユーザー名が使われていないかチェック
        $user_id = username_exists( $user_name );
        if ( $user_id !== false ) {
            echo 'User name「'. $user_name .'」is already registered.';
            exit;
        }
    
        //すでにメールアドレスが使われていないかチェック
        $user_id = email_exists( $user_email );
        if ( $user_id !== false ) {
            echo 'Your email address「'. $user_email .'」is already registered.';
            exit;
        }
    
        //問題がなければユーザーを登録する処理を開始
        $userdata = array(
            'user_login' => $user_name,       //  ログイン名
            'user_pass'  => $user_pass,       //  パスワード
            'user_email' => $user_email,      //  メールアドレス
            'show_admin_bar_front' => 'false' //  管理バーは非表示
        );
        $user_id = wp_insert_user( $userdata );
    
        // ユーザーの作成に失敗した場合
        if ( is_wp_error( $user_id ) ) {
            echo $user_id -> get_error_code(); // WP_Error() の第一引数
            echo $user_id -> get_error_message(); // WP_Error() の第二引数
            exit;
        }else{
            //ユーザーに通知する
            $notify_after_sign_up = get_option(Membership7::PLUGIN_DB_PREFIX . "notify_after_sign_up");
            wp_new_user_notification( $user_id, null, $notify_after_sign_up); 
            //登録完了後、そのままログインさせる（ 任意 ）
            wp_set_auth_cookie( $user_id, false, is_ssl() );
            
            $redirect_url_after_sign_up = get_option(Membership7::PLUGIN_DB_PREFIX . "redirect_url_after_sign_up");
            if(!empty($redirect_url_after_sign_up)){
                wp_redirect( $redirect_url_after_sign_up );
            }else{
                echo 'The URL to be forwarded after user registration has not been set. User registration was successful.';
            }
            exit;
        }
        return;
    }


} // end of class



//phpをショートコードで読み込む
function esm_short_php($params = array()) {
  extract(shortcode_atts(array(
    'file' => 'default'
  ), $params));
  ob_start();
  include(plugin_dir_path( __FILE__ ) . "/$file.php");
  return ob_get_clean();
}
 
add_shortcode('esm', 'esm_short_php');



/** after_setup_theme に処理をフック **/
add_action('after_setup_theme', function() {

    //会員登録フォームからの送信があるかどうか
    if ( isset( $_POST['my_submit'] ) && $_POST['my_submit'] === 'signup') {

        // nonceチェック
        if ( !isset( $_POST['my_nonce_name'] ) ) return;
        if ( !wp_verify_nonce( $_POST['my_nonce_name'], 'my_nonce_action' ) ) return;

        // 登録処理を実行
        $Membership7 = new Membership7();
        $Membership7 -> my_user_signup();
    }
});



// 投稿画面 メタボックスの追加 
add_action( 'add_meta_boxes', 'esm_myplg_meta_box_init' );
function esm_myplg_meta_box_init() {
    add_meta_box( 'esm_myplg-meta', 'Public setting (member site)',
    'esm_myplg_meta_box', 'post', 'side', 'default' );
}
function esm_myplg_meta_box( $post, $box ) {
    //カスタムメタボックスの値を取得
    $article_release_status = get_post_meta( $post->ID, '_article_release_status', true );
    wp_nonce_field( plugin_basename( __FILE__ ),'esm_myplg_save_meta_box' );
    echo '<p><select name="article_release_status"><option value="open"';
    if( $article_release_status === 'open' ){
        echo 'selected';
    }
    echo '>Open to all (can be viewed by non-members)</option><option value="member"';
    if( $article_release_status === 'member' ){
        echo 'selected';
    }
    echo '>Members only</option></select>';
}

/** 投稿画面 メタボックスの値を保存 **/
add_action( 'save_post', 'esm_myplg_save_meta_box' );
function esm_myplg_save_meta_box( $post_id ) {
    if( isset( $_POST['article_release_status'] ) ) {
        //自動保存の場合はスキップ
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )
        return;
        wp_verify_nonce( plugin_basename( __FILE__ ),'esm_myplg_save_meta_box' );
        update_post_meta( $post_id, '_article_release_status',sanitize_text_field( $_POST['article_release_status'] ) );
    }
}

// 投稿一覧 カラムを追加する
function esm_my_columns($columns) {
    $columns['customfield'] = "Public Settings";
    return $columns;
}
//カラムの表示内容を定義
//追加したカラムに「metaname」というカスタムフィールドの内容を表示させる
function esm_add_column($column_name, $post_id) {
    if( $column_name == 'customfield' ) {
        $stitle = get_post_meta($post_id, '_article_release_status', true);
    }
    if ( isset($stitle) && $stitle ) {
        if( $stitle === 'member' ){
            echo 'Members only';
        }else if(  $stitle === 'open' ){
            echo 'Open to all';
        }
        //echo attribute_escape($stitle);
    } else {
        echo __('None');
    }
}
add_filter( 'manage_posts_columns', 'esm_my_columns' );
add_action( 'manage_posts_custom_column', 'esm_add_column', 10, 2 );


// wp_headで会員だけに表示させる
function esm_add_head_only_members(){
    $Membership7 = new Membership7();
    $view_authority_no_when_url = $Membership7 -> get_view_authority_no_when_url();
    $article_release_status = get_post_meta( get_the_ID(), '_article_release_status', true );
    // ログインしていれば閲覧OK、そうじゃなければリダイレクト
    if ( $article_release_status === 'member' &&  !is_user_logged_in() ){ 
        wp_safe_redirect( $view_authority_no_when_url , 302);
        //wp_safe_redirect( 'http://test20210725.local/%e9%96%b2%e8%a6%a7%e6%a8%a9%e9%99%90%e3%81%aa%e3%81%84%e3%81%a7%e3%83%bc/' , 302);
        exit;
    }
}
add_action( 'wp_enqueue_scripts', 'esm_add_head_only_members', 99 );


// 一括編集
add_filter( 'bulk_actions-edit-post', function( array $actions ) {
    $actions['make_member_limited'] = __( 'Make it members only' );
    $actions['do_not_to_member_limited'] = __( 'Make it open to all (viewable by non-members)' );
    return $actions;
} );

// 一括編集 会員限定にする
add_filter( 'handle_bulk_actions-edit-post', function( string $sendback, string $doaction, array $post_ids ) {
    if ( 'make_member_limited' !== $doaction ) {
        return $sendback;
    }
    $count = 0;
    foreach ( $post_ids as $post_id ) {
        update_post_meta( $post_id, '_article_release_status',sanitize_text_field( 'member' ) );
        $count++;
    }
    return add_query_arg( 'updated', $count, $sendback );
}, 10, 3 );

// 一括編集 全公開（会員以外も閲覧可能）にする
add_filter( 'handle_bulk_actions-edit-post', function( string $sendback, string $doaction, array $post_ids ) {
    if ( 'do_not_to_member_limited' !== $doaction ) {
        return $sendback;
    }
    $count = 0;
    foreach ( $post_ids as $post_id ) {
        update_post_meta( $post_id, '_article_release_status',sanitize_text_field( 'open' ) );
        $count++;
    }
    return add_query_arg( 'updated', $count, $sendback );
}, 10, 3 );

?>