<?php
/**
 * Plugin Name: Easy Alt Text Generator
 * Description: Generate SEO-friendly image alt text from filenames — no external API required. Includes a secure bulk generator.
 * Version: 1.0.3
 * Author: wordgems
 * Text Domain: easy-alt-text-generator
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Generate alt text from filename
function eatg_alt_from_filename( $filepath ) {
    $filename = pathinfo( (string) $filepath, PATHINFO_FILENAME );
    $filename = preg_replace( '/-\d{2,5}x\d{2,5}$/', '', (string) $filename );
    $filename = str_replace( array( '-', '_' ), ' ', $filename );
    $filename = trim( preg_replace( '/\s+/', ' ', (string) $filename ) );
    return mb_convert_case( $filename, MB_CASE_TITLE, 'UTF-8' );
}

// Set alt text on upload
function eatg_set_alt_on_upload( $attachment_id ) {
    if ( ! wp_attachment_is_image( $attachment_id ) ) return;
    $current_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
    if ( '' !== trim( (string) $current_alt ) ) return;
    $file = get_attached_file( $attachment_id );
    if ( ! $file ) return;
    $alt = eatg_alt_from_filename( $file );
    if ( '' === $alt ) {
        $title = get_the_title( $attachment_id );
        $alt   = is_string( $title ) ? $title : '';
    }
    $alt = trim( $alt );
    if ( '' !== $alt ) {
        if ( mb_strlen( $alt ) > 125 ) $alt = mb_substr( $alt, 0, 122 ) . '…';
        update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt );
    }
}
add_action( 'add_attachment', 'eatg_set_alt_on_upload' );

// Register bulk page
function eatg_register_bulk_page() {
    add_media_page(
        esc_html__( 'Easy Alt Text — Bulk Generator', 'easy-alt-text-generator' ),
        esc_html__( 'Easy Alt Text Bulk', 'easy-alt-text-generator' ),
        'manage_options',
        'eatg-bulk',
        'eatg_render_bulk_page'
    );
}
add_action( 'admin_menu', 'eatg_register_bulk_page' );

// Process bulk
function eatg_process_bulk( $process_all = false ) {
    $args = array(
        'post_type'      => 'attachment',
        'post_mime_type' => 'image',
        'post_status'    => 'inherit',
        'fields'         => 'ids',
        'posts_per_page' => -1,
    );
    $query = new WP_Query( $args );
    $ids   = (array) $query->posts;
    $processed = $generated = $skipped = $errors = 0;

    foreach ( $ids as $id ) {
        $processed++;
        $current_alt = get_post_meta( $id, '_wp_attachment_image_alt', true );
        if ( ! $process_all && '' !== trim( (string) $current_alt ) ) { $skipped++; continue; }
        $file = get_attached_file( $id );
        if ( ! $file ) { $errors++; continue; }
        $alt = eatg_alt_from_filename( $file );
        if ( '' === $alt ) { $title = get_the_title( $id ); $alt = is_string( $title ) ? $title : ''; }
        $alt = trim( $alt );
        if ( '' === $alt ) { $skipped++; continue; }
        if ( mb_strlen( $alt ) > 125 ) $alt = mb_substr( $alt, 0, 122 ) . '…';
        $ok = update_post_meta( $id, '_wp_attachment_image_alt', $alt );
        if ( false !== $ok ) $generated++; else $errors++;
    }
    return compact( 'processed', 'generated', 'skipped', 'errors' );
}

// Render bulk page
function eatg_render_bulk_page() {
    if ( ! current_user_can( 'manage_options' ) ) return;
    $did_submit = false; $message_html = '';
    if ( isset( $_POST['eatg_bulk_submit'] ) ) {
        $did_submit = true;
        $nonce = isset( $_POST['eatg_bulk_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['eatg_bulk_nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'eatg_bulk_action' ) ) wp_die( esc_html__( 'Security check failed.', 'easy-alt-text-generator' ) );
        $mode = isset( $_POST['eatg_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['eatg_mode'] ) ) : 'missing';
        $results = eatg_process_bulk( 'all' === $mode );
        // translators: %1$d processed; %2$d generated; %3$d skipped; %4$d errors.
   $parts = array(
    /* translators: %d: number of images processed. */
    sprintf( esc_html__( 'Processed: %d', 'easy-alt-text-generator' ), (int) $results['processed'] ),

    /* translators: %d: number of images generated. */
    sprintf( esc_html__( 'Generated: %d', 'easy-alt-text-generator' ), (int) $results['generated'] ),

    /* translators: %d: number of images skipped. */
    sprintf( esc_html__( 'Skipped: %d', 'easy-alt-text-generator' ), (int) $results['skipped'] ),

    /* translators: %d: number of errors. */
    sprintf( esc_html__( 'Errors: %d', 'easy-alt-text-generator' ), (int) $results['errors'] ),
);

        $message_html = '<div class="notice notice-success"><p>' . esc_html( implode( ' · ', $parts ) ) . '</p></div>';
    }
    ?>
    <div class="wrap">
        <h1><?php echo esc_html__( 'Easy Alt Text — Bulk Generator', 'easy-alt-text-generator' ); ?></h1>
        <p><?php echo esc_html__( 'Generate alt text for existing media. “Missing only” updates empty alts; “All images” overwrites all.', 'easy-alt-text-generator' ); ?></p>
        <?php if ( $did_submit && $message_html ) echo wp_kses_post( $message_html ); ?>
        <form method="post">
            <?php wp_nonce_field( 'eatg_bulk_action', 'eatg_bulk_nonce' ); ?>
            <label><input type="radio" name="eatg_mode" value="missing" checked> <?php echo esc_html__( 'Missing only', 'easy-alt-text-generator' ); ?></label><br>
            <label><input type="radio" name="eatg_mode" value="all"> <?php echo esc_html__( 'All images', 'easy-alt-text-generator' ); ?></label><br>
            <?php submit_button( esc_html__( 'Run Bulk Generator', 'easy-alt-text-generator' ), 'primary', 'eatg_bulk_submit' ); ?>
        </form>
    </div>
    <?php
}
