<?php
/**
 * REST API endpoints for EACL.
 *
 * @package Easy_Add_To_Cart_Links
 */

namespace DogCat\EACL;

defined( 'ABSPATH' ) || exit;

use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

/**
 * Register and handle REST endpoints.
 */
class REST {


	/**
	 * Hook route registration.
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'rest_api_init', array( __CLASS__, 'routes' ) );
	}

	/**
	 * Register REST routes.
	 *
	 * @return void
	 */
	public static function routes(): void {
		register_rest_route(
			'eacl/v1',
			'/search',
			array(
				'methods'             => WP_REST_Server::READABLE, // GET.
				'callback'            => array( __CLASS__, 'search' ),
				'permission_callback' => function (): bool {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					's' => array(
						'type'              => 'string',
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		register_rest_route(
			'eacl/v1',
			'/link',
			array(
				'methods'             => WP_REST_Server::READABLE, // GET.
				'callback'            => array( __CLASS__, 'link' ),
				'permission_callback' => function (): bool {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'product_id'   => array(
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
					),
					'variation_id' => array(
						'type'              => 'integer',
						'required'          => false,
						'sanitize_callback' => 'absint',
					),
					'attrs'        => array(
						// Base64+JSON payload (URL-encoded in transport). Sanitize as text here; validate below.
						'type'              => 'string',
						'required'          => false,
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);
	}

	/**
	 * Product/variation search for the Classic/Gutenberg link dialog.
	 *
	 * @param  WP_REST_Request $r REST request.
	 * @return array<int, array<string,mixed>> List of items for the UI.
	 */
	public static function search( WP_REST_Request $r ): array {
		$q = $r->get_param( 's' );
		if ( ! $q ) {
			return array();
		}

		$ids = get_posts(
			array(
				'post_type'      => array( 'product', 'product_variation' ),
				's'              => $q,
				'fields'         => 'ids',
				'posts_per_page' => 20,
			)
		);

		$seen = array();
		$out  = array();

		foreach ( $ids as $id ) {
			$p = wc_get_product( $id );
			if ( ! $p ) {
				continue;
			}

			if ( $p->is_type( 'simple' ) ) {
				$pid = $p->get_id();

				if ( isset( $seen[ $pid ] ) ) {
					continue;
				}
				$seen[ $pid ] = 1;

				$out[] = array(
					'id'   => $pid,
					'name' => $p->get_name(),
					'type' => 'simple',
					'url'  => self::build_url( $pid, null, array() ),
				);

			} elseif ( $p->is_type( 'variable' ) ) {
				$pid = $p->get_id();

				if ( isset( $seen[ $pid ] ) ) {
					continue;
				}
				$seen[ $pid ] = 1;

				$vars = array();
				foreach ( $p->get_available_variations() as $var ) {
					$vid   = (int) $var['variation_id'];
					$attrs = array();

					foreach ( $var['attributes'] as $k => $v ) {
						$key           = preg_replace( '/^attribute_/', '', $k );
						$attrs[ $key ] = (string) $v;
					}

					$vars[] = array(
						'id'         => $vid,
						'in_stock'   => ! empty( $var['is_in_stock'] ),
						'attributes' => $attrs,
						'url'        => self::build_url( $pid, $vid, $attrs ),
					);
				}

				$out[] = array(
					'id'         => $pid,
					'name'       => $p->get_name(),
					'type'       => 'variable',
					'variations' => $vars,
				);

			} elseif ( $p->is_type( 'variation' ) ) {
				$parent = wc_get_product( $p->get_parent_id() );
				if ( ! $parent || isset( $seen[ $parent->get_id() ] ) ) {
					continue;
				}

				$seen[ $parent->get_id() ] = 1;
				$vars                      = array();

				foreach ( $parent->get_available_variations() as $var ) {
					$vid   = (int) $var['variation_id'];
					$attrs = array();

					foreach ( $var['attributes'] as $k => $v ) {
						$key           = preg_replace( '/^attribute_/', '', $k );
						$attrs[ $key ] = (string) $v;
					}

					$vars[] = array(
						'id'         => $vid,
						'in_stock'   => ! empty( $var['is_in_stock'] ),
						'attributes' => $attrs,
						'url'        => self::build_url( $parent->get_id(), $vid, $attrs ),
					);
				}

				$out[] = array(
					'id'         => $parent->get_id(),
					'name'       => $parent->get_name(),
					'type'       => 'variable',
					'variations' => $vars,
				);
			}
		}

		return $out;
	}

	/**
	 * Build and return a link for a given product/variation via REST.
	 *
	 * Validates and decodes the optional `attrs` payload (URL-encoded base64 JSON).
	 *
	 * @param  WP_REST_Request $r REST request.
	 * @return array<string,mixed>|WP_Error Response payload or error.
	 */
	public static function link( WP_REST_Request $r ) {
		$pid = (int) $r->get_param( 'product_id' );

		// Avoid short ternary: we want NULL when the param is omitted/empty.
		$vid_param = $r->get_param( 'variation_id' );
		$vid       = ( '' !== $vid_param && null !== $vid_param ) ? (int) $vid_param : null;

		$attrs = array();
		$blob  = (string) $r->get_param( 'attrs' );

		if ( '' !== $blob ) {
			$raw = rawurldecode( $blob );

			// Strict base64; returns false on invalid data.
			$json = base64_decode( $raw, true ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Strict decode with validation; JSON checked; user input sanitized.
			if ( false === $json ) {
				return new WP_Error(
					'eacl_bad_attrs',
					__( 'Invalid attribute payload encoding.', 'easy-add-to-cart-links' ),
					array( 'status' => 400 )
				);
			}

			$data = json_decode( $json, true );
			if ( JSON_ERROR_NONE !== json_last_error() || ! is_array( $data ) ) {
				return new WP_Error(
					'eacl_bad_attrs',
					__( 'Malformed attribute payload.', 'easy-add-to-cart-links' ),
					array( 'status' => 400 )
				);
			}

			// Normalize to string=>string map.
			foreach ( $data as $k => $v ) {
				if ( is_scalar( $v ) ) {
					$attrs[ (string) $k ] = sanitize_text_field( (string) $v );
				}
			}
		}

		$url = self::build_url( $pid, $vid, $attrs );

		return array(
			'product_id'   => $pid,
			'variation_id' => $vid,
			'url'          => $url,
		);
	}

	/**
	 * Build a relative Add-to-Cart URL for WooCommerce.
	 *
	 * @param  int                  $product_id   Product ID.
	 * @param  int|null             $variation_id Variation ID or null.
	 * @param  array<string,string> $attrs        Variation attributes.
	 * @return string Relative URL (e.g., "/?add-to-cart=123&variation_id=456&attribute_pa_size=large").
	 */
	public static function build_url( int $product_id, ?int $variation_id, array $attrs ): string {
		$args = array();

		if ( $variation_id ) {
			// Parent in add-to-cart, plus variation + attributes.
			$args['add-to-cart']  = $product_id;
			$args['variation_id'] = $variation_id;

			// Handles WooCommerce Product attributes ($attrs) like ['pa_size' => 'large', ...].
			foreach ( $attrs as $tax => $term ) {
				$args[ 'attribute_' . sanitize_title( $tax ) ] = sanitize_title( (string) $term );
			}
		} else {
			// Simple product.
			$args['add-to-cart'] = $product_id;
		}

		// Build absolute, then return relative.
		$abs = add_query_arg( $args, home_url( '/' ) );
		return wp_make_link_relative( $abs );
	}
}
