<?php
/**
 * Core plugin bootstrap.
 *
 * @package Easy_Add_To_Cart_Links
 */

namespace DogCat\EACL;

defined( 'ABSPATH' ) || exit;

/**
 * Main Plugin Orchestrator.
 */
final class Plugin {

	/**
	 * Register hooks.
	 *
	 * @return void
	 */
	public static function init(): void {
		// i18n (safe in all contexts).
		add_action( 'init', array( __CLASS__, 'i18n' ) );

		// REST route (safe in all contexts).
		if ( class_exists( __NAMESPACE__ . '\\REST' ) && method_exists( REST::class, 'init' ) ) {
			REST::init();
		}

		// Woo-dependent features.
		$has_woo = class_exists( 'WooCommerce' );

		if ( $has_woo && class_exists( __NAMESPACE__ . '\\Links' ) && method_exists( Links::class, 'init' ) ) {
			Links::init();
		}
		if ( $has_woo && class_exists( __NAMESPACE__ . '\\UX' ) && method_exists( UX::class, 'init' ) ) {
			UX::init();
		}

		// Admin assets for Block + Classic editors.
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_admin_assets' ) );
	}

	/**
	 * Enqueue the Gutenberg LinkControl enhancer and the Classic wp-link modal enhancer.
	 * Script files are expected at: /admin/js/linkcontrol.js and /admin/js/linkcontrol-classic.js.
	 *
	 * @param string $hook Current admin page hook (e.g. 'post.php' or 'post-new.php').
	 * @return void
	 */
	public static function enqueue_admin_assets( string $hook ): void {
		if ( ! in_array( $hook, array( 'post.php', 'post-new.php' ), true ) ) {
			return;
		}

		// Derive plugin root URL & path from /includes/plugin.php.
		$plugin_root_url  = trailingslashit( dirname( plugin_dir_url( __FILE__ ) ) );
		$plugin_root_path = trailingslashit( dirname( plugin_dir_path( __FILE__ ) ) );

		$block_rel   = 'admin/js/linkcontrol.js';
		$classic_rel = 'admin/js/linkcontrol-classic.js';

		$block_url   = $plugin_root_url . $block_rel;
		$classic_url = $plugin_root_url . $classic_rel;

		$block_ver   = file_exists( $plugin_root_path . $block_rel ) ? filemtime( $plugin_root_path . $block_rel ) : null;
		$classic_ver = file_exists( $plugin_root_path . $classic_rel ) ? filemtime( $plugin_root_path . $classic_rel ) : null;

		// Shared REST config used by BOTH enhancers.
		$rest = array(
			'root'    => esc_url_raw( rest_url( 'eacl/v1/' ) ),
			'nonce'   => wp_create_nonce( 'wp_rest' ),
			'siteUrl' => home_url( '/' ),
		);

		// Gutenberg (Block editor) LinkControl enhancer.
		wp_enqueue_script(
			'eacl-linkcontrol',
			$block_url,
			array( 'wp-api-fetch' ),
			$block_ver,
			true
		);
		wp_localize_script( 'eacl-linkcontrol', 'EACL', array( 'rest' => $rest ) );

		// Classic editor wp-link modal enhancer.
		wp_enqueue_script(
			'eacl-linkcontrol-classic',
			$classic_url,
			array( 'jquery', 'wplink', 'wp-api-fetch' ),
			$classic_ver,
			true
		);
		wp_localize_script( 'eacl-linkcontrol-classic', 'EACL', array( 'rest' => $rest ) );
	}

	/**
	 * Load translations (noop on .org installs; harmless elsewhere).
	 *
	 * @return void
	 */
	public static function i18n(): void {
		// Intentionally empty. WordPress.org auto-loads language packs for this plugin slug.
	}
}
