<?php
namespace Dynomapper;

if (!defined( 'WPINC' )) die;

class ContentImport
{
    public static function createInstance($viewsDir)
    {
        return new ContentImport(
            'dynomapper-content-import',
            'DYNO Mapper Content Import',
            $viewsDir,
            getenv('APP_ENV') == 'dev'
        );
    }

    private $authBaseUrl = 'https://app.dynomapper.com';
    private $pluginId;
    private $pluginName;
    private $viewsDir;
    private $devMode;
    private $templateVars = [];

    public function __construct($pluginId, $pluginName, $viewsDir, $devMode)
    {
        $this->pluginId = $pluginId;
        $this->pluginName = $pluginName;
        $this->viewsDir = $viewsDir;
        $this->devMode = $devMode;
    }

    private function importJwt()
    {
        require_once __DIR__ . '/php-jwt.php';
    }

    private function createWpHttp()
    {
        if(!class_exists( 'WP_Http' )) {
            require_once ABSPATH . WPINC. '/class-http.php';
        }

        return new \WP_Http();
    }

    public function init()
    {
        add_action('admin_init', [$this, 'registerImporter']);

        register_uninstall_hook(__FILE__, [$this, 'uninstall']);
    }

    public function registerImporter()
    {
        $this->handleRequest();

        register_importer(
            $this->pluginId,
            $this->pluginName,
            'Import the page structure and contents of your <a href="https://dynomapper.com" target="_blank">DYNO Mapper</a> project.',
            [$this, 'initImporter']
        );

        add_action('wp_ajax_' . $this->pluginId . '.importProcessApi', [$this, 'importProcessApi']);
    }

    private function handleRequest()
    {
        if (empty($_GET['import']) || $_GET['import'] != $this->pluginId) {
            return;
        }

        $this->importJwt();

        $this->templateVars = [];

        if (!empty($_GET['runImport']) && !empty($_GET['token'])) {
            //validate import
            $this->validateImport($_GET['token']);
            return;
        }

        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            check_admin_referer('content-import-credentials');

            $clientId = isset($_POST['clientId']) ? sanitize_text_field($_POST['clientId']) : '';
            $clientSecret = isset($_POST['clientSecret']) ? sanitize_text_field($_POST['clientSecret']) : '';
            $authBaseUrl = isset($_POST['authBaseUrl']) ? sanitize_text_field($_POST['authBaseUrl']) : '';

            update_option($this->pluginId . '.clientId', $clientId);
            update_option($this->pluginId . '.clientSecret', $clientSecret);
            update_option($this->pluginId . '.authBaseUrl', $authBaseUrl);

            $this->templateVars['saved'] = true;

            if (!empty($_POST['start'])) {
                $result = $this->redirectToAuthUrl($clientId, $clientSecret, $this->getAuthBaseUrl());

                if (is_wp_error($result)) {
                    $this->templateVars['saveError'] = $result;
                }
            }
        }
    }

    public function initImporter()
    {
        $this->importJwt();

        if (!empty($_GET['runImport']) && !empty($_GET['token'])) {
            //validate import
            extract($this->templateVars);
            include $this->viewsDir . '/validate-import.php';
            return;
        }

        if ($accessToken = get_transient($this->pluginId . '.accessToken')) {
            //start import
            delete_transient($this->pluginId . '.accessToken');
            delete_transient($this->pluginId . '.importState');
            $this->runImport($accessToken);
            return;
        }

        $clientIdValue = get_option($this->pluginId . '.clientId');
        $clientSecretValue = get_option($this->pluginId . '.clientSecret');
        $authBaseUrlValue = get_option($this->pluginId . '.authBaseUrl') ?: $this->authBaseUrl;

        $importFinish = (bool) get_transient($this->pluginId . '.importFinish');

        if ($importFinish) {
            delete_transient($this->pluginId . '.importFinish');
        }

        extract($this->templateVars);
        include $this->viewsDir . '/import.php';
    }

    public function getAuthBaseUrl()
    {
        if (!$this->devMode) {
            return $this->authBaseUrl;
        }

        return get_option($this->pluginId . '.authBaseUrl') ?: $this->authBaseUrl;
    }

    public function redirectToAuthUrl($clientId, $clientSecret, $baseUrl)
    {
        $http = $this->createWpHttp();

        $timestampUrl = rtrim($baseUrl, '/') . '/api/content-import/auth/timestamp?clientId=' . urlencode($clientId);
        $response = $http->get($timestampUrl);

        if (is_wp_error($response)) {
            return $response;
        }

        if ($response['response']['code'] != 200) {
            return new \WP_Error('httpError', 'Failed to validate base URL. Received HTTP status ' . $response['response']['code']);
        }

        $timestamp = (int)$response['body'];
        $redirectUrl = admin_url('admin.php?' . http_build_query(['import' => $this->pluginId, 'runImport' => 1]));

        $token = \Dynomapper\Firebase\JWT\JWT::encode([
            'sub' => $clientId,
            'iat' => $timestamp,
            'nbf' => $timestamp,
            'exp' => $timestamp + (strtotime('+3 minutes') - time()),
            'redirectUri' => $redirectUrl,
        ], $clientSecret, 'HS256', wp_generate_uuid4());

        wp_redirect(
            rtrim($baseUrl, '/') . '/api/content-import/auth?' .
            http_build_query([
                'clientId' => $clientId,
                'token' => $token,
            ])
        );

        return null;
    }

    public function validateImport($token)
    {
        $clientIdValue = get_option($this->pluginId . '.clientId');
        $clientSecretValue = get_option($this->pluginId . '.clientSecret');

        \Dynomapper\Firebase\JWT\JWT::$leeway = 30;

        $error = new \WP_Error();

        try {
            $decoded = \Dynomapper\Firebase\JWT\JWT::decode(
                $token,
                new \Dynomapper\Firebase\JWT\Key($clientSecretValue, 'HS256')
            );
        } catch (\Exception $e) {
            $error->add('token', $e->getMessage());
        }

        if ($decoded->sub != $clientIdValue) {
            $error->add('clientId', 'Invalid token subject');
        }

        if ($decoded->iss != 'dynomapper.com') {
            $error->add('token', 'Invalid token issuer');
        }

        if (!$error->has_errors()) {
            set_transient($this->pluginId . '.accessToken', $decoded->accessToken, 60);
            wp_redirect('admin.php?import=' . $this->pluginId);
            return;
        }

        $this->templateVars['error'] = $error;
    }

    public function runImport($accessToken)
    {
        include $this->viewsDir . '/run-import.php';
    }

    private function buildApiUrl($parentId = 0, $maxLevel = 0, $limit = 0, $offset = 0, $includeMetadata = 0)
    {
        return rtrim($this->getAuthBaseUrl(), '/') . '/api/content-import?' . http_build_query([
                'parentId' => $parentId,
                'maxLevel' => $maxLevel,
                'limit' => $limit,
                'offset' => $offset,
                'includeMetadata' => $includeMetadata,
            ]);
    }

    public function hasSeoPluginInstalled()
    {
        return $this->pluginYoastSeoInstalled() || $this->pluginAllInOneSeoInstalled();
    }

    public function pluginYoastSeoInstalled()
    {
        return class_exists('WPSEO_Meta') && method_exists('WPSEO_Meta', 'set_value');
    }

    public function pluginAllInOneSeoInstalled()
    {
        return defined('AIOSEO_VERSION') && version_compare(AIOSEO_VERSION, '4.1.0', '>=')
            && method_exists('\\AIOSEO\\Plugin\\Common\\Models\\Post', 'savePost');
    }

    public function importProcessApi()
    {
        $accessToken = isset($_POST['accessToken']) ? sanitize_text_field($_POST['accessToken']) : '';
        $limit = isset($_POST['limit']) ? (int) $_POST['limit'] : 0;
        $maxLevel = isset($_POST['maxLevel']) ? (int) $_POST['maxLevel'] : 0;
        $includeMetadata = isset($_POST['includeMetadata']) && (int)$_POST['includeMetadata'] && $this->hasSeoPluginInstalled();
        $menuName = !empty(trim($_POST['menuName'])) ? sanitize_text_field($_POST['menuName']) : 'DYNO Mapper';
        $createMenu = isset($_POST['createMenu']) && (int) $_POST['createMenu'];
        $excludeContent = isset($_POST['excludeContent']) && (int) $_POST['excludeContent'];

        set_time_limit(600);

        $http = $this->createWpHttp();

        $importState = get_transient($this->pluginId . '.importState');

        if (!$importState) {
            $importState = (object) [
                'limit' => $limit,
                'maxLevel' => $maxLevel,
                'accessToken' => $accessToken,
                'includeMetadata' => $includeMetadata,
                'menuName' => $menuName,
                'createMenu' => $createMenu,
                'excludeContent' => $excludeContent,
                'items' => [],
                'pageIds' => [],
                'mediaUrls' => [],
                'menuIds' => [],
                'menu' => 0,
                'requestChildren' => [],
                'parentId' => 0,
                'nextOffset' => 0,
            ];
        }

        if (!$importState->accessToken) {
            wp_send_json(['error' => true, 'reason' => 'Missing access token from the request']);
            return;
        }

        $response = $http->get($this->buildApiUrl(
            $importState->parentId,
            $importState->maxLevel,
            $importState->limit,
            $importState->nextOffset,
            $importState->includeMetadata
        ), [
            'headers' => [
                'Authorization' => 'Bearer ' . $importState->accessToken,
            ],
        ]);

        if (is_wp_error($response)) {
            wp_send_json(['error' => true, 'reason' => 'Bad API response: ' . $response->get_error_message()]);
            return;
        }

        if ($response['response']['code'] != 200) {
            wp_send_json(['error' => true, 'reason' => 'Bad API response: ' . $response['response']['message']]);
            return;
        }

        $body = json_decode($response['body']);
        unset($response);

        if (!empty($body->mediaUrls)) {
            $this->uploadMedia($body, $importState);
        }

        if ($body->nextOffset && !empty($body->items)) {
            //buffer the items in transient
            $importState->items = array_merge($importState->items, $body->items);
            $importState->nextOffset = $body->nextOffset;
            set_transient($this->pluginId . '.importState', $importState, 10800);

            wp_send_json([
                'pages' => count($importState->pageIds),
                'finish' => false,
            ]);
            return;
        }

        $finish = false;

        if (!empty($importState->items)) {
            $importState = $this->doImport($importState);
            set_transient($this->pluginId . '.importState', $importState, 10800);
        }

        if (empty($importState->parentId)) {
            $finish = true;
            delete_transient($this->pluginId . '.importState');
            set_transient($this->pluginId . '.importFinish', true, 60);
        }

        wp_send_json([
            'pages' => count($importState->pageIds),
            'finish' => $finish,
        ]);
    }

    private function doImport($importState)
    {
        if (!$importState->menu && $importState->createMenu) {
            $navMenu = wp_get_nav_menu_object($importState->menuName);

            if ($navMenu) {
                $importState->menu = $navMenu->term_id;
            } else {
                $importState->menu = wp_create_nav_menu($importState->menuName);
            }
        }

        $byParent = [];

        foreach ($importState->items as $item) {
            $byParent[$item->parent][] = $item;
        }

        $itemsStack = new SimpleStack(isset($byParent[$importState->parentId]) ? $byParent[$importState->parentId] : []);
        unset($byParent[0]);

        while ($itemsStack->isNotEmpty()) {
            $item = $itemsStack->pop();
            $postId = $this->createPost($item, $importState);

            $importState->pageIds[$item->id] = $postId;

            if ($importState->menu) {
                $menuId = $this->createMenuItem($postId, $item, $importState);
                $importState->menuIds[$item->id] = $menuId;
            }

            if (!empty($byParent[$item->id])) {
                $itemsStack->pushStable($byParent[$item->id]);
                unset($byParent[$item->id]);
            }

            if ($item->requestChildren) {
                $importState->requestChildren[] = $item->id;
            }
        }

        $importState->items = [];
        $importState->nextOffset = 0;

        if (!empty($importState->requestChildren)) {
            $importState->parentId = array_shift($importState->requestChildren);
        } else {
            $importState->parentId = 0;
        }

        return $importState;
    }

    private function uploadMedia($response, $importState)
    {
        if (empty($response->mediaUrls)) {
            return;
        }

        foreach ($response->mediaUrls as $url) {
            if (isset($importState->mediaUrls[$url])) {
                continue;
            }

            $tmp = download_url($url);

            if (is_wp_error($tmp)) {
                continue;
            }

            try {
                $id = media_handle_sideload([
                    'name' => basename($url),
                    'tmp_name' => $tmp,
                ]);

                if (!is_wp_error($id)) {
                    $importState->mediaUrls[$url] = wp_get_attachment_url($id);
                }
            } finally {
                if (file_exists($tmp)) {
                    unlink($tmp);
                }
            }
        }
    }

    private function createPost($item, $importState)
    {
        $content = '';

        if (!$importState->excludeContent) {
            $content = preg_replace_callback('~https?://[^\s">]+~iu', function ($a) use ($importState) {
                if (isset($importState->mediaUrls[$a[0]])) {
                    return $importState->mediaUrls[$a[0]];
                }

                return $a[0];
            }, implode("\n", $item->contents));
        }

        $params = [
            'post_title' => $item->title,
            'post_content' => $content,
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_parent' => !empty($importState->pageIds[$item->parent]) ? $importState->pageIds[$item->parent] : 0,
        ];

        $postId = wp_insert_post($params);

        //add seo details
        if ($importState->includeMetadata && $this->pluginYoastSeoInstalled()) {
            if (!empty($item->metaDescription)) {
                \WPSEO_Meta::set_value('metadesc', $item->metaDescription, $postId);
            }

            if (!empty($item->metaKeywords)) {
                \WPSEO_Meta::set_value('focuskw', $item->metaKeywords, $postId);
            }
        }

        if ($importState->includeMetadata &&
            $this->pluginAllInOneSeoInstalled() &&
            (!empty($item->metaKeywords) || !empty($item->metaDescription))
        ) {
            $keywords = $item->metaKeywords;

            if ($keywords) {
                $keywords = explode(',', $keywords);
                $keywords = array_map(function ($k) {
                    return ['value' => trim($k)];
                }, $keywords);
                $keywords = json_encode($keywords);
            }

            \AIOSEO\Plugin\Common\Models\Post::savePost($postId, [
                'description' => $item->metaDescription,
                'keywords' => $keywords,
            ]);
        }

        return $postId;
    }

    private function createMenuItem($postId, $item, $importState)
    {
        $menuItemParams = [
            'menu-item-title' => $item->title,
            'menu-item-status' => 'publish',
            'menu-item-parent-id' => !empty($importState->menuIds[$item->parent]) ? $importState->menuIds[$item->parent] : 0,
        ];

        if (!empty($item->contents) || !$item->url) {
            $menuItemParams['menu-item-object'] = 'page';
            $menuItemParams['menu-item-object-id'] = $postId;
            $menuItemParams['menu-item-type'] = 'post_type';
        } else {
            $menuItemParams['menu-item-type'] = 'custom';
            $menuItemParams['menu-item-url'] = $item->url;
        }

        return wp_update_nav_menu_item($importState->menu, 0, $menuItemParams);
    }

    public function uninstall()
    {
        delete_transient($this->pluginId . '.importState');
        delete_transient($this->pluginId . '.accessToken');
        delete_option($this->pluginId . '.clientId');
        delete_option($this->pluginId . '.clientSecret');
        delete_option($this->pluginId . '.authBaseUrl');
    }
}

class SimpleStack {
    private $arr;

    public function __construct($initialData = [])
    {
        $this->arr = $initialData;
    }

    public function pushStable(array $s)
    {
        foreach ($s as $value) {
            $this->arr[] = $value;
        }
    }

    public function pop()
    {
        return array_shift($this->arr);
    }

    public function isEmpty()
    {
        return empty($this->arr);
    }

    public function isNotEmpty()
    {
        return !$this->isEmpty();
    }
}
