<?php
/**
 * Dynin Replacer - Core phone number replacement logic
 */

if (!defined('ABSPATH')) {
    exit;
}

class Dynin_Replacer {
    
    private $rules = [];
    private $current_url = '';
    private $active_rule = null;
    
    public function __construct() {
        $this->load_rules();
        $this->set_current_url();
    }
    
    private function load_rules() {
		$this->rules = get_option( 'dynin_rules', null );
		if ( ! is_array( $this->rules ) ) {
			$this->rules = [];
		}

		usort(
			$this->rules,
			static function ( $a, $b ) {
				$au = isset( $a['url'] ) ? (string) $a['url'] : '';
				$bu = isset( $b['url'] ) ? (string) $b['url'] : '';
				return strlen( $bu ) - strlen( $au );
			}
		);
    }
    
    private function set_current_url() {
        if ( isset( $_SERVER['REQUEST_URI'] ) ) {
			$uri = wp_unslash( (string) $_SERVER['REQUEST_URI'] );
			$qpos = strpos( $uri, '?' );
			if ( false !== $qpos ) {
				$uri = substr( $uri, 0, $qpos );
			}
			$this->current_url = trailingslashit( esc_url_raw( $uri ) );
        } else {
			$this->current_url = '/';
        }
    }
    
    private function find_active_rule() {
		if ( null !== $this->active_rule ) {
            return $this->active_rule;
        }
        
        foreach ($this->rules as $rule) {
			$ru = isset( $rule['url'] ) ? trailingslashit( (string) $rule['url'] ) : '';
			$or = isset( $rule['original'] ) ? (string) $rule['original'] : '';
			$re = isset( $rule['replacement'] ) ? (string) $rule['replacement'] : '';
                
			if ( '' === $ru || '' === $or || '' === $re ) {
				continue;
			}

			if ( $this->current_url === $ru ) {
                    $this->active_rule = $rule;
                    return $rule;
                }
            }
        
        $this->active_rule = false;
        return false;
    }
    
    public function process_content($content) {
        $rule = $this->find_active_rule();
        
        if (!$rule) {
            return $content;
        }
		return $this->replace_phone_numbers( $content, (string) $rule['original'], (string) $rule['replacement'] );
    }
    
    public function process_buffer($buffer) {
        $rule = $this->find_active_rule();
        
        if (!$rule) {
            return $buffer;
        }
		return $this->replace_phone_numbers( $buffer, (string) $rule['original'], (string) $rule['replacement'] );
    }
    
    private function replace_phone_numbers($content, $original, $replacement) {
        $normalized = $this->normalize_phone($original);
		if ( '' === $normalized ) {
            return $content;
        }

        $replacement_formatted = $this->format_phone_for_display($replacement);
        $replacement_tel = $this->normalize_phone($replacement);
        $variations = $this->generate_phone_variations($normalized);
        
		$placeholder_prefix = '###DYNIN_PLACEHOLDER_';
        $placeholder_suffix = '###';
        $placeholders = [];
        $placeholder_count = 0;
        
        $content = preg_replace_callback(
            '/<a[^>]*href=["\']tel:[^"\']*["\'][^>]*>.*?<\/a>/i',
            function($matches) use (&$placeholders, &$placeholder_count, $placeholder_prefix, $placeholder_suffix, $variations, $replacement_tel, $replacement_formatted) {
                $link = $matches[0];

                $contains_target = false;
                foreach ($variations as $variation) {
					if ( '' === $variation ) {
						continue;
					}
					if ( stripos( $link, $variation ) !== false || stripos( $link, preg_replace( '/\D+/', '', $variation ) ) !== false ) {
                        $contains_target = true;
                        break;
                    }
                }
                
                if ($contains_target) {
					// href
                    $link = preg_replace(
                        '/(href=["\']tel:)[^"\']*/i',
                        '${1}' . $replacement_tel,
                        $link
                    );
                    
                    foreach ($variations as $var) {
						if ( '' === $var ) {
							continue;
						}
                        $var_pattern = preg_quote($var, '/');
						$link        = preg_replace( '/' . $var_pattern . '/i', $replacement_formatted, $link );
                    }
                }

                $placeholder = $placeholder_prefix . $placeholder_count . $placeholder_suffix;
                $placeholders[$placeholder] = $link;
                $placeholder_count++;
                
                return $placeholder;
            },
            $content
        );
        
        foreach ($variations as $variation) {
			if ( '' === $variation ) {
                continue;
            }
            
            $pattern = preg_quote($variation, '/');
            $content = preg_replace(
                '/\b' . $pattern . '\b/i',
                $replacement_formatted,
                $content
            );
        }
        
		foreach ( $placeholders as $ph => $original_html ) {
			$content = str_replace( $ph, $original_html, $content );
        }
        
        return $content;
    }
    
    private function normalize_phone($phone) {
		return preg_replace( '/[^0-9]/', '', (string) $phone );
    }
    
    private function generate_phone_variations($normalized) {
		$normalized = (string) $normalized;

        if (strlen($normalized) < 10) {
            return [$normalized];
        }
        
        $variations = [];
        
		// US 10 digits
		if ( 10 === strlen( $normalized ) ) {
            $area = substr($normalized, 0, 3);
            $prefix = substr($normalized, 3, 3);
            $line = substr($normalized, 6, 4);

            $variations[] = $normalized;                                    // 5551234567
			$variations[] = "($area) $prefix-$line";
			$variations[] = "$area-$prefix-$line";
			$variations[] = "$area.$prefix.$line";
			$variations[] = "$area $prefix $line";
			$variations[] = "($area)$prefix-$line";
			$variations[] = "1-$area-$prefix-$line";
			$variations[] = "+1-$area-$prefix-$line";
			$variations[] = "+1 ($area) $prefix-$line";
			$variations[] = "1 ($area) $prefix-$line";
			$variations[] = "+1$area$prefix$line";
			$variations[] = "1$area$prefix$line";
        }
        
		// 11 digits starting with country code 1
		if ( 11 === strlen( $normalized ) && '1' === $normalized[0] ) {
            $area = substr($normalized, 1, 3);
            $prefix = substr($normalized, 4, 3);
            $line = substr($normalized, 7, 4);
            
			$variations[] = "$area$prefix$line";
			$variations[] = "($area) $prefix-$line";
			$variations[] = "$area-$prefix-$line";
        }
        
		return array_values( array_unique( $variations ) );
    }
    
    private function format_phone_for_display($phone) {
        $normalized = $this->normalize_phone($phone);
        
		// Если уже есть форматирующие символы — оставляем как есть
		if ( preg_match( '/[\(\)\-\.\s]/', (string) $phone ) ) {
			return (string) $phone;
        }
        
		if ( 10 === strlen( $normalized ) ) {
            return sprintf('(%s) %s-%s',
                substr($normalized, 0, 3),
                substr($normalized, 3, 3),
                substr($normalized, 6, 4)
            );
        }
        
		if ( 11 === strlen( $normalized ) && '1' === $normalized[0] ) {
            return sprintf('1 (%s) %s-%s',
                substr($normalized, 1, 3),
                substr($normalized, 4, 3),
                substr($normalized, 7, 4)
            );
        }
        
		return (string) $phone;
    }
}