<?php
/**
 * Plugin Name: Dynamic Number Insertion
 * Description: Dynamically replace phone numbers on specific pages or posts.
 * Version: 1.0.2
 * Author: Ryan Howard
 * Author URI: https://www.ryanhoward.dev
 * Text Domain: dynamic-number-insertion
 * Requires at least: 5.8
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

define( 'DYNIN_VERSION', '1.0.2' );
define( 'DYNIN_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'DYNIN_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

require_once DYNIN_PLUGIN_DIR . 'replacer.php';
if (is_admin()) {
	require_once DYNIN_PLUGIN_DIR . 'admin-page.php';
}

/**
 * Core
 */
class DynamicNumberInsertion {
    
    private static $instance = null;

	/** @var Dynin_Replacer */
    private $replacer;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
		$this->replacer = new Dynin_Replacer();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);

        add_action('init', [$this, 'setup_replacement_method']);

        if (is_admin()) {
            add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        }
    }
    
    public function setup_replacement_method() {
        if (is_admin() || wp_doing_ajax()) {
            return;
        }
        add_action('template_redirect', [$this, 'start_buffer'], 1);
    }
    
    public function start_buffer() {
        ob_start([$this->replacer, 'process_buffer']);
    }
    
    public function enqueue_admin_assets($hook) {
		if ( 'settings_page_dynamic-number-insertion' !== $hook ) {
            return;
        }
        
        wp_enqueue_style(
			'dynin-admin',
			DYNIN_PLUGIN_URL . 'assets/admin.css',
            [],
			DYNIN_VERSION
        );
        
        wp_enqueue_script(
			'dynin-admin',
			DYNIN_PLUGIN_URL . 'assets/admin.js',
            ['jquery'],
			DYNIN_VERSION,
            true
        );
        
		wp_localize_script(
			'dynin-admin',
			'dynin_admin',
			[
            'ajax_url' => admin_url('admin-ajax.php'),
				'nonce'    => wp_create_nonce( 'dynin_admin_nonce' ),
			]
		);
    }
    
    public function activate() {
		if ( false === get_option( 'dynin_rules', false ) ) {
			update_option( 'dynin_rules', [] );
        }

        flush_rewrite_rules();
    }
    
    public function deactivate() {
        flush_rewrite_rules();
    }
}

DynamicNumberInsertion::get_instance();