<?php
/**
 * Dynin Admin Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

class Dynin_Admin {
    
    public function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'process_form']);

		add_filter('plugin_action_links_' . plugin_basename( DYNIN_PLUGIN_DIR . 'dynamic-number-insertion.php' ), [ $this, 'add_settings_link' ]);
    }

    public function add_settings_link($links) {
        $url = admin_url('options-general.php?page=dynamic-number-insertion');
        array_unshift($links, '<a href="' . esc_url($url) . '">Settings</a>');
        return $links;
    }
    
    public function add_settings_page() {
        add_options_page(__('Dynamic Number Insertion', 'dynamic-number-insertion'), __('Number Insertion', 'dynamic-number-insertion'), 'manage_options', 'dynamic-number-insertion', [$this, 'render_settings_page']);
    }
    
	/**
	 * Helpers — sanitize POST
	 */
	private function read_post_text( $key ) {
		$val = filter_input( INPUT_POST, $key, FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		return is_string( $val ) ? sanitize_text_field( wp_unslash( $val ) ) : '';
	}

	private function read_post_url_path( $key ) {
		$val = filter_input( INPUT_POST, $key, FILTER_SANITIZE_URL );
		$val = is_string( $val ) ? wp_unslash( $val ) : '';
		return $this->sanitize_url_path( $val );
	}

	private function read_get_text( $key ) {
		$val = filter_input( INPUT_GET, $key, FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		return is_string( $val ) ? sanitize_text_field( wp_unslash( $val ) ) : '';
	}

    public function process_form() {
		if ( empty( $_POST['dynin_action'] ) ) {
            return;
        }
        
		if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
		if (
			empty( $_POST['dynin_nonce'] ) ||
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['dynin_nonce'] ) ), 'dynin_settings' )
		) {
            return;
        }
        
		$action = $this->read_post_text( 'dynin_action' );
        
        switch ($action) {
            case 'add_rule':
                $this->add_rule();
                break;
            case 'delete_rule':
                $this->delete_rule();
                break;
            case 'update_rule':
                $this->update_rule();
                break;
        }
    }
    
	private function get_rules() {
		$rules = get_option( 'dynin_rules', null );
		if ( ! is_array( $rules ) ) {
			$rules = [];
		}
		return $rules;
	}

	private function save_rules( array $rules ) {
		update_option( 'dynin_rules', array_values( $rules ) ); // reindex
	}

    private function add_rule() {
        if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Insufficient permissions.', 'dynamic-number-insertion' ) );
        }

		$rules       = $this->get_rules();
		$original    = $this->read_post_text( 'dynin_original' );
		$replacement = $this->read_post_text( 'dynin_replacement' );
		$url_path    = $this->read_post_url_path( 'dynin_url' );

		// Validate
		if ( '' === $original || '' === $replacement || '' === $url_path ) {
			add_settings_error(
				'dynin_messages',
				'dynin_message',
				__( 'All fields are required.', 'dynamic-number-insertion' ),
				'error'
			);
			return;
		}

		$rules[] = [
			'id'          => wp_generate_uuid4(),
            'original' => $original,
            'replacement' => $replacement,
			'url'         => $url_path,
        ];
        
		$this->save_rules( $rules );
        
		add_settings_error(
			'dynin_messages',
			'dynin_message',
			__( 'Rule added successfully.', 'dynamic-number-insertion' ),
			'updated'
		);
    }
    
    private function update_rule() {
		$rules   = $this->get_rules();
		$rule_id = $this->read_post_text( 'dynin_rule_id' );
        
        foreach ($rules as &$rule) {
			if ( isset( $rule['id'] ) && (string) $rule['id'] === $rule_id ) {
				$rule['original']    = $this->read_post_text( 'dynin_edit_original_' . $rule_id );
				$rule['replacement'] = $this->read_post_text( 'dynin_edit_replacement_' . $rule_id );
				$rule['url']         = $this->read_post_url_path( 'dynin_edit_url_' . $rule_id );
                break;
            }
        }
		unset( $rule );

		$this->save_rules( $rules );

		add_settings_error(
			'dynin_messages',
			'dynin_message',
			__( 'Rule updated successfully.', 'dynamic-number-insertion' ),
			'updated'
		);
    }
    
    private function delete_rule() {
		$rules   = $this->get_rules();
		$rule_id = $this->read_post_text( 'dynin_rule_id' );
        
		$rules = array_filter(
			$rules,
			static function ( $rule ) use ( $rule_id ) {
				return ( isset( $rule['id'] ) ? (string) $rule['id'] : '' ) !== $rule_id;
			}
		);
        
		$this->save_rules( $rules );
        
		add_settings_error(
			'dynin_messages',
			'dynin_message',
			__( 'Rule deleted successfully.', 'dynamic-number-insertion' ),
			'updated'
		);
	}
        
	private function sanitize_url_path( $url ) {
		$url = (string) $url;
    
		$url = preg_replace( '#^https?://[^/]+#i', '', $url );
        
		if ( '' === $url || '/' !== $url[0] ) {
			$url = '/' . ltrim( $url, '/' );
        }
        
        $url = trailingslashit($url);
        
		return esc_url_raw( $url );
    }
    
    public function render_settings_page() {
		$rules   = $this->get_rules();
		$edit_id = $this->read_get_text( 'edit' );
        ?>
		<div class="wrap dynin-admin-wrap">
            <h1><?php esc_html_e('Dynamic Number Insertion', 'dynamic-number-insertion'); ?></h1>
            
			<?php settings_errors( 'dynin_messages' ); ?>
            
			<div class="dynin-rules-section">
                <h2><?php esc_html_e('Phone Number Rules', 'dynamic-number-insertion'); ?></h2>
                
				<table class="wp-list-table widefat fixed striped dynin-rules-table">
                    <thead>
                        <tr>
                            <th class="column-original"><?php esc_html_e('Original Number', 'dynamic-number-insertion'); ?></th>
                            <th class="column-replacement"><?php esc_html_e('Replacement Number', 'dynamic-number-insertion'); ?></th>
                            <th class="column-url"><?php esc_html_e('URL/Page', 'dynamic-number-insertion'); ?></th>
                            <th class="column-actions"><?php esc_html_e('Actions', 'dynamic-number-insertion'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($rules)): ?>
                            <?php foreach ($rules as $rule): ?>
							<tr class="dynin-rule-row <?php echo ( $edit_id === ( $rule['id'] ?? '' ) ) ? 'editing' : ''; ?>">
								<?php if ( $edit_id === ( $rule['id'] ?? '' ) ) : ?>
                                    <td>
                                        <form method="post" action="" id="edit-form-<?php echo esc_attr($rule['id']); ?>">
											<?php wp_nonce_field( 'dynin_settings', 'dynin_nonce' ); ?>
											<input type="hidden" name="dynin_action" value="update_rule" />
											<input type="hidden" name="dynin_rule_id" value="<?php echo esc_attr( $rule['id'] ); ?>" />
                                            <input type="text" 
												name="dynin_edit_original_<?php echo esc_attr( $rule['id'] ); ?>"
                                                   value="<?php echo esc_attr($rule['original']); ?>"
                                                   class="regular-text"
												required />
                                        </form>
                                    </td>
                                    <td>
                                        <input type="text" 
											name="dynin_edit_replacement_<?php echo esc_attr( $rule['id'] ); ?>"
                                               value="<?php echo esc_attr($rule['replacement']); ?>"
                                               class="regular-text"
                                               form="edit-form-<?php echo esc_attr($rule['id']); ?>"
											required />
                                    </td>
                                    <td>
                                        <input type="text" 
											name="dynin_edit_url_<?php echo esc_attr( $rule['id'] ); ?>"
                                               value="<?php echo esc_attr($rule['url']); ?>"
                                               class="regular-text"
                                               form="edit-form-<?php echo esc_attr($rule['id']); ?>"
											required />
                                    </td>
                                    <td>
                                        <button type="submit" 
                                                class="button button-primary button-small"
                                                form="edit-form-<?php echo esc_attr($rule['id']); ?>">
                                            <?php esc_html_e('Save', 'dynamic-number-insertion'); ?>
                                        </button>
                                        <a href="<?php echo esc_url(admin_url('options-general.php?page=dynamic-number-insertion')); ?>"
                                           class="button button-small">
                                            <?php esc_html_e('Cancel', 'dynamic-number-insertion'); ?>
                                        </a>
                                    </td>
                                <?php else: ?>
									<td><?php echo esc_html( $rule['original'] ?? '' ); ?></td>
									<td><?php echo esc_html( $rule['replacement'] ?? '' ); ?></td>
									<td><code><?php echo esc_html( $rule['url'] ?? '' ); ?></code></td>
                                    <td>
										<a href="<?php echo esc_url( add_query_arg( 'edit', (string) ( $rule['id'] ?? '' ), admin_url( 'options-general.php?page=dynamic-number-insertion' ) ) ); ?>"
                                           class="button">
                                            <?php esc_html_e('Edit', 'dynamic-number-insertion'); ?>
                                        </a>
                                        <form method="post" action="" style="display:inline;">
											<?php wp_nonce_field( 'dynin_settings', 'dynin_nonce' ); ?>
											<input type="hidden" name="dynin_action" value="delete_rule" />
											<input type="hidden" name="dynin_rule_id" value="<?php echo esc_attr( (string) ( $rule['id'] ?? '' ) ); ?>" />
                                            <button type="submit" 
                                                    class="button button-link-delete" 
												onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to delete this rule?', 'dynamic-number-insertion' ) ); ?>');">
                                                <?php esc_html_e('Delete', 'dynamic-number-insertion'); ?>
                                            </button>
                                        </form>
                                    </td>
                                <?php endif; ?>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                        
						<tr class="dynin-add-row">
                            <form method="post" action="" id="add-rule-form">
								<?php wp_nonce_field( 'dynin_settings', 'dynin_nonce' ); ?>
								<input type="hidden" name="dynin_action" value="add_rule" />
                                <td>
									<input type="text" name="dynin_original" placeholder="(555) 123-4567" class="regular-text" />
                                </td>
                                <td>
									<input type="text" name="dynin_replacement" placeholder="(777) 888-9999" class="regular-text" />
                                </td>
                                <td>
									<input type="text" name="dynin_url" placeholder="/services/" class="regular-text" />
                                </td>
                                <td>
                                    <button type="submit" class="button button-primary">
                                        <?php esc_html_e('Add Rule', 'dynamic-number-insertion'); ?>
                                    </button>
                                </td>
                            </form>
                        </tr>
                    </tbody>
                </table>
                
                <?php if (empty($rules)): ?>
                <p class="description" style="margin-top: 10px;">
                    <?php esc_html_e('No rules configured yet. Add your first rule using the form above.', 'dynamic-number-insertion'); ?>
                </p>
                <?php endif; ?>
            </div>
            
			<div class="dynin-help-section">
                <h3><?php esc_html_e('How It Works', 'dynamic-number-insertion'); ?></h3>
                <ul>
                    <li><?php esc_html_e('The plugin automatically detects various formats of the same phone number (with or without parentheses, dashes, dots, spaces, etc.)', 'dynamic-number-insertion'); ?></li>
                    <li><?php esc_html_e('Rules are matched by exact URL - enter the path exactly as it appears (e.g., /services/ or /contact/)', 'dynamic-number-insertion'); ?></li>
                    <li><?php esc_html_e('The most specific URL takes priority when multiple rules could apply', 'dynamic-number-insertion'); ?></li>
                    <li><?php esc_html_e('Changes happen before the page loads, so tracking scripts like CallRail can still work on top', 'dynamic-number-insertion'); ?></li>
                </ul>
            </div>
        </div>
        <?php
    }
}

// Initialize admin
new Dynin_Admin();