<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://guaven.com
 * @since      1.0.0
 *
 * @package    Guaven_dxtag
 * @subpackage Guaven_dxtag/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Guaven_dxtag
 * @subpackage Guaven_dxtag/includes
 * @author     Guaven Labs <support@guaven.com>
 */
class Guaven_dxtag {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Guaven_dxtag_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		$this->version = GUAVEN_DXTAG_VERSION;
		$this->plugin_name = 'guaven_dxtag';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Guaven_dxtag_Loader. Orchestrates the hooks of the plugin.
	 * - Guaven_dxtag_i18n. Defines internationalization functionality.
	 * - Guaven_dxtag_Admin. Defines all hooks for the admin area.
	 * - Guaven_dxtag_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-guaven_dxtag-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-guaven_dxtag-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/guaven_dxtag-admin-helper.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-guaven_dxtag-admin.php';


		$this->loader = new Guaven_dxtag_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Guaven_dxtag_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Guaven_dxtag_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		global $argv;
		$request=[];
		$request["argv"]=!empty($argv)?$argv:[];
		$request["get_rule_id"]=isset($_GET["rule_id"])&&$_GET["rule_id"]>0?$_GET["rule_id"]:'';
		
		$plugin_admin = new Guaven_dxtag_Admin( $this->get_plugin_name(), $this->get_version(),$request );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'init', $plugin_admin, 'register_post_type' );	
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_settings_page' );	

		$this->loader->add_action( 'admin_init', $plugin_admin, 'save_settings'  );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'onboarding'  );
		$this->loader->add_action( 'wp', $plugin_admin, 'run_cron_tasks'  );

		$this->loader->add_action( 'wp', $plugin_admin, 'redirect_admintax_to_publictax'  );

		


		$this->loader->add_action( 'add_meta_boxes',  'Guaven_Dxtag_Meta_Box', 'add'  );
		$this->loader->add_action( 'save_post', 'Guaven_Dxtag_Meta_Box', 'save', 10, 3  );
		$this->loader->add_action( 'transition_post_status',  'Guaven_Dxtag_Meta_Box', 'sync_term_data', 10, 3 );
		

		$this->loader->add_action( 'rest_api_init',$plugin_admin,'rest_api_init');
		
	}


	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Guaven_dxtag_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}

