<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://guaven.com
 * @since      1.0.0
 *
 * @package    Guaven_dxtag
 * @subpackage Guaven_dxtag/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Guaven_dxtag
 * @subpackage Guaven_dxtag/admin
 * @author     Guaven Labs <support@guaven.com>
 */
class Guaven_dxtag_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version, $request ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->request = $request;

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/guaven_dxtag-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register non-public Custom Post Type for created rulesets.
	 *
	 * @since    1.0.0
	 */
	public function register_post_type(){

		$labels = array(
			'name'                  => _x( 'DxTag Rules', 'Post Type General Name', 'wordpress' ),
			'singular_name'         => _x( 'DxTag Rule', 'Post Type Singular Name', 'wordpress' ),
			'add_new_item'          => __( 'Add New Rule', 'wordpress' ),
			'add_new'               => __( 'Add New Rule', 'wordpress' ),
		);
		$args = array(
			'label'                 => __( 'Guaven DxTag Rule', 'wordpress' ),
			'description'           => __( 'Post Type Description', 'wordpress' ),
			'labels'                => $labels,
			'supports'              => array( 'title' ),
			'hierarchical'          => false,
			'public'                => false,
			'show_ui'               => true,
			'show_in_menu'          => false,
			'show_in_admin_bar'     => false,
			'show_in_nav_menus'     => false,
			'can_export'            => true,
			'has_archive'           => true,
			'exclude_from_search'   => true,
			'publicly_queryable'    => true,
			'capability_type'       => 'page',
		);
		register_post_type( 'gvn_dxtag_rule', $args );

		$tagname=get_option('guaven_dxtag_tagslug');
		if(empty($tagname))$tagname='dxtag';
		$args = array(
            'hierarchical'      => false,
            'labels'=> array('name' => __('Smart Collections','guaven_dxtag'),
							 'menu_name'=>__('Smart Collections','guaven_dxtag'),
							 'singular_name'=>  __('Smart Collection','guaven_dxtag')
							),
            'show_ui'           => true,
            'show_admin_column' => true,
			'show_in_menu'=>false,
            'query_var'         => true,
            'rewrite'           => array( 'slug' =>$tagname),
        );
		register_taxonomy('gvn_dxtag',['post','page','product'],$args);
		$this->maybe_flush();
	}
	
	/**
	 * Flushes rewrite rules when it is needed. 
	 * Flushing rewrite rules by this plugin is called only when the main taxonomy name is created/changed. 
	 * @since    1.0.0
	 */
	function maybe_flush(){
		if(get_option('guaven_dxtag_flush')==1){
			flush_rewrite_rules();
			update_option('guaven_dxtag_flush','');
		}
	}

	public function redirect_admintax_to_publictax(){
		if(strpos($_SERVER["REQUEST_URI"],'/gvn_dxtag_rule/')===false)return;
		global $post;
		$term_exists=get_term_by('name',$post->post_title,'gvn_dxtag');	
		if(!empty($term_exists->term_id)){
			wp_redirect(get_term_link( $term_exists->term_id, 'gvn_dxtag' ),302);
			die();
		}
	}

	/**
	 * This function shows onboarding explanation section after the first plugin activation.
	 * @since    1.0.0
	 */
	function onboarding(){

		if(get_option('guaven_dxtag_activated')!='-1')return;
		update_option('guaven_dxtag_activated',1);

		wp_redirect(admin_url()."/options-general.php?page=guaven_dxtag_settings&dxtag_onboard=1");
		exit;
	}
	
	/**
	 * import_pre_installed_rules
	 *
	 * @return void
	 * @since    1.0.0
	 */
	function import_pre_installed_rules(){
		$rules=$this->list_of_pre_installed_rules();
		foreach($rules as $rule){
			$post_id=wp_insert_post(
				[
					'post_type'=>'gvn_dxtag_rule',
					'post_title'=>$rule["title"],
					'post_status'=>'draft'
				]
			);
			Guaven_DxTag_Meta_Box::update_rule_meta($post_id,$rule["metadata"]);
		}
	}
	

	/**
	 * returns the list of 
	 * pre-installed rules
	 */
	function list_of_pre_installed_rules(){

		$rules=[
			[
				'title'=>"WooCommerce OnSale Products",
				'metadata'=>['dxrule'=>['wp:post_type','woocommerce:_sale_price'],'dxvalue'=>['product | product_variation',''],'dxoperand'=>['=','!=']]
			],
			[
				'title'=>"WooCommerce Selling Out Products",
				'metadata'=>['dxrule'=>['wp:post_type','woocommerce:_stock','woocommerce:_stock'],'dxvalue'=>['product | product_variation','5','1'],'dxoperand'=>['=','<','>=']]
			],
			[
				'title'=>"Latest news that contains the word 'Fire'",
				'metadata'=>['dxrule'=>['wp:post_type','wp:post_title'],'dxvalue'=>['post','Fire'],'dxoperand'=>['=','LIKE']]
			],
			
		];

		return $rules;
	}

	/**
     * Save the plugin's settings.
     *
     */
    public static function save_settings( ) {
		
        if (!isset($_POST['guaven_dxtag_settingsnonce']) or !wp_verify_nonce($_POST['guaven_dxtag_settingsnonce'], 'guaven_dxtag_settingsnonce'))
		return;

		

        if (!current_user_can('edit_posts'))
            return;
		
        if ( !empty( $_POST["guaven_dxtag_tagslug"] )  ){
			$old=get_option('guaven_dxtag_tagslug');
			$new=sanitize_title($_POST["guaven_dxtag_tagslug"]);
            update_option( 'guaven_dxtag_tagslug', $new);
			
			if($old!=$new){
				update_option('guaven_dxtag_flush',1);
			}
        } 
       
    }
		
	/**
	 * instead of wp-ajax we use WP REST API in order to regenerate the needed DB data.
	 * this API is called by 2 sources: 1. wp-admin settings page. 2. Cron Job.
	 * @return void
	 */
	function rest_api_init(){
		register_rest_route( 'app/v1', '/dx-rule-regenerator/', array(
				'methods' => 'GET',
				'callback' => [$this, 'rule_regenerator'],
				'permission_callback' => function(){
					return current_user_can( 'administrator');
				}
			) 
		);
	}
	
	/**
	 * increases memory limit to regenerate whole data for the given rule
	 * @return void
	 */
	function check_memory_limit(){
		$memory_limit=apply_filters( 'guaven_dxtag_admin_memory_limit', '511M' );
		$default_memory_limit=ini_get('memory_limit');
		if($memory_limit>$default_memory_limit){
			ini_set('memory_limit',$memory_limit);
		}
	}

		
	/**
	 * main function of the plugin.
	 * the function finds all matched posts under given rules set by current rule 
	 * and assigns corresponging WP term ID to those posts.
	 */
	public function rule_regenerator(){
		
		$this->check_memory_limit();		

		$rule_id=$this->request["get_rule_id"];

		if((int)$rule_id==0)
			$ret='Empty data';
		else{
			$rules = get_post_meta($rule_id,'guaven_dxtag_rule',true);
			$rulepost=get_post($rule_id);
			if(empty($rulepost))die('wrong ID');
			
			$query_args=$this->query_arg_builder($rules);			
			
			$dontcleans=[];
			$dx_query = new WP_Query( $query_args );
			if ( $dx_query->have_posts() ) {
				while ( $dx_query->have_posts() ) {
					$dx_query->the_post();
					$dxid= $GLOBALS['post']->post_type=='product_variation' ? $GLOBALS['post']->post_parent: $GLOBALS['post']->ID;//for woocommerce
					wp_set_post_terms( $dxid, $rulepost->post_title, 'gvn_dxtag',true);
					$dontcleans[]=$dxid;
				}
			} 
			wp_reset_postdata();

			$term_exists=get_term_by('name',$rulepost->post_title,'gvn_dxtag');
			
			if(!empty($term_exists->term_id) and !empty($dontcleans)){
				global $wpdb;
				$wpdb->query(
					$wpdb->prepare("delete from $wpdb->term_relationships 
					where term_taxonomy_id=%d and object_id not in (%1s)",
					$term_exists->term_id,implode(",",$dontcleans))
				);
				wp_update_term_count_now( [$term_exists->term_id],'gvn_dxtag' );
			}
			if($dx_query->found_posts==0)$ret=__('Done! But there is no found product match under the rules you have just set. Try to adjust the rules.','guaven_dxtag');
			else 
			/* translators: %d here is the number of found products.*/
			$ret=sprintf(__('Success! %d post objects has been added to this DxTag.','guaven_dxtag'),$dx_query->found_posts);
		}

		$response = new WP_REST_Response($ret);
		$response->set_status(200);
		return $response;

	}

	
	
	/**
	 * query_arg_builder
	 * builds WP_Query $args for rule generator function
	 * @param  mixed $rules
	 * @return void
	 */
	function query_arg_builder($rules){
		$defaults = array(
			'posts_per_page' => '-1',
			'numberposts' => '-1',
			'suppress_filters' =>true,
		  );
		$args=['meta_query'=>[]];
		$rule_index=0;
		foreach($rules as $rule){

			$dxvalue=$rule["dxvalue"];
			$dxvalue_arr=explode("|",$rule["dxvalue"]);

			if($rule["dxrule"]=='wp:post_type'){
				$args['post_type']=$dxvalue_arr;
			} 
			elseif(strpos($rule["dxrule"],'woocommerce:')===0){
				$args['post_type']='product';
			}

			$meta_to_add=[];
			$tax_to_add=[];
			foreach($dxvalue_arr as $dxval){

				if($rule["dxrule"]=='woocommerce:product_cat'){
					$tax_to_add[]=
					[
						'taxonomy' => 'product_cat',
						'field'    => 'slug',
						'terms'    => $dxval,
					];
				}
				elseif(strpos($rule["dxrule"],'woocommerce:')===0
				or strpos($rule["dxrule"],'wp:meta_key')===0
				){
					$frmtd_rule=str_replace(["woocommerce:","wp:"],[''],$rule["dxrule"]);
					$meta_to_add[]=
					[
						'key'=>$frmtd_rule,
						'value'=>$dxval,
						'compare'=>$rule["dxoperand"],
						'type'=>in_array($rule["dxoperand"],['>=','<'])?'NUMERIC':'CHAR'
					];
				}
				elseif( $rule["dxrule"]=='wp:term' ){
					$tax_value=explode(":",$dxval);
					$tax_to_add[]=
					[
						'taxonomy' => $tax_value[0],
						'field'    => 'slug',
						'terms'    => $tax_value[1],
					];
				}
			}
			if (!empty($tax_to_add)){
				$args['tax_query'][$rule_index]=$tax_to_add;
				if(count($tax_to_add)>1)$args['tax_query'][$rule_index]['relation']='OR';	
			}
			if (!empty($meta_to_add)){
				$args['meta_query'][$rule_index]=$meta_to_add;
				if(count($meta_to_add)>1)$args['meta_query'][$rule_index]['relation']='OR';			
			}
			$rule_index++;
		}
		
		$args = wp_parse_args( $args, $defaults );
		$args=apply_filters('guaven_dxtag_query_builder_args',$args);
		return $args;
	}


	/**
	 * This function runs all needed cron jobs for the plugin
	 * @return void
	 */
	function run_cron_tasks(){
		if( empty($this->request["argv"][1]) or $this->request["argv"][1]!='update_dxtag_rules')return;
		global $wpdb;
		$pids=$wpdb->get_col("select post_id from $wpdb->postmeta 
							where meta_key='cron_update_this_dxtag' 
							and meta_value!=''
							order by meta_value ASC"
		);
		while (@ob_end_flush());      
		ob_implicit_flush(true);
		foreach($pids as $pid){
			$this->request["get_rule_id"]=$pid;
			$response= $this->rule_regenerator();
			echo $response->get_data().PHP_EOL;
			update_post_meta($pid,'cron_update_this_dxtag',time());
			usleep(1000);
		}
		exit;
	}
	

	/**
	 * Adds Admin Menu for Settings Page
	 */
	function add_settings_page() {
		add_submenu_page('options-general.php', 'Smart Collections', 'Smart Collections', 'manage_options','guaven_dxtag_settings', 
		array(
            $this,
            'load_settings_page'
        ));
	}

	/**
	 * Adds Settings Page
	 */
	function load_settings_page() {
		if(get_option('guaven_dxtag_installed')==''){
			update_option('guaven_dxtag_installed',1);	
			$this->import_pre_installed_rules();	
		}
		require_once plugin_dir_path( __FILE__ ) . 'partials/guaven_dxtag-admin-display.php';
	}


	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/guaven_dxtag-admin.js', array( 'jquery' ), $this->version, false );

	}

	public static function dxtag_footer_help_text(){
		return '<br>	<br>	<br>	<br>	<br>
		<p style="text-align:right"><b>Need a help?</b> Read the plugin <a href="https://guaven.com/dxtag/"  target="_blank">Docs</a> 
		or just 
		<a  target="_blank" href="https://guaven.com/contact/solution-request/">contact us</a></p>';
	}

}
