<?php
/**
 * Drop Print Core Utility Functions (v1.1.2)
 * Centralizes helper functions for plugin setup (like global attributes),
 * operational utilities (like checking live order file usage), and managing
 * the retrieval of user-specific settings and data synchronization
 * (Pro status, media types, pricelists) with the Drop Print server.
 */
if(!defined('ABSPATH')){exit;}

function drop_print_create_global_attribute($name, $slug){
    if(!function_exists('wc_attribute_taxonomy_name')||!function_exists('wc_create_attribute')){
        error_log('[DropPrint Client ERROR] WooCommerce attribute functions not available for attribute: ' . $name);
        return false;
    }
    $taxonomy_name = wc_attribute_taxonomy_name($slug);
    if(!taxonomy_exists($taxonomy_name)){
        $attribute_id = wc_create_attribute(
            array(
                'name' => $name,
                'slug' => $slug,
                'type' => 'select',
                'order_by' => 'menu_order',
                'has_archives' => false,
            )
        );
        if(is_wp_error($attribute_id)){
            error_log('[DropPrint Client ERROR] Failed to create WooCommerce attribute: ' . $attribute_id->get_error_message());
            return false;
        }
        register_taxonomy(
            $taxonomy_name,
            array('product'),
            array(
                'labels' => array(
                    'name' => $name,
                    'singular_name' => $name,
                ),
                'hierarchical' => false,
                'show_ui' => false,
                'query_var' => true,
                'rewrite' => false,
            )
        );
        delete_transient('wc_attribute_taxonomies');
        return true;
    }
    return true;
}

function drop_print_is_file_in_live_order($afp_base_filename){
    if(empty($afp_base_filename)){
        return false;
    }
    $live_order_statuses = apply_filters('drop_print_live_order_statuses', array(
        'wc-pending',
        'wc-processing',
        'wc-on-hold',
    ));
    if(empty($live_order_statuses)){
        return false;
    }
    $orders = wc_get_orders(array(
        'status' => $live_order_statuses,
        'limit' => -1,
        'orderby' => 'date',
        'order' => 'DESC',
    ));
    if(empty($orders)){
        return false;
    }
    foreach($orders as $order){
        if(!$order){
            continue;
        }
        $items = $order->get_items();
        foreach($items as $item){
            if(!$item instanceof WC_Order_Item_Product){
                continue;
            }
            $product_id = $item->get_product_id();
            $parent_product_id = $product_id;
            if($item->get_variation_id() > 0){
                $parent_product_id = $item->get_product_id(); 
            }
            if($parent_product_id > 0){
                $drop_print_meta = get_post_meta($parent_product_id, '_drop_print_meta', true);
                if(is_array($drop_print_meta) && isset($drop_print_meta['file']) && $drop_print_meta['file'] === $afp_base_filename){
                    return true;
                }
            }
        }
    }
    return false;
}

function drop_print_get_default_user_settings(){
    return [
        'client_user_connection_id' => null,
        'public_key_pem_b64' => null,
        'auth_username_b64' => null,
        'auth_password_b64' => null,
        'pro_status' => 0,
        'billing_currency' => null,
        'available_media' => [],
        'media_last_updated' => 0,
        'user_display_name' => null,
        'price_margin' => 50,
        'default_media_id' => 'hpr-188',
        'file_handling' => 'upload',
        'payment_frequency' => 'pay_as_you_go',
        'auto_production' => 'off',
        'default_shipping_method' => 'standard',
        'default_image_resize' => 'fillPrintArea',
        'packing_slip_attachment_id' => 0,
        'us_sales_tax_collected' => 'false',
    ];
}

function drop_print_get_user_settings($user_id = 0){
    if($user_id === 0){
        $user_id = get_current_user_id();
    }
    if($user_id === 0){
        return drop_print_get_default_user_settings();
    }
    $defaults = drop_print_get_default_user_settings();
    $saved_settings = get_user_meta($user_id, 'drop_print_user_settings', true);
    if(!is_array($saved_settings)){
        $saved_settings = [];
    }
    return wp_parse_args($saved_settings, $defaults);
}

function drop_print_client_refresh_data_and_status($user_id = 0){
    if($user_id === 0){
        $user_id = get_current_user_id();
    }
    if($user_id === 0){
        error_log('[DropPrint Client ERROR] User context not found for settings refresh.');
        return ['status' => false, 'message' => __('User context not found for settings refresh.', 'drop-print'), 'data' => []];
    }

    $settings = drop_print_get_user_settings($user_id);
    $api_base_url = defined('DROP_PRINT_API_URI') ? DROP_PRINT_API_URI : null;
    $api_user_b64 = $settings['auth_username_b64'] ?? null;
    $api_pass_b64 = $settings['auth_password_b64'] ?? null;

    if(empty($api_base_url) || empty($api_user_b64) || empty($api_pass_b64)){
        error_log('[DropPrint Client ERROR] Not connected or API details incomplete in user settings for user ID: ' . $user_id);
        return ['status' => false, 'message' => __('Not connected or API details incomplete in user settings.', 'drop-print'), 'data' => []];
    }

    $api_username = base64_decode($api_user_b64);
    $api_password = base64_decode($api_pass_b64);

    if(false === $api_username || false === $api_password){
        error_log('[DropPrint Client ERROR] Failed to decode API credentials for user ID: ' . $user_id);
        return ['status' => false, 'message' => __('Failed to decode API credentials for user.', 'drop-print'), 'data' => []];
    }
    
    $auth_header = 'Basic '.base64_encode($api_username.':'.$api_password);
    $default_args = [
        'method' => 'GET',
        'headers' => ['Authorization' => $auth_header, 'Accept' => 'application/json'],
        'timeout' => 30,
    ];

    // --- PRO STATUS FETCH ---
    $pro_status_refreshed = false;
    $pro_status_url = trailingslashit($api_base_url).'subscription-status';
    $pro_response = wp_remote_get($pro_status_url, $default_args);
    $current_user_settings = drop_print_get_user_settings($user_id); 
    if(is_wp_error($pro_response)){
        error_log('[DropPrint Client ERROR] wp_remote_get error for PRO status for user ID ' . $user_id . ': ' . $pro_response->get_error_message());
    }else{
        $status_code = wp_remote_retrieve_response_code($pro_response);
        $body = wp_remote_retrieve_body($pro_response);
        $server_data = json_decode($body, true);
        if($status_code === 200 && isset($server_data['success'], $server_data['data']['is_pro']) && $server_data['success']){
            $current_user_settings['pro_status'] = (int) $server_data['data']['is_pro'];
            $pro_status_refreshed = true;
        }else{
            error_log('[DropPrint Client ERROR] PRO status response not successful or invalid format for user ID ' . $user_id . '. Status: ' . $status_code . ', Data: ' . print_r($server_data, true));
        }
    }

    // --- MEDIA TYPES FETCH ---
    $media_types_url = trailingslashit($api_base_url).'media-types';
    $media_response = wp_remote_get($media_types_url, $default_args);
    $fetched_media_types = [];
    $media_types_fetched_successfully = false;
    if(is_wp_error($media_response)){
        error_log('[DropPrint Client ERROR] wp_remote_get error for media types for user ID ' . $user_id . ': ' . $media_response->get_error_message());
    }else{
        $media_status_code = wp_remote_retrieve_response_code($media_response);
        $media_body = wp_remote_retrieve_body($media_response);
        $media_data = json_decode($media_body, true);
        if($media_status_code === 200 && is_array($media_data)){
            $sanitized_media_data = [];
            foreach($media_data as $m_item){
                if(isset($m_item['id']) && isset($m_item['name']) && is_string($m_item['id']) && is_string($m_item['name'])){
                    $sanitized_media_data[] = ['id' => sanitize_key($m_item['id']), 'name' => sanitize_text_field($m_item['name'])];
                } else {
                    error_log('[DropPrint Client WARNING] Skipping malformed media item for user ID ' . $user_id . ': ' . print_r($m_item, true));
                }
            }
            $current_user_settings['available_media'] = $sanitized_media_data;
            $current_user_settings['media_last_updated'] = time();
            $fetched_media_types = $sanitized_media_data;
            $media_types_fetched_successfully = true;
        }else{
            error_log('[DropPrint Client ERROR] Media types response not successful or invalid format for user ID ' . $user_id . '. Status: ' . $media_status_code . ', Data: ' . print_r($media_data, true));
        }
    }

    update_user_meta($user_id, 'drop_print_user_settings', $current_user_settings);

    if(empty($fetched_media_types) && !empty($current_user_settings['available_media']) && $media_types_fetched_successfully){
        $fetched_media_types = $current_user_settings['available_media'];
    }elseif(empty($fetched_media_types) && !empty($current_user_settings['available_media'])){
        $fetched_media_types = $current_user_settings['available_media'];
    }

    // --- PRICELISTS FETCH ---
    $all_pricelists_fetched_and_saved_correctly = true;
    if(!empty($fetched_media_types)){
        $pricelist_cache_dir = DROP_PRINT_CACHE_DIR;
        if(!file_exists($pricelist_cache_dir) && !wp_mkdir_p($pricelist_cache_dir)){
            error_log('[DropPrint Client ERROR] Failed to create local cache directory: ' . $pricelist_cache_dir . ' for user ID: ' . $user_id);
            return ['status' => false, 'message' => __('Failed to create local cache directory.', 'drop-print'), 'data' => ['pro_status_updated' => $pro_status_refreshed]];
        }
        
        foreach($fetched_media_types as $media_item){
            if(empty($media_item['id'])) {
                error_log('[DropPrint Client WARNING] Skipping pricelist fetch for empty media item ID for user ID ' . $user_id);
                continue;
            }
            $pricelist_url = trailingslashit($api_base_url).'pricelist/'.$media_item['id'];
            $pl_response = wp_remote_get($pricelist_url, $default_args);
            if(is_wp_error($pl_response)){
                error_log('[DropPrint Client ERROR] wp_remote_get error for pricelist ' . $media_item['id'] . ' for user ID ' . $user_id . ': ' . $pl_response->get_error_message());
                $all_pricelists_fetched_and_saved_correctly = false;
                continue;
            }
            $pl_status_code = wp_remote_retrieve_response_code($pl_response);
            $pl_body = wp_remote_retrieve_body($pl_response);
            if($pl_status_code === 200){
                $decoded_body = json_decode($pl_body, true);
                if(json_last_error() === JSON_ERROR_NONE && is_array($decoded_body) && isset($decoded_body['products'], $decoded_body['media_id']) && $decoded_body['media_id'] === $media_item['id']){
                    $local_pricelist_filename = sanitize_key($media_item['id']).".json";
                    $local_pricelist_file = trailingslashit($pricelist_cache_dir).$local_pricelist_filename;
                    
                    if(file_put_contents($local_pricelist_file, $pl_body) === false){
                        error_log('[DropPrint Client ERROR] Failed to write pricelist to local file: ' . $local_pricelist_file . ' for media ID ' . $media_item['id'] . ', user ID ' . $user_id);
                        $all_pricelists_fetched_and_saved_correctly = false;
                    }
                }else{
                    error_log('[DropPrint Client ERROR] Invalid JSON or missing "products"/"media_id" in pricelist for ' . $media_item['id'] . ', user ID ' . $user_id . '. JSON Error: ' . json_last_error_msg() . ' | Decoded: ' . print_r($decoded_body, true));
                    $all_pricelists_fetched_and_saved_correctly = false;
                }
            }else{
                error_log('[DropPrint Client ERROR] Non-200 HTTP status code for pricelist ' . $media_item['id'] . ', user ID ' . $user_id . ': ' . $pl_status_code . '. Full body: ' . $pl_body);
                $all_pricelists_fetched_and_saved_correctly = false;
            }
        }
    }else{
        $all_pricelists_fetched_and_saved_correctly = $media_types_fetched_successfully;
    }

    $final_status = $pro_status_refreshed && $media_types_fetched_successfully && $all_pricelists_fetched_and_saved_correctly;
    if(empty($fetched_media_types) && $media_types_fetched_successfully && $pro_status_refreshed){
        $final_status = true;
        $all_pricelists_fetched_and_saved_correctly = true;
    }

    $message = $final_status ?
        __('Subscription status, media types, and pricelists refreshed successfully for current user.', 'drop-print') :
        __('Refresh process for current user completed. Some items may not have updated. Check logs for details.', 'drop-print');
    
    return [
        'status' => $final_status,
        'message' => $message,
        'data' => ['pro_status_updated' => $pro_status_refreshed, 'media_types_count' => count($fetched_media_types), 'pricelists_synced' => $all_pricelists_fetched_and_saved_correctly]
    ];
}