<?php
/**
 * Drop Print Client - AFP File Usage Tracker & Management Utilities. (v1.1)
 * This file provides functions to track which products use specific .afp files
 * via the 'drop_print_files' WordPress option.
 * It includes helper functions to add/remove references, and hooks into product
 * duplication and deletion events to maintain accurate tracking. It also contains logic to
 * check if a file is in use by live orders, enabling safer conditional deletion of .afp files
 * during product reset or deletion to prevent data loss.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
function drop_print_get_afp_filename_from_meta( int $product_id ): ?string {
	if ( ! $product_id ) {
		return null;
	}
	$meta = get_post_meta( $product_id, '_drop_print_meta', true );
	if ( is_array( $meta ) && ! empty( $meta['file'] ) ) {
		$base_filename = sanitize_file_name( $meta['file'] );
		return $base_filename . '.afp';
	}
	return null;
}
function drop_print_add_product_to_file_reference( string $afp_filename, int $product_id ): void {
	if ( empty( $afp_filename ) || ! $product_id ) {
		return;
	}
	$references = get_option( 'drop_print_files', [] );
	if ( ! is_array( $references ) ) {
		$references = [];
	}
	if ( ! isset( $references[ $afp_filename ] ) || ! is_array( $references[ $afp_filename ] ) ) {
		$references[ $afp_filename ] = [];
	}
	if ( ! in_array( $product_id, $references[ $afp_filename ], true ) ) {
		$references[ $afp_filename ][] = $product_id;
	}
	update_option( 'drop_print_files', $references );
}
function drop_print_remove_product_from_file_reference( string $afp_filename, int $product_id ): bool {
	if ( empty( $afp_filename ) || ! $product_id ) {
		return true;
	}
	$references = get_option( 'drop_print_files', [] );
	if ( ! is_array( $references ) || ! isset( $references[ $afp_filename ] ) || ! is_array( $references[ $afp_filename ] ) ) {
		return true;
	}
	$references[ $afp_filename ] = array_values( array_diff( $references[ $afp_filename ], [ $product_id ] ) );
	if ( empty( $references[ $afp_filename ] ) ) {
		unset( $references[ $afp_filename ] );
		update_option( 'drop_print_files', $references );
		return true;
	}
	update_option( 'drop_print_files', $references );
	return false;
}
function drop_print_is_afp_file_in_use_by_products( string $afp_filename ): bool {
    if ( empty( $afp_filename ) ) {
        return false;
    }
    $references = get_option( 'drop_print_files', [] );
    return isset( $references[ $afp_filename ] ) && ! empty( $references[ $afp_filename ] );
}
function drop_print_is_afp_file_in_use_by_live_orders( string $afp_filename_to_check ): bool {
	if ( empty( $afp_filename_to_check ) || ! class_exists( 'WooCommerce' ) ) {
		return false;
	}
	$active_order_statuses = wc_get_order_statuses();
	$final_statuses_to_exclude = ['wc-completed', 'wc-refunded', 'wc-cancelled', 'wc-failed', 'wc-trash'];
	foreach ( $final_statuses_to_exclude as $status_to_exclude ) {
		unset( $active_order_statuses[ $status_to_exclude ] );
	}
	$active_order_status_keys = array_keys( $active_order_statuses );
	if ( empty( $active_order_status_keys ) ) {
		return false;
	}
	$orders = wc_get_orders( [
		'status' => $active_order_status_keys,
		'limit'  => -1,
		'type'   => 'shop_order',
	] );
	if ( empty( $orders ) ) {
		return false;
	}
	$afp_filename_to_check_lower = strtolower( $afp_filename_to_check );
	foreach ( $orders as $order ) {
		if ( ! $order instanceof WC_Order ) {
			continue;
		}
		foreach ( $order->get_items() as $item ) {
			if ( ! $item instanceof WC_Order_Item_Product ) {
				continue;
			}
			$product_id = $item->get_product_id();
			$parent_id  = $product_id;
			if ( $item->get_variation_id() ) {
				$variation = wc_get_product( $item->get_variation_id() );
				if ( $variation ) {
					$parent_id = $variation->get_parent_id();
				}
			}
			$product_afp_filename = drop_print_get_afp_filename_from_meta( $parent_id );
			if ( $product_afp_filename && strtolower( $product_afp_filename ) === $afp_filename_to_check_lower ) {
				return true;
			}
		}
	}
	return false;
}
function drop_print_handle_product_duplication_afp_tracking( $new_product_id, $original_product ) {
	if ( $original_product instanceof WC_Product && $original_product->is_type( 'variable' ) ) {
		$original_product_id = $original_product->get_id();
		$afp_filename        = drop_print_get_afp_filename_from_meta( $original_product_id );
		if ( $afp_filename ) {
			drop_print_add_product_to_file_reference( $afp_filename, $new_product_id );
		}
	}
}
add_action( 'woocommerce_duplicate_product', 'drop_print_handle_product_duplication_afp_tracking', 10, 2 );
function drop_print_handle_product_deletion_afp_tracking( $post_id ) {
	if ( get_post_type( $post_id ) !== 'product' ) {
		return;
	}
	$product = wc_get_product( $post_id );
	if ( ! $product || ! $product->is_type( 'variable' ) ) {
		return;
	}
	$afp_filename = drop_print_get_afp_filename_from_meta( $post_id );
	if ( $afp_filename ) {
		$file_became_unreferenced_by_products = drop_print_remove_product_from_file_reference( $afp_filename, $post_id );
		if ( $file_became_unreferenced_by_products ) {
			if ( ! drop_print_is_afp_file_in_use_by_live_orders( $afp_filename ) ) {
				$upload_dir = wp_upload_dir();
				$base_drop_print_dir = defined( 'DROP_PRINT_UPLOAD_BASE_DIR' ) ? DROP_PRINT_UPLOAD_BASE_DIR : trailingslashit( $upload_dir['basedir'] ) . 'drop-print';
				$file_path           = trailingslashit( $base_drop_print_dir ) . $afp_filename;
				if ( file_exists( $file_path ) ) {
                    wp_delete_file( $file_path );
				}
			}
		}
	}
}
add_action( 'before_delete_post', 'drop_print_handle_product_deletion_afp_tracking', 10, 1 );