<?php
/**
 * Drop Print Client - AJAX Handlers for Quote & Order Actions (v1.1)
 * Contains WordPress AJAX action handlers for client-side JavaScript. These
 * handlers facilitate communication with the Drop Print server using user-specific
 * settings for initiating quotes, calculating totals, creating payment intents,
 * and updating local order status post-payment.
 */
if(!defined('ABSPATH')){exit;}
if(!function_exists('drop_print_get_client_server_auth_details_for_ajax')){
    function drop_print_get_client_server_auth_details_for_ajax(){
        $current_user_id = get_current_user_id();
        if($current_user_id === 0){return new WP_Error('user_not_logged_in', __('User not logged in or not identifiable for API details.', 'drop-print'));}
        $user_settings = drop_print_get_user_settings($current_user_id);
        $api_base_url = defined('DROP_PRINT_API_URI') ? DROP_PRINT_API_URI : ($user_settings['api_uri_user_specific'] ?? null);
        $auth_username_b64 = $user_settings['auth_username_b64'] ?? null;
        $auth_password_b64 = $user_settings['auth_password_b64'] ?? null;
        if(empty($api_base_url) || empty($auth_username_b64) || empty($auth_password_b64)){return new WP_Error('api_config_error', __('API connection details incomplete in your user settings. Check Drop Print settings.', 'drop-print'));}
        $api_username = base64_decode($auth_username_b64);
        $api_password = base64_decode($auth_password_b64);
        if($api_username === false || $api_password === false){return new WP_Error('api_auth_decode_error', __('Could not decode API credentials from your user settings. Re-connect via plugin settings.', 'drop-print'));}
        return ['base_url' => $api_base_url, 'username' => $api_username, 'password' => $api_password];
    }
}
if(!function_exists('drop_print_make_server_request_for_ajax')){
    function drop_print_make_server_request_for_ajax(string $server_endpoint_slug, array $payload = null, string $method = 'POST', int $timeout = 45){
        $auth_details = drop_print_get_client_server_auth_details_for_ajax();
        if(is_wp_error($auth_details)){return $auth_details;}
        $api_url = trailingslashit($auth_details['base_url']).$server_endpoint_slug;
        $client_plugin_version = defined('DROP_PRINT_VERSION') ? DROP_PRINT_VERSION : '1.1';
        $args = array(
            'method' => $method,
            'headers' => array(
                'Authorization' => 'Basic '.base64_encode($auth_details['username'].':'.$auth_details['password']),
                'Content-Type' => 'application/json; charset=utf-8',
                'Accept' => 'application/json',
                'User-Agent' => 'DropPrintClientAjax/'.esc_attr($client_plugin_version)
            ),
            'timeout' => $timeout
        );
        if($method === 'POST' && !empty($payload)){$args['body'] = wp_json_encode($payload);}
        $response = $method === 'POST' ? wp_remote_post($api_url, $args) : wp_remote_get($api_url, $args);
        if(is_wp_error($response)){return $response;}
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $server_response_data = json_decode($body, true);
        if($status_code >= 200 && $status_code < 300 && (isset($server_response_data['success']) && $server_response_data['success'] === true || $method === 'GET')){
            if($method === 'GET' && $server_response_data !== null){
                return $server_response_data;
            }elseif(isset($server_response_data['data'])){
                 return $server_response_data['data'];
            }
            return $server_response_data;
        }else{
            $error_message = __('Unknown server error.', 'drop-print');
            $error_code = isset($server_response_data['code']) ? $server_response_data['code'] : (isset($server_response_data['data']['code']) ? $server_response_data['data']['code'] : null);
            if(isset($server_response_data['message'])){$error_message = $server_response_data['message'];}
            elseif(isset($server_response_data['data']) && isset($server_response_data['data']['message'])){$error_message = $server_response_data['data']['message'];if(isset($server_response_data['data']['code']))$error_code = $server_response_data['data']['code'];}
            elseif(!empty($body) && is_string($body) && json_last_error() !== JSON_ERROR_NONE){$error_message = substr(wp_strip_all_tags($body), 0, 250).'...';}
            $wp_error_data_to_return = array('status' => $status_code, 'body' => $server_response_data, 'code_from_server' => $error_code);
            $wp_error = new WP_Error($error_code ?: 'server_api_error', $error_message, $wp_error_data_to_return);
            return $wp_error;
        }
    }
}
function drop_print_initiate_quote_session_ajax_handler(){
    if(!isset($_POST['nonce'])||!wp_verify_nonce(sanitize_key($_POST['nonce']),'drop_print_initiate_quote_session_action')){wp_send_json_error(['message'=>__('Security check failed.','drop-print'),'code'=>'nonce_failure'],403);return;}
    if(!current_user_can('edit_shop_orders')){wp_send_json_error(['message'=>__('You do not have permission.','drop-print'),'code'=>'permission_denied'],403);return;}
    $client_order_id=isset($_POST['client_order_id'])?absint(wp_unslash($_POST['client_order_id'])):0;
    if(!$client_order_id){wp_send_json_error(['message'=>__('Missing Client Order ID.','drop-print'),'code'=>'missing_order_id'],400);return;}
    $order=wc_get_order($client_order_id);
    if(!$order){wp_send_json_error(['message'=>__('Could not retrieve order.','drop-print'),'code'=>'order_not_found'],404);return;}
    $current_user_id = get_current_user_id();
    if($current_user_id === 0){wp_send_json_error(['message'=>__('User context not found.','drop-print'),'code'=>'user_context_missing'],403);return;}
    $user_settings = drop_print_get_user_settings($current_user_id);
    if(!function_exists('drop_print_build_unified_order_payload')){wp_send_json_error(['message'=>__('Client plugin error: Payload builder missing.','drop-print'),'code'=>'builder_missing'],500);return;}
    $quote_intake_payload=drop_print_build_unified_order_payload($order, $user_settings, "quote_intake_manual", false, ($user_settings['default_shipping_method']??'standard'));
    if(empty($quote_intake_payload)||empty($quote_intake_payload['items'])){wp_send_json_error(['message'=>__('No valid items to quote. This may be due to missing or inaccessible print files (e.g., 404 errors on asset URLs), or other missing required product information. Please check product setup and order details.','drop-print'),'code'=>'no_quotable_items'],400);return;}
    $server_response_data=drop_print_make_server_request_for_ajax('quotes/session',$quote_intake_payload);
    if(is_wp_error($server_response_data)){
        $error_message_to_js=$server_response_data->get_error_message();
        $error_code_to_js=$server_response_data->get_error_code();
        $error_data_wrapper=$server_response_data->get_error_data();
        $specific_validation_errors=null;
        if(isset($error_data_wrapper['body']['data']['details'])&&is_array($error_data_wrapper['body']['data']['details'])&&!empty($error_data_wrapper['body']['data']['details'])){
            $specific_validation_errors=$error_data_wrapper['body']['data']['details'];
        }elseif(isset($error_data_wrapper['body']['details'])&&is_array($error_data_wrapper['body']['details'])&&!empty($error_data_wrapper['body']['details'])){
            $specific_validation_errors=$error_data_wrapper['body']['details'];
        }
        $response_status_code=400;
        if(isset($error_data_wrapper['status'])){
            $response_status_code=(int)$error_data_wrapper['status'];
        }elseif(isset($error_data_wrapper['body']['data']['status'])){
            $response_status_code=(int)$error_data_wrapper['body']['data']['status'];
        }
        $data_for_js_error=['message'=>$error_message_to_js,'code'=>$error_code_to_js,'details'=>$specific_validation_errors];
        wp_send_json_error($data_for_js_error,$response_status_code);
        return;
    }else{
        if(!isset($server_response_data['stripe_publishable_key'])||empty($server_response_data['stripe_publishable_key'])){
            $server_response_data['stripe_publishable_key']=$user_settings['stripe_publishable_key']??'';
        }
        wp_send_json_success($server_response_data);
    }
}
add_action('wp_ajax_drop_print_initiate_quote_session','drop_print_initiate_quote_session_ajax_handler');
function drop_print_calculate_client_total_ajax_handler(){
    if(!isset($_POST['nonce'])||!wp_verify_nonce(sanitize_key($_POST['nonce']),'drop_print_calculate_client_total_action')){wp_send_json_error(['message'=>__('Security check failed.','drop-print'),'code'=>'nonce_failure'],403);return;}
    if(!current_user_can('edit_shop_orders')){wp_send_json_error(['message'=>__('You do not have permission.','drop-print'),'code'=>'permission_denied'],403);return;}
    $quote_id=isset($_POST['quote_id'])?sanitize_text_field(wp_unslash($_POST['quote_id'])):'';
    $shipping_method=isset($_POST['shipping_method_preference'])?sanitize_key(wp_unslash($_POST['shipping_method_preference'])):'standard';
    if(empty($quote_id)){wp_send_json_error(['message'=>__('Missing Quote ID.','drop-print'),'code'=>'missing_quote_id'],400);return;}
    $payload_to_server=['quote_id'=>$quote_id,'shipping_method_preference'=>$shipping_method];
    $server_response_data=drop_print_make_server_request_for_ajax('quotes/calculate',$payload_to_server);
    if(is_wp_error($server_response_data)){
        $error_data_for_js=['message'=>$server_response_data->get_error_message(),'code'=>$server_response_data->get_error_code()];
        $error_details=$server_response_data->get_error_data();
        $response_status=isset($error_details['status'])?(int)$error_details['status']:400;
        if(isset($error_details['body']['data']['quote_details']['shipping_unavailable'])&&$error_details['body']['data']['quote_details']['shipping_unavailable']===true){$error_data_for_js=$error_details['body']['data'];wp_send_json_success($error_data_for_js);return;}
        wp_send_json_error($error_data_for_js,$response_status);
    }else{wp_send_json_success($server_response_data);}
}
add_action('wp_ajax_drop_print_calculate_client_total','drop_print_calculate_client_total_ajax_handler');
function drop_print_create_client_payment_intent_ajax_handler(){
    if(!isset($_POST['nonce'])||!wp_verify_nonce(sanitize_key($_POST['nonce']),'drop_print_create_client_payment_intent_action')){wp_send_json_error(['message'=>__('Security check failed.','drop-print'),'code'=>'nonce_failure'],403);return;}
    if(!current_user_can('edit_shop_orders')){wp_send_json_error(['message'=>__('You do not have permission.','drop-print'),'code'=>'permission_denied'],403);return;}
    $quote_id=isset($_POST['quote_id'])?sanitize_text_field(wp_unslash($_POST['quote_id'])):'';
    $shipping_method=isset($_POST['shipping_method_preference'])?sanitize_key(wp_unslash($_POST['shipping_method_preference'])):'standard';
    $use_saved_card=isset($_POST['use_saved_card'])?rest_sanitize_boolean(wp_unslash($_POST['use_saved_card'])):false;
    $saved_card_id=isset($_POST['saved_card_id'])?sanitize_text_field(wp_unslash($_POST['saved_card_id'])):null;
    if(empty($quote_id)){wp_send_json_error(['message'=>__('Missing Quote ID for payment intent.','drop-print'),'code'=>'missing_quote_id_for_pi'],400);return;}
    $payload_to_server=['quote_id'=>$quote_id,'shipping_method_preference'=>$shipping_method,'charge_automatically'=>false];
    if($use_saved_card&&!empty($saved_card_id)){$payload_to_server['payment_method_id_to_use_if_not_default']=$saved_card_id;}
    $server_response_data=drop_print_make_server_request_for_ajax('payments/intent',$payload_to_server);
    if(is_wp_error($server_response_data)){
        $error_data_for_js=['message'=>$server_response_data->get_error_message(),'code'=>$server_response_data->get_error_code()];
        $error_details=$server_response_data->get_error_data();
        $response_status=isset($error_details['status'])?(int)$error_details['status']:400;
        wp_send_json_error($error_data_for_js,$response_status);
    }else{wp_send_json_success($server_response_data);}
}
add_action('wp_ajax_drop_print_create_client_payment_intent','drop_print_create_client_payment_intent_ajax_handler');
add_action('wp_ajax_nopriv_drop_print_create_client_payment_intent','drop_print_create_client_payment_intent_ajax_handler');
function drop_print_update_client_order_post_payment_ajax_handler(){
    if(!isset($_POST['nonce'])||!wp_verify_nonce(sanitize_key($_POST['nonce']),'drop_print_update_client_order_post_payment_action')){wp_send_json_error(['message'=>__('Security check failed.','drop-print'),'code'=>'nonce_failure'],403);return;}
    if(!current_user_can('edit_shop_orders')){wp_send_json_error(['message'=>__('You do not have permission.','drop-print'),'code'=>'permission_denied'],403);return;}
    $client_order_id=isset($_POST['client_order_id'])?absint(wp_unslash($_POST['client_order_id'])):0;
    $quote_id=isset($_POST['quote_id'])?sanitize_text_field(wp_unslash($_POST['quote_id'])):null;
    $payment_intent_id=isset($_POST['payment_intent_id'])?sanitize_text_field(wp_unslash($_POST['payment_intent_id'])):null;
    $status_to_set=isset($_POST['status'])?sanitize_key(wp_unslash($_POST['status'])):'awaiting_webhook_confirmation';
    if(!$client_order_id||!$quote_id||!$payment_intent_id){wp_send_json_error(['message'=>__('Missing required data to update order post-payment.','drop-print'),'code'=>'missing_post_payment_data'],400);return;}
    $order=wc_get_order($client_order_id);
    if(!$order){wp_send_json_error(['message'=>__('Order not found.','drop-print'),'code'=>'order_not_found_post_payment'],404);return;}
    $current_details=$order->get_meta('_drop_print_order_details',true);
    if(!is_array($current_details)){$current_details=[];}
    $new_details=array_merge($current_details,['status'=>$status_to_set,'quote_id'=>$quote_id,'payment_intent_id'=>$payment_intent_id,'last_client_update'=>time()]);
    $order->update_meta_data('_drop_print_order_details',$new_details);
    $order->save_meta_data();
    wp_send_json_success(['message'=>__('Client order updated post-payment.','drop-print')]);
}
add_action('wp_ajax_drop_print_update_client_order_post_payment','drop_print_update_client_order_post_payment_ajax_handler');