<?php
/**
 * Drop Print Client - File Upload Handler (v1.1)
 * Handles chunked file uploads via AJAX. Assembles chunks, generates a
 * WordPress media library thumbnail, encrypts the original image using the
 * current user's public key, and wraps it in the .afp file format.
 * Tracks .afp file usage by product ID.
 */
if(!defined('ABSPATH'))exit;
require_once ABSPATH.'wp-admin/includes/image.php';
require_once ABSPATH.'wp-admin/includes/file.php';
require_once ABSPATH.'wp-admin/includes/media.php';
function drop_print_ensure_dir_exists_upload_handler($path){
    if(!file_exists($path)){
        if(!wp_mkdir_p($path)){
            return false;
        }
        @chmod($path,0755);
    }
    if(!is_writable($path)){
        if(!@chmod($path,0755)||!is_writable($path)){
            return false;
        }
    }
    return true;
}
function drop_print_cleanup_chunks_upload_handler($path){
    if(empty($path)||!is_dir($path))return;
    try{
        $iterator=new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($path,FilesystemIterator::SKIP_DOTS|FilesystemIterator::UNIX_PATHS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach($iterator as $file){
            if($file->isFile()&&'part'===$file->getExtension()) wp_delete_file($file->getRealPath());
        }
        if(!(new \FilesystemIterator($path))->valid()) @rmdir($path);
    }catch(Exception $e){
    }
}
function drop_print_handle_chunked_upload(){
    $final_filepath=null;
    $temp_chunk_dir=null;
    $generated_thumbnail_for_afp_path=null;
    $generated_thumbnail_data=null;
    $generated_thumbnail_mime_type=null;
    $compressed_filepath=null;
    $afp_filepath=null;
    $public_key_resource=null;
    $afp_file_handle=null;
    $temp_assembled_filepath=null;
    $process_succeeded=false;
    try{
        check_ajax_referer('drop_print_nonce','nonce');
        $current_user_id = get_current_user_id();
        if($current_user_id === 0){
            throw new Exception(__('User not identified for upload.','drop-print'),403);
        }
        if(!current_user_can('upload_files')||!current_user_can('edit_products')){
            throw new Exception(__('Permission denied.','drop-print'),403);
        }
        $product_id=absint($_POST['product_id']??0);
        if(!$product_id)throw new Exception(__('Missing or invalid Product ID.','drop-print'),400);
        $user_settings = drop_print_get_user_settings($current_user_id);
        $client_connection_id_from_settings = $user_settings['client_user_connection_id'] ?? null;
        if(empty($client_connection_id_from_settings)){
            throw new Exception(__('Client connection ID is not configured in your user settings. Cannot process file.','drop-print'),412);
        }
        if(!isset($_FILES['file'])||empty($_FILES['file']['tmp_name'])||!is_uploaded_file($_FILES['file']['tmp_name'])|| !isset($_FILES['file']['error']) ||UPLOAD_ERR_OK!==$_FILES['file']['error']){
            $error_code = isset($_FILES['file']['error']) ? absint($_FILES['file']['error']) : UPLOAD_ERR_NO_FILE;
            $upload_errors=[
                UPLOAD_ERR_INI_SIZE=>__('File exceeds upload_max_filesize directive in php.ini.','drop-print'),
                UPLOAD_ERR_FORM_SIZE=>__('File exceeds MAX_FILE_SIZE directive specified in HTML form.','drop-print'),
                UPLOAD_ERR_PARTIAL=>__('File was only partially uploaded.','drop-print'),
                UPLOAD_ERR_NO_FILE=>__('No file was uploaded.','drop-print'),
                UPLOAD_ERR_NO_TMP_DIR=>__('Missing a temporary folder.','drop-print'),
                UPLOAD_ERR_CANT_WRITE=>__('Failed to write file to disk.','drop-print'),
                UPLOAD_ERR_EXTENSION=>__('A PHP extension stopped the file upload.','drop-print'),
            ];
            throw new Exception(__('Upload Error: ','drop-print').($upload_errors[$error_code]??__('Unknown upload error','drop-print')).' (Code: '.$error_code.')',400);
        }
        $required_params=['dzuuid','dzchunkindex','dztotalfilesize','dzchunksize','dztotalchunkcount','dzfilename'];
        $params=[];
        foreach($required_params as $param){
            if(!isset($_POST[$param])||''===$_POST[$param]){
                // translators: %1$s: Name of the missing parameter.
                throw new Exception(sprintf(__('Missing required chunk parameter: %1$s','drop-print'),esc_html($param)),400);
            }
            if('dzuuid'===$param)$params[$param]=sanitize_key($_POST[$param]);
            elseif(in_array($param,['dzchunkindex','dztotalfilesize','dzchunksize','dztotalchunkcount'],true))$params[$param]=absint($_POST[$param]);
            elseif('dzfilename'===$param)$params[$param]=sanitize_file_name(wp_unslash($_POST[$param]));
            else $params[$param]=sanitize_text_field(wp_unslash($_POST[$param]));
        }
        if($params['dztotalchunkcount']<=0||$params['dzchunkindex']<0||$params['dzchunkindex']>=$params['dztotalchunkcount']){
            throw new Exception(__('Invalid chunk index/count received.','drop-print'),400);
        }
        $upload_dir_info=wp_upload_dir();
        if(!empty($upload_dir_info['error']))throw new Exception(__('WordPress upload directory error: ','drop-print').$upload_dir_info['error'],500);
        $final_basedir=trailingslashit($upload_dir_info['basedir']).'drop-print';
        $chunks_basedir=trailingslashit($final_basedir).'chunks';
        $final_baseurl=trailingslashit($upload_dir_info['baseurl']).'drop-print';
        if(!drop_print_ensure_dir_exists_upload_handler($final_basedir)||!drop_print_ensure_dir_exists_upload_handler($chunks_basedir)){
            throw new Exception(__('Upload directory setup failed. Check permissions.','drop-print'),500);
        }
        $file_uuid=$params['dzuuid'];
        $chunk_index=$params['dzchunkindex'];
        $total_chunks=$params['dztotalchunkcount'];
        $original_name=$params['dzfilename'];
        $temp_chunk_dir=trailingslashit($chunks_basedir).$file_uuid;
        $chunk_filepath=trailingslashit($temp_chunk_dir).$chunk_index.'.part';
        if(!drop_print_ensure_dir_exists_upload_handler($temp_chunk_dir))throw new Exception(__('Cannot create temporary chunk directory.','drop-print'),500);
        if(!move_uploaded_file($_FILES['file']['tmp_name'],$chunk_filepath))throw new Exception(__('Failed to move uploaded chunk.','drop-print'),500);
        clearstatcache();
        $chunk_files=is_dir($temp_chunk_dir)?array_diff(scandir($temp_chunk_dir),['.','..']):[];
        if(count($chunk_files)<$total_chunks){
            wp_send_json_success(['message'=>__('Chunk received.','drop-print'),'chunk_index'=>$chunk_index]);
        }
        $temp_assembled_filename=wp_unique_filename($final_basedir,'temp_assembly_'.$file_uuid.'_'.$original_name);
        $temp_assembled_filepath=trailingslashit($final_basedir).$temp_assembled_filename;
        $output_resource=null;
        $assembly_success=false;
        try{
            $output_resource=@fopen($temp_assembled_filepath,'wb');
            if(!$output_resource){
                // translators: %1$s: Filename.
                throw new Exception(sprintf(__('Failed to open temporary assembly file for writing: %1$s','drop-print'),esc_html(basename($temp_assembled_filepath))));
            }
            for($i=0;$i<$total_chunks;$i++){
                $chunk_part_path=trailingslashit($temp_chunk_dir).$i.'.part';
                if(!file_exists($chunk_part_path)){
                    // translators: %1$d: Chunk number.
                    throw new Exception(sprintf(__('Missing chunk %1$d during assembly.','drop-print'),$i));
                }
                $chunk_resource=@fopen($chunk_part_path,'rb');
                if(!$chunk_resource){
                    // translators: %1$d: Chunk number.
                    throw new Exception(sprintf(__('Cannot open chunk %1$d.','drop-print'),$i));
                }
                while(!feof($chunk_resource)){
                    $buffer=fread($chunk_resource,8192);
                    if(false===$buffer){
                        @fclose($chunk_resource);
                        // translators: %1$d: Chunk number.
                        throw new Exception(sprintf(__('Error reading chunk %1$d.','drop-print'),$i));
                    }
                    if(false===fwrite($output_resource,$buffer)){
                        @fclose($chunk_resource);
                        // translators: %1$d: Chunk number.
                        throw new Exception(sprintf(__('Error writing chunk %1$d to assembly file.','drop-print'),$i));
                    }
                }
                @fclose($chunk_resource);
            }
            $assembly_success=true;
        }finally{
            if($output_resource)@fclose($output_resource);
        }
        drop_print_cleanup_chunks_upload_handler($temp_chunk_dir);
        $temp_chunk_dir=null;
        clearstatcache(true,$temp_assembled_filepath);
        if(!$assembly_success||!file_exists($temp_assembled_filepath)||filesize($temp_assembled_filepath)<=0){
            if(file_exists($temp_assembled_filepath)) wp_delete_file($temp_assembled_filepath);
            $temp_assembled_filepath=null;
            throw new Exception(__('File assembly failed or result is empty.','drop-print'),500);
        }
        $pathinfo=pathinfo($original_name);
        $safe_original_base=sanitize_title($pathinfo['filename']);
        $original_extension=isset($pathinfo['extension'])?strtolower($pathinfo['extension']):'';
        $new_base_name=$safe_original_base.'-'.$file_uuid;
        $new_final_filename=$new_base_name.($original_extension?'.'.$original_extension:'');
        $new_final_filename=wp_unique_filename($final_basedir,$new_final_filename);
        $final_filepath=trailingslashit($final_basedir).$new_final_filename;
        $final_filename_saved=$new_final_filename;
        if(!rename($temp_assembled_filepath,$final_filepath)){
            $final_filepath=$temp_assembled_filepath;
            $final_filename_saved=$temp_assembled_filename;
            $new_base_name=pathinfo($final_filename_saved,PATHINFO_FILENAME);
        }
        $temp_assembled_filepath=null;
        $attachment_id=null;
        $media_library_thumb_url=null;
        $width=null;
        $height=null;
        $original_hash=null;
        $hash_algo='sha256';
        $meta_changed=false;
        $meta_array=get_post_meta($product_id,'_drop_print_meta',true);
        if(!is_array($meta_array))$meta_array=[];
        $image_editor=wp_get_image_editor($final_filepath);
        if(!is_wp_error($image_editor)){
            $resize_op_result=$image_editor->resize(300,300,false);
            if(!is_wp_error($resize_op_result)){
                $path_parts_for_thumb=pathinfo($final_filename_saved);
                $temp_thumb_for_afp_filename=wp_unique_filename($final_basedir,$path_parts_for_thumb['filename'].'-afpthumb.'.($path_parts_for_thumb['extension']?:'jpg'));
                $generated_thumbnail_for_afp_path=trailingslashit($final_basedir).$temp_thumb_for_afp_filename;
                $saved_afp_thumb_details=$image_editor->save($generated_thumbnail_for_afp_path);
                if(!is_wp_error($saved_afp_thumb_details)&&file_exists($generated_thumbnail_for_afp_path)){
                    $generated_thumbnail_data=file_get_contents($generated_thumbnail_for_afp_path);
                    $generated_thumbnail_mime_type=$saved_afp_thumb_details['mime-type']??mime_content_type($generated_thumbnail_for_afp_path);
                    if(false===$generated_thumbnail_data||empty($generated_thumbnail_mime_type)){
                        $generated_thumbnail_data=null;
                        $generated_thumbnail_mime_type=null;
                    }
                    $wp_ml_thumb_filename=wp_unique_filename($final_basedir,$path_parts_for_thumb['filename'].'-wpmlthumb.'.($path_parts_for_thumb['extension']?:'jpg'));
                    $wp_ml_thumb_path=trailingslashit($final_basedir).$wp_ml_thumb_filename;
                    if(copy($generated_thumbnail_for_afp_path,$wp_ml_thumb_path)){
                        $file_array_for_ml=['name'=>basename($wp_ml_thumb_path),'tmp_name'=>$wp_ml_thumb_path,'error'=>0,'size'=>filesize($wp_ml_thumb_path)];
                        $attachment_id=media_handle_sideload($file_array_for_ml,$product_id,null,['post_parent'=>$product_id]);
                        if(!is_wp_error($attachment_id)){
                            if(!isset($meta_array['preview_id'])||absint($meta_array['preview_id'])!==$attachment_id){
                                $meta_array['preview_id']=$attachment_id;
                                $meta_changed=true;
                            }
                            $media_library_thumb_url=wp_get_attachment_thumb_url($attachment_id);
                        }else{
                            $attachment_id=null;
                        }
                        if(file_exists($wp_ml_thumb_path)) wp_delete_file($wp_ml_thumb_path);
                    }
                }
                if($generated_thumbnail_for_afp_path&&file_exists($generated_thumbnail_for_afp_path)) wp_delete_file($generated_thumbnail_for_afp_path);
                $generated_thumbnail_for_afp_path=null;
            }
        }
        $dimensions=@getimagesize($final_filepath);
        if(false===$dimensions||empty($dimensions[0])||empty($dimensions[1])){$width=0;
            $height=0;
        }else{$width=absint($dimensions[0]);
            $height=absint($dimensions[1]);
            $new_dims=['width'=>$width,'height'=>$height];
            if(!isset($meta_array['dimensions'])||$meta_array['dimensions']!==$new_dims){
                $meta_array['dimensions']=$new_dims;
                $meta_changed=true;
            }
        }
        if(!isset($meta_array['file'])||$meta_array['file']!==$new_base_name){
            $meta_array['file']=sanitize_text_field($new_base_name);
            $meta_changed=true;
        }
        if($meta_changed)update_post_meta($product_id,'_drop_print_meta',$meta_array);
        $original_hash=hash_file($hash_algo,$final_filepath);
        if(false===$original_hash){
            $original_hash='';
        }
        $source_file_for_processing=$final_filepath;
        if(class_exists('ZipArchive')){
            $path_parts_for_zip=pathinfo($final_filepath);
            $compressed_filepath_temp=trailingslashit($path_parts_for_zip['dirname']).$path_parts_for_zip['filename'].'_temp.zip';
            $zip=new ZipArchive();
            if(true===$zip->open($compressed_filepath_temp,ZipArchive::CREATE|ZipArchive::OVERWRITE)){
                if($zip->addFile($final_filepath,$original_name)&&$zip->close()){
                    $compressed_filepath=$compressed_filepath_temp;
                    $source_file_for_processing=$compressed_filepath;
                }else{
                    $zip->close();
                    if(file_exists($compressed_filepath_temp)) wp_delete_file($compressed_filepath_temp);
                }
            }
        }
        $is_encrypted=false;
        $public_key_pem=null;
        $user_public_key_b64 = $user_settings['public_key_pem_b64'] ?? null;
        if(extension_loaded('openssl') && !empty($user_public_key_b64)){
            $decoded_pem_key=base64_decode($user_public_key_b64,true);
            if(false!==$decoded_pem_key){
                $public_key_pem=trim($decoded_pem_key);
                if(!str_starts_with($public_key_pem,'-----BEGIN PUBLIC KEY-----')){
                    $public_key_pem=null;
                }
            }else{
                $public_key_pem=null;
            }
        }
        $afp_filename=$new_base_name.'.afp';
        $afp_filepath=trailingslashit($final_basedir).$afp_filename;
        $afp_file_url=trailingslashit($final_baseurl).$afp_filename;
        $afp_file_handle=@fopen($afp_filepath,'wb');
        if(!$afp_file_handle){
            // translators: %1$s: Filename.
            throw new Exception(sprintf(__('Failed to open final AFP file for writing: %1$s','drop-print'),esc_html(basename($afp_filepath))));
        }
        fwrite($afp_file_handle,pack('C',0));
        if($generated_thumbnail_data&&$generated_thumbnail_mime_type){
            fwrite($afp_file_handle,pack('C',1));
            $mime_bytes=mb_convert_encoding($generated_thumbnail_mime_type,'UTF-8');
            $mime_len=strlen($mime_bytes);
            if($mime_len>255){
                fseek($afp_file_handle,-1,SEEK_CUR);
                fwrite($afp_file_handle,pack('C',0));
            }else{
                fwrite($afp_file_handle,pack('C',$mime_len));
                fwrite($afp_file_handle,$mime_bytes);
                fwrite($afp_file_handle,pack('N',strlen($generated_thumbnail_data)));
                fwrite($afp_file_handle,$generated_thumbnail_data);
            }
        }else{
            fwrite($afp_file_handle,pack('C',0));
        }
        fwrite($afp_file_handle,hex2bin($original_hash?:str_repeat('00',32)));
        $client_id_bytes=mb_convert_encoding(strval($client_connection_id_from_settings),'UTF-8');
        fwrite($afp_file_handle,pack('n',strlen($client_id_bytes)));
        fwrite($afp_file_handle,$client_id_bytes);
        if($public_key_pem){
            $max_attempts=3;
            $encryption_success=false;
            $encryption_error='';
            for($attempt=1;$attempt<=$max_attempts;$attempt++){
                $public_key_resource=null;
                try{
                    $cipher='aes-256-cbc';
                    $public_key_resource=openssl_pkey_get_public($public_key_pem);
                    if(!$public_key_resource)throw new Exception('Failed to load public key resource. '.openssl_error_string());
                    $symmetric_key=random_bytes(32);
                    $iv_length=openssl_cipher_iv_length($cipher);
                    if(false===$iv_length)throw new Exception('Could not get IV length for cipher: '.$cipher);
                    $iv=openssl_random_pseudo_bytes($iv_length);
                    $encrypted_symmetric_key='';
                    if(!openssl_public_encrypt($symmetric_key,$encrypted_symmetric_key,$public_key_resource,OPENSSL_PKCS1_OAEP_PADDING))throw new Exception('Failed to encrypt symmetric key. '.openssl_error_string());
                    openssl_free_key($public_key_resource);
                    $public_key_resource=null;
                    $plaintext=file_get_contents($source_file_for_processing);
                    if(false===$plaintext)throw new Exception('Failed to read source file for encryption.');
                    $encrypted_data=openssl_encrypt($plaintext,$cipher,$symmetric_key,0,$iv);
                    if(false===$encrypted_data)throw new Exception('Failed to encrypt data. '.openssl_error_string());
                    unset($plaintext,$symmetric_key);
                    fwrite($afp_file_handle,$iv);
                    fwrite($afp_file_handle,pack('N',strlen($encrypted_symmetric_key)));
                    fwrite($afp_file_handle,$encrypted_symmetric_key);
                    if(false===fwrite($afp_file_handle,$encrypted_data))throw new Exception('Failed writing encrypted data.');
                    unset($encrypted_data);
                    $is_encrypted=true;
                    $encryption_success=true;
                    break;
                }catch(Exception $e){
                    $encryption_error=$e->getMessage();
                    if($public_key_resource)openssl_free_key($public_key_resource);
                    if($attempt<$max_attempts)usleep(500000);
                }
            }
            if(!$encryption_success){
                fseek($afp_file_handle,0);
                fwrite($afp_file_handle,pack('C',0));
                if($generated_thumbnail_data&&$generated_thumbnail_mime_type){
                    fwrite($afp_file_handle,pack('C',1));
                    $mime_bytes=mb_convert_encoding($generated_thumbnail_mime_type,'UTF-8');
                    $mime_len=strlen($mime_bytes);
                    if($mime_len>255){fwrite($afp_file_handle,pack('C',0));
                    }else{
                        fwrite($afp_file_handle,pack('C',$mime_len));
                        fwrite($afp_file_handle,$mime_bytes);
                        fwrite($afp_file_handle,pack('N',strlen($generated_thumbnail_data)));
                        fwrite($afp_file_handle,$generated_thumbnail_data);
                    }
                }else{
                    fwrite($afp_file_handle,pack('C',0));
                }
                fwrite($afp_file_handle,hex2bin($original_hash?:str_repeat('00',32)));
                fwrite($afp_file_handle,pack('n',strlen($client_id_bytes)));
                fwrite($afp_file_handle,$client_id_bytes);
                $source_content=file_get_contents($source_file_for_processing);
                if(false===$source_content)throw new Exception('Failed read for unencrypted AFP write after encryption failure.');
                fwrite($afp_file_handle,$source_content);
                unset($source_content);
                $is_encrypted=false;
            }
        }else{
            $source_content=file_get_contents($source_file_for_processing);
            if(false===$source_content)throw new Exception('Failed read for unencrypted AFP write.');
            fwrite($afp_file_handle,$source_content);
            unset($source_content);
            $is_encrypted=false;
        }
        if($is_encrypted){
            fseek($afp_file_handle,0);
            fwrite($afp_file_handle,pack('C',1));
        }
        fclose($afp_file_handle);
        $afp_file_handle=null;
        clearstatcache(true,$afp_filepath);
        $min_header_size_no_thumb=1+1+32+2+strlen($client_id_bytes);
        if(!file_exists($afp_filepath)||filesize($afp_filepath)<$min_header_size_no_thumb){
            throw new Exception(__('Verification failed: Final AFP file missing or incomplete.','drop-print'),500);
        }
        if($compressed_filepath&&file_exists($compressed_filepath)){
            if(!wp_delete_file($compressed_filepath));
        }
        $compressed_filepath=null;
        if($final_filepath&&file_exists($final_filepath)){
            if(!wp_delete_file($final_filepath));
        }
        $final_filepath=null;
        $response_data=[
            'file_location'=>$afp_file_url,
            'width'=>$width,
            'height'=>$height,
            'attachment_url'=>$media_library_thumb_url,
            'thumbnail_id'=>$attachment_id,
            'afp_base_filename'=>$new_base_name,
        ];
        if(!empty($new_base_name) && $product_id > 0){
            $tracked_files = get_option('drop_print_files', []);
            if(!is_array($tracked_files)){
                $tracked_files = [];
            }
            if(!isset($tracked_files[$new_base_name])){
                $tracked_files[$new_base_name] = [];
            }
            if(!in_array($product_id, $tracked_files[$new_base_name], true)){
                $tracked_files[$new_base_name][] = $product_id;
                $tracked_files[$new_base_name] = array_values(array_unique($tracked_files[$new_base_name]));
            }
            update_option('drop_print_files', $tracked_files);
        }
        $process_succeeded=true;
        wp_send_json_success($response_data);
    }catch(Exception $e){
        $error_code=is_int($e->getCode())&&$e->getCode()>=400&&$e->getCode()<600?$e->getCode():500;
        if(!empty($generated_thumbnail_for_afp_path)&&file_exists($generated_thumbnail_for_afp_path)) wp_delete_file($generated_thumbnail_for_afp_path);
        wp_send_json_error(['message'=>__('Processing failed: ','drop-print').$e->getMessage()],$error_code);
    }finally{
        if($afp_file_handle)@fclose($afp_file_handle);
        if($public_key_resource)openssl_free_key($public_key_resource);
        if(!empty($temp_chunk_dir))drop_print_cleanup_chunks_upload_handler($temp_chunk_dir);
        if(!$process_succeeded){
            if(!empty($afp_filepath)&&file_exists($afp_filepath)) wp_delete_file($afp_filepath);
            if(!empty($final_filepath)&&file_exists($final_filepath)) wp_delete_file($final_filepath);
            if(!empty($compressed_filepath)&&file_exists($compressed_filepath)) wp_delete_file($compressed_filepath);
            if(!empty($temp_assembled_filepath)&&file_exists($temp_assembled_filepath)) wp_delete_file($temp_assembled_filepath);
        }
    }
}
add_action('wp_ajax_drop_print_process_upload','drop_print_handle_chunked_upload');