<?php
/**
 * Drop Print AJAX Handler for Product Reset & Asset Deletion (v1.1).
 * Handles the 'wp_ajax_drop_print_delete_assets' action.
 * This function stages a product for reset by setting meta flags and clearing
 * staged variation data. It attempts immediate deletion of the Media Library thumbnail.
 * For .afp files, it uses reference counting ('drop_print_files' option) and checks against
 * live orders (via 'drop_print_is_file_in_live_order()') before physical
 * deletion. This version gracefully handles repeated reset attempts by ensuring
 * the reset flag is correctly staged or confirmed before proceeding.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
function drop_print_handle_delete_assets() {
	$deleted_thumb_status = 'skipped';
	$deleted_afp_status   = 'skipped';
	$reset_staged_status  = 'failed';
	try {
		check_ajax_referer( 'drop_print_nonce', 'nonce' );
		if ( ! current_user_can( 'edit_products' ) || ! current_user_can( 'delete_posts' ) || ! current_user_can( 'upload_files' ) ) {
			throw new Exception( __( 'Insufficient permissions.', 'drop-print' ), 403 );
		}
		$product_id        = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		$thumbnail_id      = isset( $_POST['thumbnail_id'] ) ? absint( $_POST['thumbnail_id'] ) : 0;
		$afp_base_filename = isset( $_POST['afp_base_filename'] ) ? sanitize_file_name( wp_unslash( $_POST['afp_base_filename'] ) ) : null;
		if ( ! $product_id ) {
			throw new Exception( __( 'Missing Product ID.', 'drop-print' ), 400 );
		}
		$meta_data = get_post_meta( $product_id, '_drop_print_meta', true );
		if ( ! is_array( $meta_data ) ) {
			$meta_data = array();
		}
		$meta_data['doing_reset'] = true;
		unset( $meta_data['variations'], $meta_data['variations_processed'] );
		$update_result = update_post_meta( $product_id, '_drop_print_meta', $meta_data );
		$current_meta_after_attempt = get_post_meta( $product_id, '_drop_print_meta', true );
		if ( ($update_result !== false) || (is_array($current_meta_after_attempt) && isset($current_meta_after_attempt['doing_reset']) && $current_meta_after_attempt['doing_reset'] === true) ) {
			$reset_staged_status = 'staged';
		} else {
			throw new Exception( __( 'Failed to stage reset flag in product meta. The meta was not updated as expected.', 'drop-print' ), 500 );
		}
		if ( $thumbnail_id > 0 ) {
			if ( get_post_status( $thumbnail_id ) ) {
				$delete_thumb_result = wp_delete_attachment( $thumbnail_id, true );
				if ( false !== $delete_thumb_result ) {
					$deleted_thumb_status = 'deleted';
				} else {
					$deleted_thumb_status = 'failed';
				}
			} else {
				$deleted_thumb_status = 'not_found';
			}
		}
		if ( ! empty( $afp_base_filename ) ) {
			$tracked_files = get_option('drop_print_files', []);
			if (!is_array($tracked_files)) $tracked_files = [];
			$should_delete_physical_file = false;
			$reason_file_kept = '';
			if (isset($tracked_files[$afp_base_filename])) {
				$original_references = $tracked_files[$afp_base_filename];
				$tracked_files[$afp_base_filename] = array_values(array_diff($original_references, [$product_id]));
				if (empty($tracked_files[$afp_base_filename])) {
					if (function_exists('drop_print_is_file_in_live_order') && drop_print_is_file_in_live_order($afp_base_filename)) {
						$should_delete_physical_file = false;
						$reason_file_kept = 'kept_live_order';
					} else {
						$should_delete_physical_file = true;
						unset($tracked_files[$afp_base_filename]);
					}
				} else {
					$should_delete_physical_file = false;
					$reason_file_kept = 'kept_refs_exist';
				}
				update_option('drop_print_files', $tracked_files);
			} else {
                if (function_exists('drop_print_is_file_in_live_order') && drop_print_is_file_in_live_order($afp_base_filename)) {
                    $should_delete_physical_file = false;
                    $reason_file_kept = 'kept_live_order_untracked';
                } else {
				    $should_delete_physical_file = true;
                }
			}
			if ($should_delete_physical_file) {
				$uploads = wp_upload_dir();
				if ( ! empty( $uploads['error'] ) ) {
					$deleted_afp_status = 'failed_dir_error';
				} else {
					$filename       = $afp_base_filename . '.afp';
					$drop_print_dir = trailingslashit( $uploads['basedir'] ) . 'drop-print';
					$full_path      = trailingslashit( $drop_print_dir ) . $filename;
					$real_base_dir  = realpath( $drop_print_dir );
					$real_file_path = realpath( $full_path );
					if ( $real_base_dir && $real_file_path && 0 === strpos( $real_file_path, untrailingslashit( $real_base_dir ) ) ) {
						if ( file_exists( $full_path ) ) {
							if ( wp_delete_file( $full_path ) ) { // Changed from @unlink
								$deleted_afp_status = 'deleted';
							} else {
								$deleted_afp_status = 'failed_unlink';
							}
						} else {
							$deleted_afp_status = 'not_found';
						}
					} else {
						$deleted_afp_status = 'invalid_path';
					}
				}
			} else {
				$deleted_afp_status = $reason_file_kept ?: 'kept_unknown_reason';
			}
		}
		/* translators: %1$s: Status of thumbnail deletion (e.g., "deleted", "failed", "skipped"). %2$s: Status of AFP file deletion (e.g., "deleted", "kept_live_order", "skipped"). */
		$message_format = __( 'Drop Print reset staged. Thumbnail: %1$s. AFP File: %2$s. Please Update/Publish the product to finalize.', 'drop-print' );
		$message = sprintf(
			$message_format,
			esc_html( $deleted_thumb_status ),
			esc_html( $deleted_afp_status )
		);
		wp_send_json_success( array( 'message' => $message ) );
	} catch ( Exception $e ) {
		$error_code    = is_int( $e->getCode() ) && $e->getCode() >= 400 ? $e->getCode() : 500;
		$error_message = $e->getMessage();
		wp_send_json_error(
			array( 'message' => __( 'Reset failed:', 'drop-print' ) . ' ' . $error_message ),
			$error_code
		);
	}
}
add_action( 'wp_ajax_drop_print_delete_assets', 'drop_print_handle_delete_assets' );