<?php
/**
 * Drop Print Client - AJAX Handlers for Settings Page (v1.1)
 * This file previously contained various AJAX handlers. Its primary remaining function
 * is `drop_print_ajax_get_cached_pricelist`, which is used by the product edit page
 * to load pricelist data from local JSON cache files. This version includes improved
 * path validation and error reporting for cases where cached pricelist files
 * might be missing or inaccessible, providing clearer feedback to the client-side.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
if (!function_exists('drop_print_get_client_settings')) {
	if (!function_exists('drop_print_get_client_default_settings_for_ajax_pricelist')) {
		function drop_print_get_client_default_settings_for_ajax_pricelist() {
			return ['billing_currency' => 'GBP', 'pro_status' => 0,];
		}
	}
	if (!function_exists('drop_print_get_client_settings_for_ajax_pricelist')) {
		function drop_print_get_client_settings_for_ajax_pricelist() {
			return wp_parse_args(get_option('drop_print_settings', []), drop_print_get_client_default_settings_for_ajax_pricelist());
		}
	}
	$ajax_use_specific_settings_getter_pricelist = 'drop_print_get_client_settings_for_ajax_pricelist';
} else {
	$ajax_use_specific_settings_getter_pricelist = 'drop_print_get_client_settings';
}
function drop_print_ajax_get_cached_pricelist() {
	global $ajax_use_specific_settings_getter_pricelist;
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['nonce'] ), 'drop_print_nonce' ) ) {
		wp_send_json_error( [ 'message' => __( 'Nonce verification failed.', 'drop-print' ) ], 403 );
		return;
	}
	if ( ! current_user_can( 'edit_products' ) ) {
		wp_send_json_error( [ 'message' => __( 'Permission denied.', 'drop-print' ) ], 403 );
		return;
	}
	if ( ! isset( $_POST['media_id'] ) || empty( $_POST['media_id'] ) ) {
		wp_send_json_error( [ 'message' => __( 'Missing required media ID.', 'drop-print' ) ], 400 );
		return;
	}
	$media_id = sanitize_key( $_POST['media_id'] );
	$cache_dir = DROP_PRINT_CACHE_DIR;
	$filename = "{$media_id}.json";
	$filepath = trailingslashit( $cache_dir ) . $filename;
	if ( ! file_exists( $filepath ) ) {
		$billing_currency_val = 'N/A';
		$settings_for_currency = call_user_func($ajax_use_specific_settings_getter_pricelist);
		if ($settings_for_currency && !empty($settings_for_currency['billing_currency'])) {
			$billing_currency_val = $settings_for_currency['billing_currency'];
		}
        // translators: %1$s: Media ID. %2$s: Billing currency.
		$error_message_format = __( 'Cached pricelist for media "%1$s" (billing: %2$s) not found. Please try refreshing data via the main Drop Print settings page.', 'drop-print' );
		wp_send_json_error( [ 'message' => sprintf( $error_message_format, esc_html( $media_id ), esc_html( $billing_currency_val ) ) ], 404 );
		return;
	}
	$real_cache_dir = realpath( $cache_dir );
	$real_filepath = realpath( $filepath );
	if ( ! $real_cache_dir || ! $real_filepath || 0 !== strpos( $real_filepath, untrailingslashit( $real_cache_dir ) ) ) {
		wp_send_json_error( [ 'message' => __( 'Invalid pricelist cache path or security concern.', 'drop-print' ) ], 500 );
		return;
	}
	if ( ! is_readable( $filepath ) ) {
		wp_send_json_error( [ 'message' => __( 'Cannot read pricelist cache file.', 'drop-print' ) ], 500 );
		return;
	}
	$json_content = file_get_contents( $filepath );
	if ( false === $json_content ) {
		wp_send_json_error( [ 'message' => __( 'Error reading cached pricelist file.', 'drop-print' ) ], 500 );
		return;
	}
	$data = json_decode( $json_content, true );
	if ( null === $data && json_last_error() !== JSON_ERROR_NONE ) {
		wp_send_json_error( [ 'message' => __( 'Error decoding cached pricelist data.', 'drop-print' ) ], 500 );
		return;
	}
	wp_send_json_success( $data );
}
add_action( 'wp_ajax_drop_print_get_cached_pricelist', 'drop_print_ajax_get_cached_pricelist' );