<?php
/**
 * Drop Print Admin Settings Page UI & Handlers (v1.1.2).
 * Renders the admin settings page and handles form submissions. Settings are
 * managed on a per-user basis. Displays connection status for the current
 * user, API details, and configurable options. Handles Pro activation notices
 * and triggers data refresh upon successful Pro activation via URL parameters.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
require_once ABSPATH . 'wp-admin/includes/media.php';
require_once ABSPATH . 'wp-admin/includes/file.php';
require_once ABSPATH . 'wp-admin/includes/image.php';

function drop_print_add_admin_menu_settings_page() {
	add_menu_page(
		__( 'Drop Print Settings', 'drop-print' ),
		__( 'Drop Print', 'drop-print' ),
		'manage_options',
		'drop-print-settings',
		'drop_print_render_settings_page',
		DROP_PRINT_PLUGIN_URL . 'assets/images/icon-drop-print-settings.png',
		99
	);
}
add_action( 'admin_menu', 'drop_print_add_admin_menu_settings_page' );

function drop_print_client_enqueue_admin_scripts_conditional( $hook_suffix ) {
	$client_settings_page_hook = 'toplevel_page_drop-print-settings';
	if ( 'index.php' === $hook_suffix || $client_settings_page_hook === $hook_suffix ) {
		wp_enqueue_script(
			'drop-print-client-settings-js',
			DROP_PRINT_PLUGIN_URL . 'assets/js/admin/drop-print-settings.js',
			array( 'jquery' ),
			defined( 'DROP_PRINT_VERSION' ) ? DROP_PRINT_VERSION : '1.1.2',
			true
		);
	}
}
add_action( 'admin_enqueue_scripts', 'drop_print_client_enqueue_admin_scripts_conditional' );

function drop_print_check_server_status_on_settings_page() {
	if ( ! defined( 'DROP_PRINT_API_URI' ) || empty( DROP_PRINT_API_URI ) ) {
		return array( 'status' => false, 'message' => __( 'API URI Not Defined', 'drop-print' ), 'code' => null );
	}
	$response = wp_remote_get( esc_url_raw( DROP_PRINT_API_URI ), array( 'timeout' => 15 ) );
	if ( is_wp_error( $response ) ) {
		return array( 'status' => false, 'message' => sprintf(__( 'The Drop Print API Server is offline. If this is persistent, contact support quoting: "%s"', 'drop-print' ), $response->get_error_message()), 'code' => $response->get_error_code() );
	}
	$status_code = wp_remote_retrieve_response_code( $response );
	if ( 200 === $status_code ) {
		return array( 'status' => true, 'message' => __( 'Online', 'drop-print' ), 'code' => $status_code );
	}
	else {
		return array( 'status' => false, 'message' => sprintf( __( 'Offline (HTTP %d)', 'drop-print' ), $status_code ), 'code' => $status_code );
	}
}

function drop_print_handle_disconnect_user_on_settings_page() {
	if ( isset( $_POST['drop_print_disconnect_user'] ) && check_admin_referer( 'drop_print_disconnect_user_action' ) ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Permission denied.', 'drop-print' ) );
		}
		$user_id = get_current_user_id();
		if ( $user_id > 0 ) {
			delete_user_meta( $user_id, 'drop_print_user_settings' );
		}
		$cache_dir = DROP_PRINT_CACHE_DIR;
		if ( file_exists( $cache_dir ) && is_dir( $cache_dir ) ) {
			$files = glob( trailingslashit( $cache_dir ) . '*.json' );
			if ( $files !== false ) {
				foreach ( $files as $file ) {
					if ( is_file( $file ) ) {
						wp_delete_file( $file );
					}
				}
			}
		}
		wp_safe_redirect( admin_url( 'admin.php?page=drop-print-settings&disconnected=1' ) );
		exit;
	}
}
add_action( 'admin_init', 'drop_print_handle_disconnect_user_on_settings_page' );

function drop_print_handle_refresh_user_data_on_settings_page() {
	if ( isset( $_POST['drop_print_refresh_user_data'] ) && check_admin_referer( 'drop_print_refresh_user_data_action' ) ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Permission denied.', 'drop-print' ) );
		}
		$user_id = get_current_user_id();
		$result  = array( 'status' => false, 'message' => __( 'User not found or unable to refresh.', 'drop-print' ) );
		if ( $user_id > 0 && function_exists( 'drop_print_client_refresh_data_and_status' ) ) {
			$result = drop_print_client_refresh_data_and_status( $user_id );
		}
		$query_arg = $result['status'] ? 'refreshed' : 'refresh_failed';
		$message   = urlencode( $result['message'] );
		wp_safe_redirect( admin_url( 'admin.php?page=drop-print-settings&' . $query_arg . '=1&message=' . $message . '&refreshed_by_action=true' ) );
		exit;
	}
}
add_action( 'admin_init', 'drop_print_handle_refresh_user_data_on_settings_page' );

function drop_print_handle_save_user_settings_on_settings_page() {
	if ( ! isset( $_POST['save_user_settings_drop_print'] ) || ! check_admin_referer( 'drop_print_save_user_settings_nonce' ) ) {
		return;
	}
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'drop-print' ) );
	}
	$user_id = get_current_user_id();
	if ( $user_id === 0 ) {
		add_settings_error( 'general', 'settings_save_failed', esc_html__( 'Could not save settings: User not identified.', 'drop-print' ), 'error' );
		return;
	}
	$settings                       = drop_print_get_user_settings( $user_id );
	$defaults                       = drop_print_get_default_user_settings();
	$settings['file_handling']      = isset( $_POST['file_handling'] ) ? sanitize_text_field( wp_unslash( $_POST['file_handling'] ) ) : $defaults['file_handling'];
	$price_margin_input             = isset( $_POST['price_margin'] ) ? wp_unslash( $_POST['price_margin'] ) : $defaults['price_margin'];
	$price_margin                   = absint( $price_margin_input );
	$settings['price_margin']       = ( $price_margin < 0 || $price_margin > 1000 ) ? $defaults['price_margin'] : $price_margin;
	$settings['default_media_id']   = isset( $_POST['default_media_id'] ) ? sanitize_key( wp_unslash( $_POST['default_media_id'] ) ) : $defaults['default_media_id'];
	$settings['default_image_resize'] = isset( $_POST['default_image_resize'] ) ? sanitize_key( wp_unslash( $_POST['default_image_resize'] ) ) : $defaults['default_image_resize'];
	$settings['default_shipping_method'] = isset( $_POST['default_shipping_method'] ) ? sanitize_key( wp_unslash( $_POST['default_shipping_method'] ) ) : $defaults['default_shipping_method'];
	$settings['us_sales_tax_collected'] = isset( $_POST['us_sales_tax_collected'] ) && in_array( wp_unslash( $_POST['us_sales_tax_collected'] ), array( 'true', 'false' ), true ) ? sanitize_text_field( wp_unslash( $_POST['us_sales_tax_collected'] ) ) : $defaults['us_sales_tax_collected'];
	$is_pro_user_for_settings       = (bool) ( $settings['pro_status'] ?? 0 );

	if ( $is_pro_user_for_settings ) {
		$settings['auto_production'] = isset( $_POST['auto_production'] ) ? sanitize_text_field( wp_unslash( $_POST['auto_production'] ) ) : $defaults['auto_production'];
	} else {
		$settings['auto_production'] = 'off';
	}
	if ( isset( $_FILES['packing_slip'] ) && isset( $_FILES['packing_slip']['error'] ) && UPLOAD_ERR_OK === $_FILES['packing_slip']['error'] ) {
		$file_info = wp_check_filetype_and_ext( $_FILES['packing_slip']['tmp_name'], $_FILES['packing_slip']['name'] );
		if ( $file_info['type'] !== 'image/png' ) {
			add_settings_error( 'packing_slip_upload', 'invalid_file_type', esc_html__( 'Invalid file type. Please upload a PNG file.', 'drop-print' ), 'error' );
		} else {
			$attachment_id = media_handle_upload( 'packing_slip', 0 );
			if ( is_wp_error( $attachment_id ) ) {
				add_settings_error( 'packing_slip_upload', 'upload_error', sprintf( esc_html__( 'Packing slip upload failed: %s', 'drop-print' ), $attachment_id->get_error_message() ), 'error' );
			} else {
				$old_attachment_id = $settings['packing_slip_attachment_id'] ?? 0;
				if ( $old_attachment_id && $old_attachment_id != $attachment_id ) {
					wp_delete_attachment( $old_attachment_id, true );
				}
				$settings['packing_slip_attachment_id'] = $attachment_id;
				add_settings_error( 'packing_slip_upload', 'upload_success', sprintf( esc_html__( 'Packing slip "%s" uploaded successfully.', 'drop-print' ), esc_html( basename( get_attached_file( $attachment_id ) ) ) ), 'updated' );
			}
		}
	} elseif ( isset( $_FILES['packing_slip'], $_FILES['packing_slip']['error'] ) && UPLOAD_ERR_NO_FILE !== $_FILES['packing_slip']['error'] ) {
		add_settings_error( 'packing_slip_upload', 'upload_error_generic', esc_html__( 'Packing slip upload failed. Please check file size and type.', 'drop-print' ) . ' (Error Code: ' . esc_html( (int) $_FILES['packing_slip']['error'] ) . ')', 'error' );
	}
	update_user_meta( $user_id, 'drop_print_user_settings', $settings );
	add_settings_error( 'general', 'settings_saved', esc_html__( 'Settings saved successfully for current user!', 'drop-print' ), 'updated' );
}
add_action( 'admin_init', 'drop_print_handle_save_user_settings_on_settings_page' );

function drop_print_render_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'drop-print' ) );
	}
	$user_id = get_current_user_id();
	if ( $user_id === 0 ) {
		echo '<div class="notice notice-error"><p>' . esc_html__( 'Cannot display settings: User not identified.', 'drop-print' ) . '</p></div>';
		return;
	}
	$pro_activated_notice          = '';
	$refresh_notice                = '';
	$get_pro_param                 = isset( $_GET['pro'] ) ? sanitize_key( $_GET['pro'] ) : '';
	$get_refreshed_by_action_param = isset( $_GET['refreshed_by_action'] ) ? sanitize_key( $_GET['refreshed_by_action'] ) : '';
	$get_refreshed_param           = isset( $_GET['refreshed'] ) ? sanitize_key( $_GET['refreshed'] ) : '';
	$get_refresh_failed_param      = isset( $_GET['refresh_failed'] ) ? sanitize_key( $_GET['refresh_failed'] ) : '';
	$get_message_param             = isset( $_GET['message'] ) ? sanitize_text_field( wp_unslash( $_GET['message'] ) ) : '';
	if ( $get_pro_param === 'true' && ! $get_refreshed_by_action_param ) {
		if ( function_exists( 'drop_print_client_refresh_data_and_status' ) ) {
			$refresh_result = drop_print_client_refresh_data_and_status( $user_id );
			if ( $refresh_result['status'] ) {
				$pro_activated_notice = '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Congratulations! Your Drop Print Pro account has been successfully activated and your details refreshed.', 'drop-print' ) . '</p></div>';
			} else {
				$pro_activated_notice = '<div class="notice notice-warning is-dismissible"><p>' . esc_html__( 'Pro account activated! However, we could not automatically refresh your details. Please use the refresh button below. Message: ', 'drop-print' ) . esc_html( $refresh_result['message'] ) . '</p></div>';
			}
		} else {
			$pro_activated_notice = '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Congratulations! Your Drop Print Pro account has been successfully activated. Please refresh your data.', 'drop-print' ) . '</p></div>';
		}
	}
	if ( $get_refreshed_by_action_param ) {
		if ( $get_refreshed_param === '1' && $get_message_param ) {
			$refresh_notice = '<div class="notice notice-success is-dismissible"><p>' . esc_html( urldecode( $get_message_param ) ) . '</p></div>';
		} elseif ( $get_refresh_failed_param === '1' && $get_message_param ) {
			$refresh_notice = '<div class="notice notice-error is-dismissible"><p>' . esc_html( urldecode( $get_message_param ) ) . '</p></div>';
		}
	}
	$settings                      = drop_print_get_user_settings( $user_id );
	$defaults                      = drop_print_get_default_user_settings();
	$user_display_name             = $settings['user_display_name'] ?? null;
	$api_url_display               = defined( 'DROP_PRINT_API_URI' ) ? DROP_PRINT_API_URI : __( 'API URI Not Defined', 'drop-print' );
	$cost_currency                 = $settings['billing_currency'] ?? null;
	$server_status_details         = drop_print_check_server_status_on_settings_page();
	$server_available              = $server_status_details['status'];
	$server_offline_message        = $server_status_details['message'];
	$is_pro                        = (bool) ( $settings['pro_status'] ?? 0 );
	$current_packing_slip_attachment_id = $settings['packing_slip_attachment_id'] ?? 0;
	$current_packing_slip_url      = '';
	$current_packing_slip_filename = '';
	if ( $current_packing_slip_attachment_id > 0 && get_post( $current_packing_slip_attachment_id ) ) {
		$current_packing_slip_url      = wp_get_attachment_url( $current_packing_slip_attachment_id );
		$current_packing_slip_filename = basename( get_attached_file( $current_packing_slip_attachment_id ) );
	}
	$woocommerce_dimension_unit = get_option( 'woocommerce_dimension_unit', 'in' );
	$measurement_mapping         = array( 'cm' => 'cm', 'm' => 'cm', 'mm' => 'cm', 'in' => 'in', 'yd' => 'in', 'ft' => 'in' );
	$drop_print_measurement_pref = $measurement_mapping[ $woocommerce_dimension_unit ] ?? 'cm';
	$available_media             = $settings['available_media'] ?? array();
	$default_media_id_saved      = $settings['default_media_id'] ?? $defaults['default_media_id'];
	$final_default_media_id      = $default_media_id_saved;
	$is_saved_id_valid           = false;
	if ( ! empty( $available_media ) ) {
		foreach ( $available_media as $media_item ) {
			if ( isset( $media_item['id'] ) && $media_item['id'] === $default_media_id_saved ) {
				$is_saved_id_valid = true;
				break;
			}
		}
		if ( ! $is_saved_id_valid ) {
			$fallback_options = array( 'hpr-188' );
			if ( isset( $available_media[0]['id'] ) ) {
				$fallback_options[] = $available_media[0]['id'];
			}
			$found_fallback = false;
			foreach ( $fallback_options as $fallback_id ) {
				foreach ( $available_media as $media_item ) {
					if ( isset( $media_item['id'] ) && $media_item['id'] === $fallback_id ) {
						$final_default_media_id = $fallback_id;
						$found_fallback          = true;
						break 2;
					}
				}
			}
			if ( ! $found_fallback && ! empty( $available_media ) && isset( $available_media[0]['id'] ) ) {
				$final_default_media_id = $available_media[0]['id'];
			} elseif ( ! $found_fallback ) {
				$final_default_media_id = $defaults['default_media_id'];
			}
		}
	} else {
		$final_default_media_id = $defaults['default_media_id'];
	}
	$current_default_shipping     = $settings['default_shipping_method'] ?? $defaults['default_shipping_method'];
	$available_shipping_methods   = array(
		'standard'  => __( 'Standard', 'drop-print' ),
		'budget'    => __( 'Budget', 'drop-print' ),
		'express'   => __( 'Express', 'drop-print' ),
		'overnight' => __( 'Overnight', 'drop-print' ),
	);
	$current_default_image_resize = $settings['default_image_resize'] ?? $defaults['default_image_resize'];
	$available_image_resize_options = array(
		'fillPrintArea'      => __( 'Fill Paper Size', 'drop-print' ),
		'fitPrintArea'       => __( 'Fit to Paper Size', 'drop-print' ),
		'stretchToPrintArea' => __( 'Stretch to Fit Paper Size', 'drop-print' ),
	);
	$current_us_sales_tax_collected = $settings['us_sales_tax_collected'] ?? $defaults['us_sales_tax_collected'];
	?>
	<div class="wrap drop-print-settings-wrap">
		<h1><?php esc_html_e( 'Drop Print Settings (Your Account)', 'drop-print' ); ?></h1>
		<?php
		if ( ! empty( $pro_activated_notice ) ) {
			echo wp_kses_post( $pro_activated_notice );
		}
		if ( ! empty( $refresh_notice ) ) {
			echo wp_kses_post( $refresh_notice );
		}
		settings_errors( 'drop_print_pro_status' );
		settings_errors( 'packing_slip_upload' );
		settings_errors( 'general' );
		$get_disconnected_param = isset( $_GET['disconnected'] ) ? sanitize_key( $_GET['disconnected'] ) : '';
		if ( $get_disconnected_param === '1' ) {
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Successfully disconnected your user account from Drop Print.', 'drop-print' ) . '</p></div>';
		}

		// Display server offline notice only if the Drop Print user has a client ID (is connected)
		if ( ! empty( $settings['client_user_connection_id'] ) && ! $server_available ) {
			echo '<div class="notice notice-error is-dismissible"><p>' . esc_html( $server_offline_message ) . '</p></div>';
		}
		?>
		<table class="form-table drop-print-status-table">
			<tbody>
			<?php if ( empty( $settings['client_user_connection_id'] ) ) : ?>
				<?php
				$connect_page_url = admin_url( 'admin.php?page=drop-print-settings' );
				$current_wp_user = wp_get_current_user();
				$client_user_email = ($current_wp_user && $current_wp_user->ID > 0 && !empty($current_wp_user->user_email)) ? $current_wp_user->user_email : '';
				$authenticate_url = DROP_PRINT_API_URI . "/authenticate";
				if (!empty($client_user_email)) {
					$authenticate_url = add_query_arg('existing_client_user_email', urlencode($client_user_email), $authenticate_url);
				}

				if ( function_exists( 'drop_print_display_connect_notice' ) ) {
					// This notice is added via action, not direct echo here.
				} else {
					echo '<tr><td colspan="2"><p>';
					printf( wp_kses_post( __( 'You are not currently connected to Drop Print. Please <a href="%s">Connect your account</a> to manage settings.', 'drop-print' ) ), esc_url( $connect_page_url ) );
					echo '</p></td></tr>';
				}
				?>
			<?php else : // This is where the connected user's details and the "Your Settings" form should be. ?>
				<tr>
					<th scope="row"><?php esc_html_e( 'Server URI', 'drop-print' ); ?></th>
					<td>
						<?php echo esc_url( $api_url_display ); ?>
						<?php if ( $server_available ) : ?>
							<span class="drop-print-status is-online"> - <?php esc_html_e( 'Online', 'drop-print' ); ?></span>
						<?php else : ?>
							<span class="drop-print-status is-offline"> - <?php esc_html_e( 'Offline', 'drop-print' ); ?></span>
						<?php endif; ?>
						<?php
						// Display login link for connected users
						if ( $user_id > 0 && !empty($settings['client_user_connection_id']) ) {
							$current_wp_user = wp_get_current_user();
							$login_email = !empty($current_wp_user->user_email) ? $current_wp_user->user_email : '';
							$login_url = DROP_PRINT_API_URI . "/authenticate";
							if (!empty($login_email)) {
								$login_url = add_query_arg('existing_client_user_email', urlencode($login_email), $login_url);
							}
							echo '<p class="description"><a href="' . esc_url($login_url) . '" target="_blank" rel="noopener">' . esc_html__('Login to your account at Drop Print', 'drop-print') . '</a></p>';
						}
						?>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Your Username', 'drop-print' ); ?></th>
					<td><strong><?php echo esc_html( $user_display_name ?: __( 'N/A', 'drop-print' ) ); ?></strong></td>
				</tr>
				<?php if ( $cost_currency ) : ?>
					<tr>
						<th scope="row"><?php esc_html_e( 'Your Billing Currency', 'drop-print' ); ?></th>
						<td>
							<strong><?php echo esc_html( $cost_currency ); ?></strong>
							<p class="description"><?php esc_html_e( 'This is your fixed billing currency set during account connection.', 'drop-print' ); ?></p>
						</td>
					</tr>
				<?php endif; ?>
				<tr>
					<th scope="row"><?php esc_html_e( 'Your Account Status', 'drop-print' ); ?></th>
					<td>
						<strong><?php echo $is_pro ? esc_html__( 'Pro User', 'drop-print' ) : esc_html__( 'Standard User', 'drop-print' ); ?></strong>
						<?php if ( ! $is_pro ) : ?>
							<a href="https://api.drop-print.com/pro" target="_blank" style="margin-left:10px;"><?php esc_html_e( 'Upgrade to Pro', 'drop-print' ); ?></a>
						<?php endif; ?>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Your Connection', 'drop-print' ); ?></th>
					<td style="display: flex; gap: 10px; align-items: center;">
						<form method="POST" action="" style="margin:0;">
							<?php wp_nonce_field( 'drop_print_disconnect_user_action' ); ?>
							<button type="submit" name="drop_print_disconnect_user" class="button button-secondary drop-print-disconnect-button"><?php esc_html_e( 'Disconnect My Account', 'drop-print' ); ?></button>
						</form>
						<form method="POST" action="" style="margin:0;">
							<?php wp_nonce_field( 'drop_print_refresh_user_data_action' ); ?>
							<button type="submit" name="drop_print_refresh_user_data" class="button button-secondary" <?php disabled( ! $server_available ); ?>><?php esc_html_e( 'Reload Account Info', 'drop-print' ); ?></button>
						</form>
					</td>
				</tr>
			<?php endif; // End of check for empty client_user_connection_id ?>
			</tbody>
		</table>
		<?php if ( ! empty( $settings['client_user_connection_id'] ) ) : // ADDED THIS CONDITIONAL WRAPPER ?>
			<h3><?php esc_html_e( 'Your Settings', 'drop-print' ); ?></h3>
			<form method="POST" action="" enctype="multipart/form-data">
				<?php wp_nonce_field( 'drop_print_save_user_settings_nonce' ); ?>
				<input type="hidden" name="action" value="drop_print_save_user_settings_on_settings_page_action" />
				<table class="form-table">
					<tbody>
					<tr>
						<th scope="row"><label for="packing_slip_upload_trigger"><?php esc_html_e( 'Your Packing Slip', 'drop-print' ); ?></label></th>
						<td>
							<div id="drop-print-current-packing-slip-container" style="<?php echo $current_packing_slip_filename ? '' : 'display:none;'; ?>">
								<?php if ( $current_packing_slip_filename && $current_packing_slip_url ) : ?>
									<p class="description">
										<a href="<?php echo esc_url( $current_packing_slip_url ); ?>" target="_blank" rel="noopener"><?php echo esc_html( $current_packing_slip_filename ); ?></a>
										<span class="drop-print-replace-packing-slip" style="cursor:pointer; margin-left: 10px;" title="<?php esc_attr_e( 'Replace Packing Slip', 'drop-print' ); ?>"><?php echo esc_html( '?' ); ?></span>
									</p>
								<?php else : ?>
									<p class="description"><?php esc_html_e( 'No packing slip uploaded, or unable to retrieve details.', 'drop-print' ); ?></p>
								<?php endif; ?>
							</div>
							<div id="drop-print-upload-packing-slip-container" style="<?php echo $current_packing_slip_filename ? 'display:none;' : ''; ?>">
								<input type="file" id="packing_slip_upload_trigger" name="packing_slip" accept="image/png">
								<p class="description"><?php esc_html_e( 'Upload an A4 PNG file (72 DPI recommended).', 'drop-print' ); ?></p>
							</div>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="us_sales_tax_collected"><?php esc_html_e( 'US Sales Tax Collected?', 'drop-print' ); ?></label></th>
						<td>
							<fieldset>
								<legend class="screen-reader-text"><?php esc_html_e( 'US Sales Tax Collected', 'drop-print' ); ?></legend>
								<label><input type="radio" name="us_sales_tax_collected" value="true" <?php checked( $current_us_sales_tax_collected, 'true' ); ?>> <?php esc_html_e( 'Yes', 'drop-print' ); ?></label><br />
								<label><input type="radio" name="us_sales_tax_collected" value="false" <?php checked( $current_us_sales_tax_collected, 'false' ); ?>> <?php esc_html_e( 'No', 'drop-print' ); ?></label>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Indicate if US sales tax is collected and remitted for orders. This information may be passed to the print server.', 'drop-print' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop-print-measurement-unit-preference"><?php esc_html_e( 'Measurements', 'drop-print' ); ?></label></th>
						<td>
							<fieldset>
								<legend class="screen-reader-text"><?php esc_html_e( 'Measurement Unit Preference', 'drop-print' ); ?></legend>
								<label><input type="radio" name="drop_print_measurement_pref_display" value="cm" <?php checked( $drop_print_measurement_pref, 'cm' ); ?> disabled> <?php esc_html_e( 'Centimetres (cm)', 'drop-print' ); ?></label><br />
								<label><input type="radio" name="drop_print_measurement_pref_display" value="in" <?php checked( $drop_print_measurement_pref, 'in' ); ?> disabled> <?php esc_html_e( 'Inches (in)', 'drop-print' ); ?></label>
							</fieldset>
							<?php
							$wc_settings_product_url   = esc_url( admin_url( 'admin.php?page=wc-settings&tab=products&section' ) );
							$wc_settings_link_text     = esc_html__( 'WooCommerce Settings', 'drop-print' );
							$wc_settings_link          = sprintf( '<a href="%s" target="_blank" rel="noopener">%s</a>', $wc_settings_product_url, $wc_settings_link_text );
							$description_text_format = __( 'Unit preference derived from the <i>Dimensions unit</i> setting in %s (this is a site-wide WooCommerce setting).', 'drop-print' );
							$description_with_link   = sprintf( $description_text_format, $wc_settings_link );
							echo '<p class="description">' . wp_kses_post( $description_with_link ) . '</p>';
							?>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop-print-file-handling"><?php esc_html_e( 'Your File Handling', 'drop-print' ); ?></label></th>
						<td>
							<select name="file_handling" id="drop-print-file-handling">
								<option value="upload" <?php selected( $settings['file_handling'] ?? $defaults['file_handling'], 'upload' ); ?>><?php esc_html_e( 'Upload Wizard (Recommended)', 'drop-print' ); ?></option>
								<option value="manual" <?php selected( $settings['file_handling'] ?? $defaults['file_handling'], 'manual' ); ?> disabled><?php esc_html_e( 'Manual URI (Coming Soon!)', 'drop-print' ); ?></option>
							</select>
							<p class="description"><?php esc_html_e( 'Determines how print files are associated with products.', 'drop-print' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop-print-price-margin"><?php esc_html_e( 'Your Default Price Margin (%)', 'drop-print' ); ?></label></th>
						<td>
							<input type="number" id="drop-print-price-margin" name="price_margin" value="<?php echo esc_attr( $settings['price_margin'] ?? $defaults['price_margin'] ); ?>" min="0" max="1000" step="1" class="small-text">
							<p class="description"><?php esc_html_e( 'Default percentage margin added to the base cost for calculating selling prices.', 'drop-print' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop-print-default-image-resize"><?php esc_html_e( 'Your Default Image Resize', 'drop-print' ); ?></label></th>
						<td>
							<select name="default_image_resize" id="drop-print-default-image-resize">
								<?php foreach ( $available_image_resize_options as $option_key => $option_name ) : ?>
									<option value="<?php echo esc_attr( $option_key ); ?>" <?php selected( $current_default_image_resize, $option_key ); ?>>
										<?php echo esc_html( $option_name ); ?>
									</option>
								<?php endforeach; ?>
							</select>
							<p class="description"><?php esc_html_e( 'Determines how uploaded images are scaled to fit the selected print area by default.', 'drop-print' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop-print-default-media-id"><?php esc_html_e( 'Your Default Media Selection', 'drop-print' ); ?></label></th>
						<td>
							<select name="default_media_id" id="drop-print-default-media-id" <?php if ( empty( $available_media ) ) echo 'disabled'; ?>>
								<?php if ( ! empty( $available_media ) ) : ?>
									<?php foreach ( $available_media as $media_item ) : ?>
										<?php if ( isset( $media_item['id'], $media_item['name'] ) ) : ?>
											<option value="<?php echo esc_attr( $media_item['id'] ); ?>" <?php selected( $final_default_media_id, $media_item['id'] ); ?>>
												<?php echo esc_html( $media_item['name'] ); ?>
											</option>
										<?php endif; ?>
									<?php endforeach; ?>
								<?php else : ?>
									<option value="<?php echo esc_attr( $defaults['default_media_id'] ); ?>" selected disabled><?php esc_html_e( 'Click "Reload Account Info" to load media types for your account.', 'drop-print' ); ?></option>
								<?php endif; ?>
							</select>
							<p class="description">
								<?php
								if ( empty( $available_media ) ) {
									esc_html_e( 'Refresh your data to populate media types. ', 'drop-print' );
								}
								if ( ! $is_pro ) {
									printf( wp_kses_post( __( 'Expect additions to this list in the future. <a href="%s" target="_blank" rel="noopener">Learn more</a>.', 'drop-print' ) ), 'https://drop-print.com/products' );
								} else {
									esc_html_e( 'Select your preferred default media type for new products.', 'drop-print' );
								}
								?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop-print-default-shipping-method"><?php esc_html_e( 'Your Default Shipping Method', 'drop-print' ); ?></label></th>
						<td>
							<select name="default_shipping_method" id="drop-print-default-shipping-method" <?php if ( ! $is_pro ) echo 'disabled'; ?>>
								<?php foreach ( $available_shipping_methods as $method_key => $method_name ) : ?>
									<option value="<?php echo esc_attr( $method_key ); ?>" <?php selected( $current_default_shipping, $method_key ); ?>>
										<?php echo esc_html( $method_name ); ?>
									</option>
								<?php endforeach; ?>
							</select>
							<p class="description">
								<?php
								if ( ! $is_pro ) {
									printf( wp_kses_post( __( 'Selecting a default shipping method is a Pro feature for your account. Standard shipping will be used. <a href="%s" target="_blank" rel="noopener">Upgrade to Pro</a> to enable this option.', 'drop-print' ) ), 'https://api.drop-print.com/pro' );
								} else {
									esc_html_e( 'Select your preferred default shipping method for new products. You can also change this per order during "Quote & Pay".', 'drop-print' );
								}
								?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop_print_payment_frequency_display"><?php esc_html_e( 'Payment Frequency (Account Level)', 'drop-print' ); ?></label></th>
						<td>
							<input type="text" id="drop_print_payment_frequency_display" name="payment_frequency_display" value="<?php echo esc_attr( $settings['payment_frequency'] === 'pay_as_you_go' ? __( 'Pay as You Go', 'drop-print' ) : $settings['payment_frequency'] ); ?>" disabled>
							<p class="description">
								<?php esc_html_e( 'All orders are processed on a Pay as You Go basis. This is set at your Drop Print account level.', 'drop-print' ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="drop_print_auto_production"><?php esc_html_e( 'Auto Production (Pro Only)', 'drop-print' ); ?></label></th>
						<td>
							<select name="auto_production" id="drop_print_auto_production" <?php if ( ! $is_pro ) echo 'disabled'; ?>>
								<option value="off" <?php selected( $settings['auto_production'] ?? $defaults['auto_production'], 'off' ); ?>><?php esc_html_e( 'Off', 'drop-print' ); ?></option>
								<option value="on" <?php selected( $settings['auto_production'] ?? $defaults['auto_production'], 'on' ); ?> <?php if ( ! $is_pro ) echo 'disabled'; ?>><?php esc_html_e( 'On', 'drop-print' ); ?></option>
							</select>
							<p class="description">
								<?php
								if ( ! $is_pro ) {
									printf( wp_kses_post( __( 'Automatically send your orders to print. This is a <a href="%1$s" target="_blank" rel="noopener">Pro account</a> feature and requires a valid payment card on file at <a href="%2$s" target="_blank" rel="noopener">Drop Print</a>.', 'drop-print' ) ), 'https://api.drop-print.com/pro', 'https://api.drop-print.com/my-account/payment-methods/' );
								} else {
									printf( wp_kses_post( __( 'Automatically send orders to print upon customer payment. Requires a valid payment card on file at <a href="%s" target="_blank" rel="noopener">Drop Print</a>.', 'drop-print' ) ), 'https://api.drop-print.com/my-account/payment-methods/' );
								}
								?>
							</p>
						</td>
					</tr>
					</tbody>
				</table>
				<?php submit_button( __( 'Save My Settings', 'drop-print' ), 'primary', 'save_user_settings_drop_print' ); ?>
			</form>
		<?php endif; // END of conditional wrapper for Your Settings and Form ?>
	</div>
	<?php
}