/**
 * Drop Print Client - Quote & Pay Modal UI (v1.1)
 *
 * Handles the user interface rendering and interactions for the "Quote & Pay"
 * modal. This includes populating the modal with quote data and displaying price
 * breakdowns, including the surcharge which now has a dedicated class for styling.
 * It manages the Stripe payment form, using the latest Elements API for robust
 * card input and validation, and provides user feedback during payment.
 * Relies on DropPrintModalCore for AJAX calls and state management.
 */
window.DropPrintModalUI = (function($) {
    function escHtml(unsafe) {
        if (typeof unsafe !== 'string') return '';
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
    function getL10n(textKey, fallbackText, replacements = null) {
        let l10nText = typeof drop_print_vars !== 'undefined' && drop_print_vars.text && drop_print_vars.text[textKey] ? drop_print_vars.text[textKey] : fallbackText;
        if (window.wp && window.wp.i18n && window.wp.i18n.__) {
            l10nText = window.wp.i18n.__(l10nText, 'drop-print');
        }
        if (replacements && typeof replacements === 'object') {
            for (const key in replacements) {
                l10nText = l10nText.replace(new RegExp('%' + key + '%', 'g'), replacements[key]);
            }
        }
        return l10nText;
    }
    function formatL10nCurrency(amount, currencyCode) {
        const number = parseFloat(amount);
        if (isNaN(number)) return String(amount);
        try {
            return new Intl.NumberFormat(typeof drop_print_vars !== 'undefined' ? drop_print_vars.locale || navigator.language || 'en-US' : 'en-US', {
                style: 'currency',
                currency: currencyCode || 'USD'
            }).format(number);
        } catch (e) {
            return `${number.toFixed(2)} ${currencyCode || ''}`;
        }
    }
    function getJsFlagEmoji(countryCode) {
        if (typeof countryCode !== 'string' || countryCode.length !== 2) return '';
        const code = countryCode.toUpperCase();
        const flags = {
            'AU': "🇦🇺",
            'GB': "🇬🇧",
            'US': "🇺🇸",
            'NL': "🇳🇱",
            'ES': "🇪🇸",
            'LV': "🇱🇻",
            'CA': "🇨🇦",
            'SE': "🇸🇪",
            'DE': "🇩🇪"
        };
        return flags[code] || code;
    }
    function updateModalButtonState(state, $button, $spinner, messageKey = null, fallbackMessage = null) {
        if (!$button || !$button.length || !$spinner || !$spinner.length) return;
        const originalText = $button.data('original-text') || getL10n('text_quote_pay', 'Quote & Pay');
        const text = messageKey && fallbackMessage ? getL10n(messageKey, fallbackMessage) : originalText;
        switch (state) {
            case 'loading':
                $button.prop('disabled', true).text(text);
                $spinner.css('visibility', 'visible');
                $button.removeClass('button-primary').addClass('button-secondary');
                break;
            case 'reset':
            default:
                $button.prop('disabled', false).text(originalText);
                $spinner.css('visibility', 'hidden');
                $button.removeClass('button-secondary').addClass('button-primary');
                break;
        }
    }
    function mountStripeCardElement(shouldMount, allowPaymentStatus) {
        const stripeInstance = DropPrintModalCore.getStripeInstance();
        let currentCardElement = DropPrintModalCore.getStripeCardElement();
        let currentStripeElements = DropPrintModalCore.getStripeElements();
        const $paymentModal = DropPrintModalCore.getPaymentModalElement();
        const $errorContainer = $paymentModal ? ($paymentModal.find('#dp-card-errors').length ? $paymentModal.find('#dp-card-errors') : $paymentModal.find('#dp-payment-form-container')) : $('body');
        if (currentCardElement) {
            try {
                currentCardElement.unmount();
            } catch (e) {}
            try {
                currentCardElement.destroy();
            } catch (e) {}
            DropPrintModalCore.setStripeCardElement(null);
        }
        const cardElementDomNode = document.getElementById('dp-card-element');
        if (shouldMount && cardElementDomNode) {
            if (!stripeInstance) {
                $errorContainer.html(`<p class="dp-error" style="color:red;font-weight:bold;">${getL10n('text_stripe_not_loaded_card_unavailable','Payment gateway not configured or failed to load. Card form unavailable.')}</p>`).show();
                if ($paymentModal) $paymentModal.find('#dp-payment-form-container button[type="submit"]').hide();
                return;
            }
            try {
                if (!currentStripeElements || typeof currentStripeElements.create !== 'function') {
                    currentStripeElements = stripeInstance.elements();
                    DropPrintModalCore.setStripeElements(currentStripeElements);
                }
                const styleOptions = {
                    base: {
                        fontSize: '16px',
                        color: '#32325d',
                        fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif',
                        '::placeholder': {
                            color: '#aab7c4'
                        }
                    },
                    invalid: {
                        color: '#fa755a',
                        iconColor: '#fa755a'
                    }
                };
                const newCardElement = currentStripeElements.create('card', {
                    style: styleOptions,
                    hidePostalCode: true
                });
                DropPrintModalCore.setStripeCardElement(newCardElement);
                newCardElement.mount('#dp-card-element');
                newCardElement.on('change', event => {
                    const displayErrorElement = document.getElementById('dp-card-errors');
                    if (displayErrorElement) {
                        displayErrorElement.textContent = event.error ? event.error.message : '';
                    }
                    if ($paymentModal) {
                        const isButtonDisabled = !(event.complete && allowPaymentStatus);
                        $paymentModal.find('#dp-submit-payment').prop('disabled', isButtonDisabled);
                    }
                });
                $(cardElementDomNode).show();
                if ($paymentModal) {
                    $paymentModal.find('#dp-submit-payment').prop('disabled', true);
                }
            } catch (e) {
                alert(getL10n('text_error_prefix', 'Error:') + ' ' + getL10n('text_stripe_form_init_fail', 'Failed to initialize payment form.'));
                DropPrintModalCore.closePaymentModal();
            }
        } else if (!shouldMount && cardElementDomNode) {
            $(cardElementDomNode).hide();
        }
    }
    function populateModalWithPriceData(ajaxResponseData, isInitialPopulation) {
        const $paymentModal = DropPrintModalCore.getPaymentModalElement();
        if (!$paymentModal) return;
        let currentIsUsingNewCardForm = DropPrintModalCore.getIsUsingNewCardForm();
        const $quoteBreakdown = $paymentModal.find('#dp-quote-breakdown');
        const $paymentFormContainer = $paymentModal.find('#dp-payment-form-container');
        const $proUpsellArea = $paymentModal.find('#dp-pro-upsell-area');
        const $shippingSelectorContainer = $paymentModal.find('#dp-shipping-method-selector-container');
        $paymentFormContainer.empty().hide();
        const $existingPayButton = $paymentModal.find('#dp-submit-payment');
        if ($existingPayButton.length) $existingPayButton.remove();
        if ($quoteBreakdown.find('.dp-error').length || $quoteBreakdown.find('.dp-modal-sub-loader').length || isInitialPopulation) {
            $quoteBreakdown.empty();
        }
        const quoteDetails = ajaxResponseData.quote_details || ajaxResponseData;
        if (isInitialPopulation) {
            const newCardFormStateBasedOnInitialData = !(ajaxResponseData.saved_card_details_from_server && ajaxResponseData.saved_card_details_from_server.last4);
            DropPrintModalCore.setIsUsingNewCardForm(newCardFormStateBasedOnInitialData);
            currentIsUsingNewCardForm = newCardFormStateBasedOnInitialData;
        }
        const upsellDetailsFromServer = ajaxResponseData.quote_details?.upsell_details;
        const potentialSavingsAmount = parseFloat(upsellDetailsFromServer?.potential_savings_amount || 0);
        const potentialSavingsCurrency = upsellDetailsFromServer?.potential_savings_currency || quoteDetails?.currency || 'USD';
        const proPricingPercentage = parseFloat(upsellDetailsFromServer?.pro_pricing_percentage_cheaper || 0);
        const proSignupUrl = drop_print_vars.pro_signup_url || 'https://api.drop-print.com/pro/';
        const savingsAmountFormatted = formatL10nCurrency(potentialSavingsAmount, potentialSavingsCurrency);
        const addPaymentMethodUrl = drop_print_vars.add_payment_method_url || 'https://api.drop-print.com/my-account/add-payment-method/';
        const manageCardsUrl = drop_print_vars.manage_cards_url || 'https://api.drop-print.com/my-account/payment-methods/';
        $proUpsellArea.empty().hide();
        if (!drop_print_vars.is_pro_user && potentialSavingsAmount > 0 && proPricingPercentage > 0) {
            const upsellHeadline = getL10n('text_pro_upsell_headline_v3', '%savings_percentage%% OFF TODAY', {
                savings_percentage: proPricingPercentage.toFixed(0)
            });
            const upsellBody = getL10n('text_pro_upsell_body_v3', 'Save %savings_amount% on this order by upgrading to Pro now! Enjoy order discounts, select from numerous delivery methods and use a saved payment card <a href="%manage_cards_url%" target="_blank" rel="noopener" title="Manage payment methods (opens new tab)"><span class="dashicons dashicons-external"></span></a> to automate your workflow to send orders directly to print. Learn more at: <a href="%pro_url%" target="_blank" rel="noopener" title="Learn about Pro (opens new tab)">drop-print.com/pro <span class="dashicons dashicons-external"></span></a>.', {
                savings_amount: savingsAmountFormatted,
                manage_cards_url: manageCardsUrl,
                pro_url: proSignupUrl
            });
            $proUpsellArea.html(`<h4>${upsellHeadline}</h4><p>${upsellBody}</p>`).show();
        }
        let shippingSelectorHtml = '';
        if (drop_print_vars.is_pro_user) {
            shippingSelectorHtml = `<label for="dp-shipping-method-select" class="dp-shipping-label">${getL10n('text_shipping_method_label','Shipping Method:')}</label><select id="dp-shipping-method-select" name="dp_shipping_method_select" class="dp-shipping-method-select">`;
            for (const method_key in drop_print_vars.available_shipping_methods) {
                shippingSelectorHtml += `<option value="${method_key}" ${method_key===DropPrintModalCore.getSelectedShippingMethod()?'selected':''}>${drop_print_vars.available_shipping_methods[method_key]}</option>`;
            }
            shippingSelectorHtml += `</select>`;
        } else {
            const currentMethodName = drop_print_vars.available_shipping_methods[DropPrintModalCore.getSelectedShippingMethod()] || DropPrintModalCore.getSelectedShippingMethod();
            shippingSelectorHtml = `<label for="dp-shipping-method-display" class="dp-shipping-label">${getL10n('text_shipping_method_label','Shipping Method:')}</label><input type="text" id="dp-shipping-method-display" value="${currentMethodName}" disabled class="dp-shipping-method-display">`;
            if (!drop_print_vars.is_pro_user && (potentialSavingsAmount > 0 || (quoteDetails.shipping_unavailable && isInitialPopulation))) {
                shippingSelectorHtml += `<p class="description dp-pro-upsell-shipping"><small>${getL10n('text_pro_shipping_upsell','Pro users can select from Budget to Overnight shipping.')}</small></p>`;
            }
        }
        $shippingSelectorContainer.html(shippingSelectorHtml).show();
        if (drop_print_vars.is_pro_user) {
            $paymentModal.find('#dp-shipping-method-select').off('change').on('change', function() {
                DropPrintModalCore.fetchCalculatedTotal(DropPrintModalCore.getCurrentQuoteId(), $(this).val());
            });
        }
        let breakdownHtml = `<h4>${getL10n('text_production_quote_title','Production Quote:')}</h4><ul>`;
        let allowPayment = true;
        if (quoteDetails.shipping_unavailable === true) {
            allowPayment = false;
            const unavailableDetails = quoteDetails.shipping_unavailable_details || {};
            const failedMethodDisplay = drop_print_vars.available_shipping_methods[unavailableDetails.method_used] || unavailableDetails.method_used || DropPrintModalCore.getSelectedShippingMethod();
            const countryDisplay = unavailableDetails.destination_country_code || getL10n('text_selected_country', 'the selected country');
            let userErrorMessage = unavailableDetails.message || getL10n('text_shipping_unavailable_default_msg_sprintf', 'Shipping method "%s1" is not available for %s2.', {
                s1: failedMethodDisplay,
                s2: countryDisplay
            });
            $quoteBreakdown.html(`<p class="dp-error" style="color:red;font-weight:bold;">${userErrorMessage}</p>`);
            if (drop_print_vars.is_pro_user) {
                $quoteBreakdown.append(`<p style="font-size:0.9em;margin-top:10px;">${getL10n('text_shipping_unavailable_pro_prompt','Please try selecting an alternative shipping method from the dropdown above.')}</p>`);
                $shippingSelectorContainer.show();
            } else {
                $quoteBreakdown.append(`<p style="font-size:0.9em;margin-top:10px;">${getL10n('text_contact_support_shipping_unavailable','As this shipping method is unavailable, please contact support to proceed with this order.')}</p>`);
                $shippingSelectorContainer.hide();
            }
        } else {
            let itemsTotalForDisplay = 0;
            if (quoteDetails && quoteDetails.items && Array.isArray(quoteDetails.items)) {
                quoteDetails.items.forEach(item => {
                    const itemCost = parseFloat(item.cost);
                    if (!isNaN(itemCost)) itemsTotalForDisplay += (itemCost * (parseInt(item.quantity, 10) || 1));
                    let itemBaseTitle = item.item_base_title || item.sku;
                    let itemAttributeString = item.item_attribute_string || '';
                    let nameAndAttributesHTML = `<span class="dp-item-base-title">${escHtml(itemBaseTitle)}</span>`;
                    if (itemAttributeString) {
                        nameAndAttributesHTML += `<br><small class="dp-item-attributes">${escHtml(itemAttributeString)}</small>`;
                    }
                    let quantityHTML = item.quantity && parseInt(item.quantity, 10) > 1 ? ` <span class="dp-item-quantity">(x${parseInt(item.quantity,10)})</span>` : '';
                    let productionCountryHtml = '';
                    if (quoteDetails.production_country_code) {
                        productionCountryHtml = `<small class="dp-production-country">${getL10n('text_printed_in','Printed in ')}${getJsFlagEmoji(quoteDetails.production_country_code)}</small>`;
                    }
                    breakdownHtml += `<li><div class="dp-item-details"><div class="dp-item-name-wrapper">${nameAndAttributesHTML}${quantityHTML}</div>${productionCountryHtml}</div><span class="dp-item-cost">${formatL10nCurrency(itemCost,quoteDetails.currency)}</span></li>`;
                });
            }
            const totalItemsCostFromServer = (quoteDetails?.total_items_cost !== undefined) ? parseFloat(quoteDetails.total_items_cost) : itemsTotalForDisplay;
            breakdownHtml += `<li><span>${getL10n('text_print_services_total','Printing Total')}</span> <span>${formatL10nCurrency(totalItemsCostFromServer,quoteDetails?.currency)}</span></li>`;
            const shippingInfo = quoteDetails?.shipping;
            const shippingCost = shippingInfo?.cost || 0;
            const shippingName = drop_print_vars.available_shipping_methods[DropPrintModalCore.getSelectedShippingMethod()] || DropPrintModalCore.getSelectedShippingMethod();
            let shippingLineInnerHtml = `<span>${getL10n('text_delivery','Delivery')} (${shippingName})</span> <span>${formatL10nCurrency(shippingCost,quoteDetails?.currency)}</span>`;
            breakdownHtml += `<li class="dp-delivery-line">${shippingLineInnerHtml}</li>`;
            if (quoteDetails.surcharge_details && quoteDetails.surcharge_details.amount > 0) {
                const surcharge = quoteDetails.surcharge_details;
                const baseLabel = getL10n('text_surcharge_label', 'Processing Fee');
                const detailsText = `(${surcharge.percentage_applied}% + ${formatL10nCurrency(surcharge.fixed_fee_applied, quoteDetails.currency)})`;
                const surchargeLabelHtml = `${baseLabel} <span class="dp-surcharge-details">${detailsText}</span>`;
                breakdownHtml += `<li class="dp-surcharge-line"><span>${surchargeLabelHtml}</span><span>${formatL10nCurrency(surcharge.amount, quoteDetails.currency)}</span></li>`;
            }
            const totalDue = quoteDetails?.total || 0;
            breakdownHtml += `</ul><div class="quote-total"><span>${getL10n('text_total_due','Total Due')}</span> <span>${formatL10nCurrency(totalDue,quoteDetails?.currency)}</span></div>`;
            if (ajaxResponseData.is_pro_user_on_server && drop_print_vars.is_pro_user) {
                breakdownHtml += `<p class="dp-pro-pricing-notice"><small>${getL10n('text_pro_pricing_applied','Pro discount applied.')}</small></p>`;
            }
            $quoteBreakdown.html(breakdownHtml);
        }
        let paymentFormHtml = '';
        let payButtonText = getL10n('text_pay_now', 'Pay Now');
        if (!currentIsUsingNewCardForm && ajaxResponseData.saved_card_details_from_server && ajaxResponseData.saved_card_details_from_server.last4) {
            const scd = ajaxResponseData.saved_card_details_from_server;
            payButtonText = getL10n('text_pay_with_saved_card_sprintf', 'Pay with %brand% •••• %last4%', {
                brand: scd.brand || 'Card',
                last4: scd.last4
            });
            paymentFormHtml = `<form id="dp-payment-form" class="drop-print-payment-form"><div class="dp-saved-card-info"><p><a href="${manageCardsUrl}" target="_blank" rel="noopener noreferrer" class="dp-manage-cards-link">${getL10n('text_manage_payment_methods_link','Manage payment methods on Drop Print')}</a></p><p><a href="#" id="dp-use-new-card-link">${getL10n('text_use_different_card_link','Use a different card?')}</a></p></div><div id="dp-card-element-container" style="display:none;"><div class="form-row"><label for="dp-card-element">${getL10n('text_card_details_label','Card Details')}</label><div id="dp-card-element" class="drop-print-card-element"></div><div id="dp-card-errors" class="drop-print-card-errors" role="alert"></div></div></div><button id="dp-submit-payment" class="drop-print-pay-button button-primary" type="submit">${payButtonText}</button></form>`;
            $paymentFormContainer.html(paymentFormHtml).show();
            mountStripeCardElement(false, allowPayment);
            if (!allowPayment) {
                $paymentModal.find('#dp-submit-payment').hide().prop('disabled', true);
            }
        } else {
            let cardPromptHtml = '';
            if (!drop_print_vars.is_pro_user) {
                cardPromptHtml = `<p class="dp-save-card-prompt"><small>${getL10n('text_add_card_for_future_sprintf_v2','You can <a href="%add_card_url%" target="_blank" rel="noopener" class="dp-add-card-link-close-modal">add this card to your account</a> for faster checkout next time (modal will close).',{add_card_url:addPaymentMethodUrl})}</small></p>`;
            } else {
                const autoProdSetting = drop_print_vars.client_auto_production_setting || 'off';
                if (autoProdSetting === 'on') {
                    cardPromptHtml = `<p class="dp-save-card-prompt dp-warning-text">${getL10n('text_pro_add_card_for_auto_prod_on_v2','Warning: Auto-Production is ON, but orders will not proceed as no payment card is on file. <a href="%manage_cards_url%" target="_blank" rel="noopener">Please add one</a>. Alternatively, enter your card details below to pay for this order now.',{manage_cards_url:manageCardsUrl})}</p>`;
                } else {
                    cardPromptHtml = `<p class="dp-save-card-prompt"><small>${getL10n('text_pro_add_card_for_auto_prod_off_v2','Pro Feature: Auto-Production is OFF. <a href="%manage_cards_url%" target="_blank" rel="noopener">Add a payment method</a> to enable Auto-Production later.',{manage_cards_url:manageCardsUrl})}</small></p>`;
                }
            }
            paymentFormHtml = `<form id="dp-payment-form" class="drop-print-payment-form"><div id="dp-card-element-container"><div class="form-row"><label for="dp-card-element">${getL10n('text_card_details_label','Card Details')}</label><div id="dp-card-element" class="drop-print-card-element"></div><div id="dp-card-errors" class="drop-print-card-errors" role="alert"></div></div></div>${cardPromptHtml}<button id="dp-submit-payment" class="drop-print-pay-button button-primary" type="submit">${payButtonText}</button></form>`;
            $paymentFormContainer.html(paymentFormHtml).show();
            mountStripeCardElement(true, allowPayment);
            if (!allowPayment) {
                $paymentModal.find('#dp-submit-payment').hide().prop('disabled', true);
            }
        }
        $paymentModal.find('#dp-use-new-card-link').off('click').on('click', function(e) {
            e.preventDefault();
            if (!DropPrintModalCore.getIsUsingNewCardForm()) {
                DropPrintModalCore.setIsUsingNewCardForm(true);
                populateModalWithPriceData(DropPrintModalCore.getCurrentQuoteDataFromServer(), false);
            }
        });
        $paymentModal.find('.dp-add-card-link-close-modal').off('click.dpClose').on('click.dpClose', function(e) {
            DropPrintModalCore.closePaymentModal();
            const $triggerBtn = DropPrintModalCore.getCurrentModalTriggerButton();
            const $triggerSpinner = DropPrintModalCore.getCurrentModalSpinner();
            if ($triggerBtn && $triggerSpinner && typeof DropPrintModalUI !== 'undefined' && DropPrintModalUI.updateModalButtonState) {
                DropPrintModalUI.updateModalButtonState('reset', $triggerBtn, $triggerSpinner);
            }
        });
        $paymentModal.find('#dp-payment-form').off('submit').on('submit', function(e) {
            handleFinalPaymentSubmit(e);
        });
        if (quoteDetails.shipping_unavailable === true && drop_print_vars.is_pro_user && $quoteBreakdown.find('.dp-shipping-notice').length === 0) {
            $quoteBreakdown.append(`<p class="dp-shipping-notice" style="font-size:0.9em;color:#cc0000;">${getL10n('text_pro_select_valid_shipping_to_pay','Please select a valid shipping method to enable payment.')}</p>`);
        }
    }
    function handleFinalPaymentSubmit(e) {
        e.preventDefault();
        const $paymentModal = DropPrintModalCore.getPaymentModalElement();
        const currentQuoteId = DropPrintModalCore.getCurrentQuoteId();
        if (!$paymentModal || !currentQuoteId) return;
        const $form = $(e.currentTarget);
        const $submitButton = $form.find('#dp-submit-payment');
        const $processingDiv = $paymentModal.find('#dp-payment-processing');
        const $errorDiv = $paymentModal.find('#dp-card-errors');
        $errorDiv.text('');
        $submitButton.prop('disabled', true).addClass('is-loading');
        $processingDiv.show();
        const currentQuoteData = DropPrintModalCore.getCurrentQuoteDataFromServer();
        const isUsingNewCard = DropPrintModalCore.getIsUsingNewCardForm();
        $.ajax({
            url: drop_print_vars.ajax_url,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'drop_print_create_client_payment_intent',
                quote_id: currentQuoteId,
                shipping_method_preference: DropPrintModalCore.getSelectedShippingMethod(),
                use_saved_card: !isUsingNewCard,
                saved_card_id: (!isUsingNewCard && currentQuoteData && currentQuoteData.saved_card_details_from_server) ? currentQuoteData.saved_card_details_from_server.id : null,
                nonce: drop_print_vars.create_payment_intent_nonce
            },
            success: function(response) {
                if (response.success && response.data && response.data.stripe_client_secret) {
                    DropPrintModalCore.setStripeClientSecret(response.data.stripe_client_secret);
                    confirmStripePaymentOnClient($submitButton, $processingDiv, $errorDiv);
                } else {
                    const errorMsg = (response.data && response.data.message) ? getL10n(response.data.message, response.data.message) : getL10n('text_payment_session_init_fail', 'Failed to initialize payment session.');
                    $errorDiv.text(getL10n('text_error_prefix', 'Error:') + ' ' + errorMsg);
                    $submitButton.prop('disabled', false).removeClass('is-loading');
                    $processingDiv.hide();
                }
            },
            error: function(jqXHR) {
                const errorData = jqXHR.responseJSON && jqXHR.responseJSON.data ? jqXHR.responseJSON.data : {};
                const errorMsg = errorData.message || getL10n('text_payment_session_init_fail', 'Failed to initialize payment session.');
                $errorDiv.text(getL10n('text_error_prefix', 'Error:') + ' ' + errorMsg);
                $submitButton.prop('disabled', false).removeClass('is-loading');
                $processingDiv.hide();
            }
        });
    }
    function confirmStripePaymentOnClient($submitButton, $processingDiv, $errorDiv) {
        const stripeInstance = DropPrintModalCore.getStripeInstance();
        const currentStripeClientSecret = DropPrintModalCore.getStripeClientSecret();
        const $paymentModal = DropPrintModalCore.getPaymentModalElement();
        let isUsingNewCardForm = DropPrintModalCore.getIsUsingNewCardForm();
        let stripeCardElement = DropPrintModalCore.getStripeCardElement();
        if (!$submitButton && $paymentModal) $submitButton = $paymentModal.find('#dp-submit-payment');
        if (!stripeInstance || !currentStripeClientSecret) {
            $errorDiv.text(getL10n('text_stripe_not_ready_for_confirm', 'Payment gateway not ready or session invalid.'));
            if ($submitButton) $submitButton.prop('disabled', false).removeClass('is-loading');
            if ($processingDiv) $processingDiv.hide();
            return;
        }
        let confirmPromise;
        if (isUsingNewCardForm) {
            if (!stripeCardElement) {
                $errorDiv.text(getL10n('text_card_details_not_ready_for_confirm', 'Card details are not ready.'));
                if ($submitButton) $submitButton.prop('disabled', false).removeClass('is-loading');
                if ($processingDiv) $processingDiv.hide();
                return;
            }
            const paymentMethodPayload = {
                payment_method: {
                    card: stripeCardElement
                }
            };
            confirmPromise = stripeInstance.confirmCardPayment(currentStripeClientSecret, paymentMethodPayload);
        } else {
            confirmPromise = stripeInstance.confirmCardPayment(currentStripeClientSecret);
        }
        confirmPromise.then(function(result) {
            if ($processingDiv) $processingDiv.hide();
            if (result.error) {
                $errorDiv.text(result.error.message);
                if ($submitButton) $submitButton.prop('disabled', false).removeClass('is-loading');
                if (!isUsingNewCardForm && (result.error.code === 'card_declined' || result.error.type === 'card_error')) {
                    DropPrintModalCore.setIsUsingNewCardForm(true);
                    if ($paymentModal) {
                        const $modalContent = $paymentModal.find('#dp-modal-main-content');
                        const $loader = $paymentModal.find('.drop-print-modal-loader');
                        if ($modalContent) $modalContent.hide();
                        if ($loader) $loader.show().find('p').text(getL10n('text_saved_card_failed_loading_new', 'Saved card failed. Loading new card form...'));
                        DropPrintModalCore.fetchCalculatedTotal(DropPrintModalCore.getCurrentQuoteId(), DropPrintModalCore.getSelectedShippingMethod(), true);
                    }
                }
            } else if (result.paymentIntent && result.paymentIntent.status === 'succeeded') {
                const $triggerButton = DropPrintModalCore.getCurrentModalTriggerButton();
                if ($triggerButton && $triggerButton.length) {
                    const $statusContainer = $triggerButton.closest('.drop-print-status-container');
                    if ($statusContainer.length) {
                        const processingText = getL10n('text_status_processing_payment_button', 'Processing Payment');
                        $statusContainer.html(`<span class="order-status status-on-hold"><span>${processingText}</span></span>`);
                    }
                }
                handlePaymentSuccessFeedback();
                DropPrintModalCore.updateLocalOrderStatusPostPayment(result.paymentIntent.id, 'awaiting_webhook_confirmation');
            } else if (result.paymentIntent) {
                $errorDiv.text(getL10n('text_payment_status_label', 'Payment status:') + ' ' + result.paymentIntent.status);
                if ($submitButton) $submitButton.prop('disabled', false).removeClass('is-loading');
            } else {
                $errorDiv.text(getL10n('text_payment_confirm_failed_unknown', 'Payment confirmation failed. Please try again.'));
                if ($submitButton) $submitButton.prop('disabled', false).removeClass('is-loading');
            }
        }).catch(function(error) {
            if ($processingDiv) $processingDiv.hide();
            $errorDiv.text(getL10n('text_unexpected_error_payment_confirm_sprintf', 'An unexpected error occurred: %s', {
                s: error.message
            }));
            if ($submitButton) $submitButton.prop('disabled', false).removeClass('is-loading');
        });
    }
    function handlePaymentSuccessFeedback() {
        const $paymentModal = DropPrintModalCore.getPaymentModalElement();
        if ($paymentModal) {
            $paymentModal.find('#dp-modal-main-content').hide();
            $paymentModal.find('#dp-payment-success-msg').html(`<h4>${getL10n('text_payment_successful_title','Payment Successful!')}</h4><p>${getL10n('text_order_queued_status_update_soon','Order is being queued. You will see a status update shortly.')}</p>`).show();
            setTimeout(() => {
                DropPrintModalCore.closePaymentModal();
            }, 3500);
        }
        const $triggerBtn = DropPrintModalCore.getCurrentModalTriggerButton();
        const $triggerSpinner = DropPrintModalCore.getCurrentModalSpinner();
        if ($triggerBtn && $triggerSpinner && typeof DropPrintModalUI !== 'undefined' && DropPrintModalUI.updateModalButtonState) {
            DropPrintModalUI.updateModalButtonState('reset', $triggerBtn, $triggerSpinner);
        }
    }
    return {
        getL10n: getL10n,
        formatL10nCurrency: formatL10nCurrency,
        getJsFlagEmoji: getJsFlagEmoji,
        updateModalButtonState: updateModalButtonState,
        populateModalWithPriceData: populateModalWithPriceData,
        handleFinalPaymentSubmit: handleFinalPaymentSubmit,
        confirmStripePaymentOnClient: confirmStripePaymentOnClient,
        handlePaymentSuccessFeedback: handlePaymentSuccessFeedback
    };
})(jQuery);