/**
 * Drop Print Client - Quote & Pay Modal Core Logic (v1.1)
 *
 * Manages the underlying state, AJAX requests, and core logic for the "Quote &
 * Pay" modal. It handles fetching initial quote details (now with enhanced error
 * display for prerequisite failures), calculating totals by communicating with
 * client-side AJAX handlers, updating local order status post-payment, and
 * managing Stripe.js objects. Relies on 'drop_print_vars' localized from PHP.
 */
jQuery(document).ready(function($){
    let currentClientOrderId=null;
    let currentQuoteId=null;
    let $currentModalTriggerButton=null;
    let $currentModalSpinner=null;
    let $paymentModal=null;
    let stripe=null;
    let stripeElements=null;
    let stripeCardElement=null;
    let currentStripeClientSecret=null;
    let currentSelectedShippingMethod=typeof drop_print_vars!=='undefined'?(drop_print_vars.default_shipping_method||'standard'):'standard';
    let isUsingNewCardForm=true;
    let serverStripePublishableKey=null;
    let currentQuoteDataFromServer=null;
    function localGetL10n(textKey,fallbackText,replacements=null){
        let l10nText=fallbackText;
        if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.getL10n){
            l10nText=DropPrintModalUI.getL10n(textKey,fallbackText,replacements);
        }else if(typeof drop_print_vars!=='undefined'&&drop_print_vars[textKey]){
            l10nText=drop_print_vars[textKey];
            if(replacements&&typeof replacements==='object'){
                for(const key in replacements){
                    l10nText=l10nText.replace(new RegExp('%'+key+'%','g'),replacements[key]);
                }
            }
        }
        return l10nText;
    }
    function openPaymentModalInitialLoading(){
        closePaymentModal();
        currentSelectedShippingMethod=typeof drop_print_vars!=='undefined'?(drop_print_vars.default_shipping_method||'standard'):'standard';
        const modalHtml=`
        <div class="drop-print-payment-modal-overlay" id="dp-payment-modal">
        <div class="drop-print-payment-modal-content">
        <span class="drop-print-payment-modal-close" id="dp-modal-close" title="${localGetL10n('text_close_modal','Close')}">&times;</span>
        <div class="drop-print-modal-loader">
        <span class="spinner is-active"></span>
        <p>${localGetL10n('text_initiating_quote_session','Initiating quote session...')}</p>
        </div>
        <div id="dp-modal-main-content" style="display:none;">
        <div class="drop-print-modal-header">${localGetL10n('text_payment_for_order','Payment for Order',{s:currentClientOrderId})}</div>
        <div id="dp-pro-upsell-area" style="display:none; margin-bottom:15px; padding:10px; background-color:#f0f6fc; border:1px solid #c9e0f5; border-radius:4px;"></div>
        <div id="dp-shipping-method-selector-container" style="margin-bottom:15px;"></div>
        <div class="drop-print-quote-breakdown" id="dp-quote-breakdown"></div>
        <div id="dp-payment-form-container"></div>
        <div id="dp-payment-processing" style="display:none;text-align:center;margin-top:10px;">
        <span class="spinner is-active" style="float:none;vertical-align:middle;"></span> ${localGetL10n('text_processing_payment','Processing Payment...')}
        </div>
        </div>
        <div id="dp-payment-success-msg" style="display:none;text-align:center;padding:30px;">
        <h4>${localGetL10n('text_payment_successful_title','Payment Successful!')}</h4>
        <p>${localGetL10n('text_awaiting_production_confirmation','Awaiting production confirmation from the server.')}</p>
        </div>
        </div>
        </div>`;
        $('body').append(modalHtml);
        $paymentModal=$('#dp-payment-modal');
        $paymentModal.on('click','#dp-modal-close',()=>{
            closePaymentModal();
            if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState&&$currentModalTriggerButton&&$currentModalSpinner){
                DropPrintModalUI.updateModalButtonState('reset',$currentModalTriggerButton,$currentModalSpinner);
            }
        });
    }
    function closePaymentModal(){
        if($paymentModal){
            let cardElementToClear=DropPrintModalCore.getStripeCardElement();
            if(cardElementToClear&&typeof cardElementToClear.destroy==='function'){
                try{cardElementToClear.destroy();}catch(e){}
            }
            if(cardElementToClear&&typeof cardElementToClear.unmount==='function'){
                 try{cardElementToClear.unmount();}catch(e){}
            }
            $paymentModal.remove();
            $paymentModal=null;
        }
        stripe=null;
        stripeElements=null;
        stripeCardElement=null;
        DropPrintModalCore.setStripeInstance(null);
        DropPrintModalCore.setStripeElements(null);
        DropPrintModalCore.setStripeCardElement(null);
        currentStripeClientSecret=null;
        currentQuoteId=null;
        isUsingNewCardForm=true;
        serverStripePublishableKey=null;
        currentQuoteDataFromServer=null;
    }
    function fetchInitialQuoteIdAndDetails(){
        if(!currentClientOrderId)return;
        if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState){
            DropPrintModalUI.updateModalButtonState('loading',$currentModalTriggerButton,$currentModalSpinner,'text_initiating_quote_session','Initiating quote session...');
        }
        openPaymentModalInitialLoading();
        $.ajax({
            url:drop_print_vars.ajax_url,
            type:'POST',
            dataType:'json',
            data:{
                action:'drop_print_initiate_quote_session',
                client_order_id:currentClientOrderId,
                nonce:drop_print_vars.quote_initiate_nonce
            },
            success:function(response){
                if(response.success&&response.data&&response.data.quote_id){
                    currentQuoteId=response.data.quote_id;
                    serverStripePublishableKey=response.data.stripe_publishable_key||drop_print_vars.stripe_publishable_key;
                    if(serverStripePublishableKey&&typeof Stripe==='function'&&!stripe){
                        try{stripe=Stripe(serverStripePublishableKey);}catch(e){
                            alert(localGetL10n('text_error_prefix','Error:')+' '+localGetL10n('text_stripe_init_fail','Payment gateway failed to initialize. Please close and try again.'));
                            closePaymentModal();
                            if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState&&$currentModalTriggerButton&&$currentModalSpinner){
                                DropPrintModalUI.updateModalButtonState('reset',$currentModalTriggerButton,$currentModalSpinner);
                            }
                            return;
                        }
                    }
                    DropPrintModalCore.setStripeInstance(stripe);
                    fetchCalculatedTotal(currentQuoteId,currentSelectedShippingMethod,true);
                }else{
                    const errorResponseData = response || {};
                    const errorData = errorResponseData.data || {};
                    let displayMessage = errorData.message || localGetL10n('text_generic_error','An unknown error occurred.');
                    if(errorData.details && Array.isArray(errorData.details) && errorData.details.length > 0){
                        let detailsString = "\n\n" + localGetL10n('text_specific_errors_found', 'Specific issues found:') + "\n- " + errorData.details.join("\n- ");
                        displayMessage += detailsString;
                    }
                    alert(localGetL10n('text_error_prefix','Error:')+' '+displayMessage);
                    closePaymentModal();
                     if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState&&$currentModalTriggerButton&&$currentModalSpinner){
                        DropPrintModalUI.updateModalButtonState('reset',$currentModalTriggerButton,$currentModalSpinner);
                    }
                }
            },
            error:function(jqXHR){
                const errorResponseData = jqXHR.responseJSON || {};
                const errorData = errorResponseData.data || {};
                let displayMessage = errorData.message || localGetL10n('text_generic_error','An unknown error occurred.');
                if(errorData.details && Array.isArray(errorData.details) && errorData.details.length > 0){
                    let detailsString = "\n\n" + localGetL10n('text_specific_errors_found', 'Specific issues found:') + "\n- " + errorData.details.join("\n- ");
                    displayMessage += detailsString;
                }
                alert(localGetL10n('text_error_prefix','Error:')+' '+displayMessage);
                closePaymentModal();
                 if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState&&$currentModalTriggerButton&&$currentModalSpinner){
                    DropPrintModalUI.updateModalButtonState('reset',$currentModalTriggerButton,$currentModalSpinner);
                }
            }
        });
    }
    function fetchCalculatedTotal(quoteId,shippingMethod,isInitialPopulation=false){
        if((!$paymentModal&&!isInitialPopulation)||!quoteId)return;
        const $loader=$paymentModal.find('.drop-print-modal-loader');
        const $mainContent=$paymentModal.find('#dp-modal-main-content');
        const $quoteBreakdown=$paymentModal.find('#dp-quote-breakdown');
        const $paymentFormContainer=$paymentModal.find('#dp-payment-form-container');
        currentSelectedShippingMethod=shippingMethod;
        if(isInitialPopulation){
            $mainContent.hide();
            $loader.show().find('p').text(localGetL10n('text_requesting_quote','Requesting Quote...'));
        }else{
            $quoteBreakdown.html(`<p class="dp-modal-sub-loader">${localGetL10n('text_updating_quote','Updating quote...')}</p>`);
            $paymentFormContainer.empty().hide();
            const $payButton=$paymentModal.find('#dp-submit-payment');
            if($payButton.length)$payButton.remove();
        }
        $.ajax({
            url:drop_print_vars.ajax_url,
            type:'POST',
            dataType:'json',
            data:{
                action:'drop_print_calculate_client_total',
                quote_id:quoteId,
                shipping_method_preference:shippingMethod,
                nonce:drop_print_vars.calculate_total_nonce
            },
            success:function(response){
                if(response.success&&response.data){
                    currentQuoteDataFromServer=response.data;
                    if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.populateModalWithPriceData){
                        DropPrintModalUI.populateModalWithPriceData(currentQuoteDataFromServer,isInitialPopulation);
                    }
                }else{
                    const errorMsg=(response.data&&response.data.message)?response.data.message:localGetL10n('text_generic_error','An unknown error occurred.');
                    if(isInitialPopulation){
                        alert(localGetL10n('text_error_prefix','Error:')+' '+errorMsg);
                        closePaymentModal();
                        if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState&&$currentModalTriggerButton&&$currentModalSpinner){
                           DropPrintModalUI.updateModalButtonState('reset',$currentModalTriggerButton,$currentModalSpinner);
                        }
                    }else{
                        $quoteBreakdown.html(`<p class="dp-error" style="color:red;font-weight:bold;">${localGetL10n('text_error_prefix','Error:')+' '+errorMsg}</p>`);
                        const $payButton=$paymentModal.find('#dp-submit-payment');
                        if($payButton.length)$payButton.remove();
                    }
                }
                if(isInitialPopulation){
                    $loader.hide();
                    $mainContent.show();
                }
            },
            error:function(jqXHR){
                const errorData=jqXHR.responseJSON&&jqXHR.responseJSON.data?jqXHR.responseJSON.data:{};
                const errorMsg=errorData.message||localGetL10n('text_generic_error','An unknown error occurred.');
                if(isInitialPopulation){
                    alert(localGetL10n('text_error_prefix','Error:')+' '+errorMsg);
                    closePaymentModal();
                    if(typeof DropPrintModalUI!=='undefined'&&DropPrintModalUI.updateModalButtonState&&$currentModalTriggerButton&&$currentModalSpinner){
                        DropPrintModalUI.updateModalButtonState('reset',$currentModalTriggerButton,$currentModalSpinner);
                    }
                }else{
                    $quoteBreakdown.html(`<p class="dp-error" style="color:red;font-weight:bold;">${localGetL10n('text_error_prefix','Error:')+' '+errorMsg}</p>`);
                    const $payButton=$paymentModal.find('#dp-submit-payment');
                    if($payButton.length)$payButton.remove();
                }
                if(isInitialPopulation){
                    $loader.hide();
                    $mainContent.show();
                }
            }
        });
    }
    function updateLocalOrderStatusPostPayment(paymentIntentId,statusKey){
        $.ajax({
            url:drop_print_vars.ajax_url,
            type:'POST',
            dataType:'json',
            data:{
                action:'drop_print_update_client_order_post_payment',
                client_order_id:currentClientOrderId,
                quote_id:currentQuoteId,
                payment_intent_id:paymentIntentId,
                status:statusKey,
                nonce:drop_print_vars.update_order_post_payment_nonce
            }
        });
    }
    $(document).on('click','.drop-print-send-button',function(e){
        e.preventDefault();
        if($paymentModal)return;
        $currentModalTriggerButton=$(this);
        $currentModalSpinner=$currentModalTriggerButton.next('.drop-print-spinner');
        currentClientOrderId=$currentModalTriggerButton.data('order-id');
        if(!currentClientOrderId){
            alert(localGetL10n('text_error_missing_order_id','Error: Order ID is missing.'));
            return;
        }
        if($currentModalTriggerButton.prop('disabled'))return;
        fetchInitialQuoteIdAndDetails();
    });
    window.DropPrintModalCore={
        closePaymentModal:closePaymentModal,
        fetchCalculatedTotal:fetchCalculatedTotal,
        updateLocalOrderStatusPostPayment:updateLocalOrderStatusPostPayment,
        getIsUsingNewCardForm:function(){return isUsingNewCardForm;},
        setIsUsingNewCardForm:function(val){isUsingNewCardForm=val;},
        getCurrentQuoteDataFromServer:function(){return currentQuoteDataFromServer;},
        getCurrentQuoteId:function(){return currentQuoteId;},
        getStripeInstance:function(){return stripe;},
        setStripeInstance:function(stripeInstance){stripe=stripeInstance;},
        setStripeCardElement:function(el){stripeCardElement=el;},
        getStripeCardElement:function(){return stripeCardElement;},
        setStripeElements:function(els){stripeElements=els;},
        getStripeElements:function(){return stripeElements;},
        setStripeClientSecret:function(secret){currentStripeClientSecret=secret;},
        getStripeClientSecret:function(){return currentStripeClientSecret;},
        getSelectedShippingMethod:function(){return currentSelectedShippingMethod;},
        getPaymentModalElement:function(){return $paymentModal;},
        getCurrentClientOrderId:function(){return currentClientOrderId;},
        getCurrentModalTriggerButton:function(){return $currentModalTriggerButton;},
        getCurrentModalSpinner:function(){return $currentModalSpinner;}
    };
});