/**
 * Manages fetching, validation, and rendering of product lists from JSON pricelists (v1.1)
 * for the Drop Print panel. It populates the product selection UI. When pricelist loading
 * fails, an error overlay is shown. The "Refresh Pricelist" button in this overlay
 * now directs the user (by opening a new tab) to the main Drop Print settings page,
 * where they can use the dedicated refresh functionality. The overlay message is
 * updated to reflect this new user flow, removing direct AJAX refresh from this screen.
 */
(function($) {
    'use strict';
    window.dropPrint = window.dropPrint || {};
    function dropPrintIsJsonDataValid(data) {
        if (typeof data === 'object' && data !== null &&
            Array.isArray(data.products) &&
            typeof data.media_id === 'string' && data.media_id.length > 0 &&
            typeof data.media_name === 'string' && data.media_name.length > 0 &&
            typeof data.currency_code === 'string' && data.currency_code.length > 0
        ) {
            return true;
        }
        return false;
    }
    function dropPrintHandleJsonLoadError(reason = 'Could not load product data.') {
        const { dropPrintUpdateOverlay, state } = window.dropPrint || {};
        if (typeof dropPrintUpdateOverlay !== 'function') {
            return;
        }
        const message = `<p>${reason}</p><p>The pricelist might be missing or corrupt. Please go to the <strong>Drop Print Settings page</strong> to use the "Refresh Subscription & Pricelists" button, then return here and try selecting the media type again.</p>`;
        dropPrintUpdateOverlay('is-error', 'Pricelist Error', message, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok', 'docs', 'refresh'] });
        if (state) state.printSizesData = null;
        window.dropPrint.printSizesData = null;
    }
    function dropPrintGetCurrencySymbol(currencyCode) {
        const symbols = { 'USD': '$', 'GBP': '£', 'EUR': '€', 'AUD': 'A$', 'CAD': 'C$', 'ZAR': 'R' };
        if (symbols[currencyCode]) return symbols[currencyCode];
        if (typeof get_woocommerce_currency_symbol === 'function' && currencyCode === window.dropPrint?.config?.wcCurrency) {
            return window.dropPrint.config.wcCurrencySymbol || currencyCode;
        }
        return currencyCode;
    }
    function dropPrintRenderProductList() {
        const { elements, printSizesData, config, state } = window.dropPrint || {};
        if (!elements?.productListContainer?.length) {
            return;
        }
        if (!printSizesData || !dropPrintIsJsonDataValid(printSizesData)) {
            dropPrintHandleJsonLoadError('RenderProductList called with invalid or missing printSizesData.');
            elements.productListContainer.html('<p class="drop-print-error">No products to display. Data is invalid or missing. Try selecting another media type or refresh data in settings.</p>');
            return;
        }
        if (!config || typeof config.priceMargin === 'undefined' || !config.wcCurrencySymbol || !config.wcMeasurementUnit) {
            elements.productListContainer.html('<p class="drop-print-error">Error: Critical configuration missing for product rendering.</p>');
            return;
        }
        const productListContainer = elements.productListContainer;
        const wcMeasurement = config.wcMeasurementUnit;
        const storePriceCurrencySymbol = config.wcCurrencySymbol;
        const pricelistCurrencyCode = printSizesData.currency_code.toUpperCase();
        const pricelistCurrencySymbol = dropPrintGetCurrencySymbol(pricelistCurrencyCode);
        let productListHtml = '';
        const flagMap = { GB: '🇬🇧', US: '🇺🇸', NL: '🇳🇱', AU: '🇦🇺', ES: '🇪🇸', CA: '🇨🇦', DE: '🇩🇪' };
        const productsToRender = printSizesData.products;
        if (productsToRender.length > 0) {
            productsToRender.forEach((item) => {
                if (!item || typeof item !== 'object' || !item.sku || !item.print_pixels ||
                    !item.dimensions_in || !item.dimensions_cm || typeof item.price === 'undefined' ||
                    typeof item.shipping_price_lowest === 'undefined') {
                    return;
                }
                const itemCostRaw = item.price;
                const itemShippingCostRaw = item.shipping_price_lowest;
                let displayItemCost = 'N/A';
                let displayShippingCostText = '';
                let sellPriceValueAttr = 'value=""';
                let sellPriceDataAttr = 'data-price=""';
                let isPriceEditable = false;
                let itemCost = null;
                const parsedItemCost = parseFloat(itemCostRaw);
                if (itemCostRaw !== undefined && itemCostRaw !== null && !isNaN(parsedItemCost) && parsedItemCost >= 0) {
                    itemCost = parsedItemCost;
                    displayItemCost = `${pricelistCurrencySymbol}${itemCost.toFixed(2)}`;
                    sellPriceValueAttr = `value=""`;
                    sellPriceDataAttr = `data-price=""`;
                    isPriceEditable = true;
                } else {
                    itemCost = null;
                }
                const parsedShippingCost = parseFloat(itemShippingCostRaw);
                if (itemShippingCostRaw !== undefined && !isNaN(parsedShippingCost) && parsedShippingCost > 0) {
                    const shippingCost = parsedShippingCost;
                    displayShippingCostText = `<span class="drop-print-shipping-estimate"> + delivery <span class="drop-print-shipping-estimate-price">from ${pricelistCurrencySymbol}${shippingCost.toFixed(2)}<span class="drop-print-shipping-info-icon" title="View delivery cost information"> ℹ️</span></span></span>`;
                } else if (parsedShippingCost === 0) {
                    displayShippingCostText = '<span class="drop-print-shipping-info-icon" title="View delivery cost information"> (i)</span>';
                } else {
                    displayShippingCostText = '<span class="drop-print-shipping-info-icon" title="View delivery cost information"> (i)</span>';
                }
                let baseDimension = '';
                let unitSuffix = '';
                let aSeriesTitlePart = '';
                if (item.a_series && typeof item.a_series === 'string' && item.a_series.trim() !== '') {
                    aSeriesTitlePart = `${item.a_series.trim()} `;
                }
                if (wcMeasurement === 'cm' && item.dimensions_cm) {
                    baseDimension = item.dimensions_cm;
                    unitSuffix = ' cm';
                } else if (wcMeasurement === 'in' && item.dimensions_in) {
                    baseDimension = item.dimensions_in;
                    unitSuffix = ' Inches';
                } else {
                    if (item.dimensions_cm) {
                        baseDimension = item.dimensions_cm;
                        unitSuffix = ' cm';
                    } else if (item.dimensions_in) {
                        baseDimension = item.dimensions_in;
                        unitSuffix = ' Inches';
                    } else {
                        baseDimension = 'Unknown Dimensions';
                    }
                }
                const mediaName = printSizesData.media_name;
                const chosenDimensionString = baseDimension !== 'Unknown Dimensions' ? baseDimension + unitSuffix : 'Unknown Dimensions';
                const displayProductTitle = `${aSeriesTitlePart}${chosenDimensionString}`;
                const fullTitleForDataAttr = `${displayProductTitle} - ${mediaName}`;
                let flagsHtml = '';
                if (Array.isArray(item.source_countries) && item.source_countries.length > 0) {
                    flagsHtml = item.source_countries.map(code => flagMap[code] ? `<span title="${code}">${flagMap[code]}</span>` : '').join(' ');
                } else {
                    flagsHtml = '<span title="Origin Unknown">?</span>';
                }
                const shippingTitle = flagsHtml && item.source_countries.length > 0 ? 'Ships from: ' + item.source_countries.join(', ') : 'Shipping Origin Unknown';
                productListHtml += `
<div id="drop-print-product-${item.sku}" class="drop-print-product" data-title="${fullTitleForDataAttr}" data-pixels="${item.print_pixels}" data-sku="${item.sku}" data-cm="${item.dimensions_cm}" data-in="${item.dimensions_in}" data-cost="${itemCost ?? ''}" data-currency="${pricelistCurrencyCode || ''}">
    <div class="drop-print-product-info">
        <div class="drop-print-product-title">${displayProductTitle}</div>
        <div class="drop-print-product-sku">SKU: ${item.sku}</div>
        <div class="drop-print-image-quality-title">Quality:</div>
        <div class="drop-print-image-quality-container"><div class="drop-print-image-quality-indicator" title="Print quality indicator"></div></div>
        <div class="drop-print-shipping-info" title="${shippingTitle}">Ships from: ${flagsHtml}</div>
        <div class="drop-print-item-cost">Cost to you: ${displayItemCost} ${displayShippingCostText}</div>
    </div>
    <div class="drop-print-product-actions">
        <div class="drop-print-sell-price-wrapper">
            <label for="drop-print-sell-price-${item.sku}" class="drop-print-sell-price-label" title="Your Selling Price">${storePriceCurrencySymbol}</label>
            <input type="number" class="drop-print-product-sell-price" id="drop-print-sell-price-${item.sku}" ${sellPriceValueAttr} ${sellPriceDataAttr} step="0.01" min="0" placeholder="Price" aria-label="Your selling price for ${displayProductTitle}" ${!isPriceEditable ? 'disabled title="Cannot set price. Base cost unavailable."' : 'title="Edit your selling price"'} />
        </div>
        <input type="checkbox" class="drop-print-product-checkbox" id="drop-print-product-checkbox-${item.sku}" value="${item.sku}" disabled>
        <label for="drop-print-product-checkbox-${item.sku}" class="screen-reader-text">Select ${displayProductTitle}</label>
    </div>
</div>`;
            });
        } else {
            productListHtml = '<p class="drop-print-info">No products found in this media type matching the current criteria.</p>';
        }
        productListContainer.empty().html(productListHtml || '<p class="drop-print-info">No products available for this selection.</p>');
        $(document).trigger('dropPrintProductListRendered');
        if (typeof window.dropPrint.triggerCurrencyMismatchCheck === 'function') {
            window.dropPrint.triggerCurrencyMismatchCheck(pricelistCurrencyCode);
        }
    }
    window.dropPrint.dropPrintRenderProductList = dropPrintRenderProductList;
    function dropPrintLoadPricelistFromJson(mediaId, usePreloaded = false) {
        const { config, elements, dropPrintUpdateOverlay, state } = window.dropPrint || {};
        if (typeof $ !== 'function' || typeof dropPrintUpdateOverlay !== 'function') {
            return $.Deferred().reject('Missing core jQuery/Overlay').promise();
        }
        if (usePreloaded && config?.defaultPricelist) {
            if (!dropPrintIsJsonDataValid(config.defaultPricelist)) {
                dropPrintHandleJsonLoadError('Preloaded default pricelist data is invalid.');
                window.dropPrint.printSizesData = null;
                if (state) state.printSizesData = null;
                return $.Deferred().reject('Invalid preloaded data').promise();
            } else {
                window.dropPrint.printSizesData = config.defaultPricelist;
                if (state) state.printSizesData = config.defaultPricelist;
                dropPrintRenderProductList();
                if (typeof dropPrintUpdateOverlay === 'function' && elements?.overlay?.hasClass('is-loading')) {
                    dropPrintUpdateOverlay('', '', '');
                }
                return $.Deferred().resolve(config.defaultPricelist).promise();
            }
        }
        if (!mediaId) {
            dropPrintHandleJsonLoadError('Cannot load pricelist: Media information is missing for the request.');
            if (state) state.printSizesData = null;
            window.dropPrint.printSizesData = null;
            return $.Deferred().reject('Missing media_id').promise();
        }
        if (elements?.productListContainer) {
            elements.productListContainer.html('<p class="drop-print-product-list-loading">Loading print products...</p>');
        }
        if (typeof dropPrintUpdateOverlay === 'function' && !elements?.overlay?.hasClass('is-loading')) {
            dropPrintUpdateOverlay('is-loading', 'Loading Pricelist', 'Fetching product information...', { showActions: false });
        }
        return $.ajax({
            url: config.ajaxUrl,
            dataType: 'json',
            method: 'POST',
            data: {
                action: config.getPricelistAction,
                nonce: config.nonce,
                media_id: mediaId
            }
        })
        .done(function(response) {
            if (!response.success || !dropPrintIsJsonDataValid(response.data)) {
                const errorMsg = response.data?.message || (response.success === false && typeof response.data === 'string' ? response.data : 'Fetched product data is invalid.');
                dropPrintHandleJsonLoadError(errorMsg);
                window.dropPrint.printSizesData = null;
                if (state) state.printSizesData = null;
            } else {
                window.dropPrint.printSizesData = response.data;
                if (state) state.printSizesData = response.data;
                dropPrintRenderProductList();
                if (typeof dropPrintUpdateOverlay === 'function' && elements?.overlay?.hasClass('is-loading')) {
                    dropPrintUpdateOverlay('', '', '');
                }
            }
        })
        .fail(function(jqXHR, textStatus, errorThrown) {
            let reason = `Failed to load product data (${textStatus}).`;
            if (jqXHR.responseJSON?.data?.message) {
                reason = jqXHR.responseJSON.data.message;
            } else if (jqXHR.responseText) {
                const shortResponse = jqXHR.responseText.length > 200 ? jqXHR.responseText.substring(0, 200) + '...' : jqXHR.responseText;
                reason += ` Server response: ${shortResponse}`;
            }
            dropPrintHandleJsonLoadError(reason);
            window.dropPrint.printSizesData = null;
            if (state) state.printSizesData = null;
        });
    }
    window.dropPrint.dropPrintLoadPricelistFromJson = dropPrintLoadPricelistFromJson;
    window.dropPrint.dropPrintIsJsonDataValid = dropPrintIsJsonDataValid;
    $(function() {
        const refreshButtonSelector = '#drop-print-overlay-refresh-btn';
        $('#drop-print-product-data-panel').on('click', refreshButtonSelector, function(e) {
            e.preventDefault();
            const {
                config: currentDropPrintConfig,
                dropPrintUpdateOverlay
            } = window.dropPrint || {};
            if (!currentDropPrintConfig || !currentDropPrintConfig.settingsPageUrl) {
                if(typeof dropPrintUpdateOverlay === 'function') {
                    dropPrintUpdateOverlay('is-error', 'Navigation Error', 'Unable to determine the settings page URL. Please check plugin configuration.', { buttonsToShow: ['ok'] });
                } else {
                    alert('Error: Settings page URL is not configured.');
                }
                return;
            }
            window.open(currentDropPrintConfig.settingsPageUrl, '_blank');
            if(typeof dropPrintUpdateOverlay === 'function') {
                 dropPrintUpdateOverlay('', '', '');
            }
        });
    });
}(jQuery));