/**
 * Handles the preparation and staging of selected products as WooCommerce variations.
 * Gathers checked items, their prices, and selected sizing options from the product list,
 * validates them, and sends them via AJAX to be saved as product meta.
 * Manages UI feedback and initiates file upload if part of a new file workflow.
 * Includes a currency mismatch check on initialization.
 */
(function($) {
    'use strict';
    window.dropPrint = window.dropPrint || {};
    function dropPrintInitVariationPrep() {
        const { elements, state, config, dropPrintUpdateOverlay, dropPrintReset } = window.dropPrint || {};
        if (config.metaVariationsCurrency && config.metaVariationsCurrency !== config.wcCurrency) {
            const title = "Warning: Currency Mismatch";
            const htmlMessage = `<p>You appear to have changed the WooCommerce currency (currently <strong>${config.wcCurrency}</strong>) since Drop Print variations were last saved for this product (using <strong>${config.metaVariationsCurrency}</strong>).</p><p>Proceeding may lead to incorrect pricing calculations if you add or modify variations now.</p><p><strong>Recommendation:</strong> Either revert the store currency to <strong>${config.metaVariationsCurrency}</strong> before editing, or reset Drop Print for this product and re-upload your file under the new currency (<strong>${config.wcCurrency}</strong>).</p>`;
            const options = { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok', 'docs', 'reset'] };
            if (typeof dropPrintUpdateOverlay === 'function') {
                dropPrintUpdateOverlay('is-prompt', title, htmlMessage, options);
            } else {
                alert(`Warning: Currency Mismatch\n\nYou appear to have changed the WooCommerce currency (${config.wcCurrency}) since setting up Drop Print variations for this product (originally ${config.metaVariationsCurrency}).\nProceeding may lead to incorrect pricing calculations.\nIt is strongly recommended you revert the store currency to ${config.metaVariationsCurrency} before managing variations for this product, or reset Drop Print data and start fresh with the new currency.`);
            }
        }
        if (!elements || !state || !config || typeof dropPrintUpdateOverlay !== 'function' || typeof dropPrintReset !== 'function') {
            return;
        }
        const $prepareButton = elements.dropPrintAddVariationsButton ? $(elements.dropPrintAddVariationsButton) : null;
        if (!$prepareButton || !$prepareButton.length) {
            return;
        }
        if ($prepareButton.data('dp-variation-initialized')) { return; }
        $prepareButton.data('dp-variation-initialized', true);
        $prepareButton.on('click', function(e) {
            e.preventDefault();
            const productListContainer = elements.productListContainer;
            if (!productListContainer || !productListContainer.length) {
                alert('DropPrint Error: Product list not found.');
                return;
            }
            const newlySelectedProducts = [];
            let hasInvalidPriceOrTitle = false;
            productListContainer.find('.drop-print-product .drop-print-product-checkbox:checked:not(:disabled)').each(function() {
                const $checkbox = $(this);
                const $product = $checkbox.closest('.drop-print-product');
                const sku = $product.data('sku');
                const attributeAndVariationTitle = $product.data('title');
                const $priceInput = $product.find('.drop-print-product-sell-price');
                let sellPrice = null;
                if (!attributeAndVariationTitle) {
                    alert(`DropPrint Error: Product with SKU ${sku} is missing the required 'data-title' attribute and cannot be processed.`);
                    hasInvalidPriceOrTitle = true;
                    return false;
                }
                if ($priceInput.length && !$priceInput.prop('disabled')) {
                    const rawValue = $priceInput.val();
                    sellPrice = parseFloat(rawValue);
                    if (isNaN(sellPrice) || sellPrice < 0) {
                        alert(`Invalid sell price for "${attributeAndVariationTitle || sku}" ("${rawValue}"). Please enter a valid positive number.`);
                        sellPrice = null;
                        hasInvalidPriceOrTitle = true;
                        return false;
                    }
                } else {
                    alert(`Cannot process checked item "${attributeAndVariationTitle || sku}" as its price is not available or editable.`);
                    hasInvalidPriceOrTitle = true;
                    return false;
                }
                const sizingOption = $product.data('selected-resize-method') || config?.default_image_resize || 'fillPrintArea';
                if (sku && sellPrice !== null) {
                    newlySelectedProducts.push({
                        sku: String(sku),
                        price: sellPrice,
                        title: String(attributeAndVariationTitle),
                        sizing: sizingOption
                    });
                } else {
                    hasInvalidPriceOrTitle = true;
                    return false;
                }
            });
            if (hasInvalidPriceOrTitle) {
                $prepareButton.prop('disabled', false);
                return;
            }
            if (newlySelectedProducts.length === 0) {
                alert('No products selected.');
                return;
            }
            if (!state) {
                alert('An internal error occurred. Cannot proceed.');
                return;
            }
            const isNewFileWorkflow = !!state.currentFile;
            const uploadAlreadyDone = state.uploadCompleteForDimensions === true;
            dropPrintUpdateOverlay('is-loading', 'Staging Variations', 'Saving selection...', { showActions: false });
            $prepareButton.prop('disabled', true);
            const stageAjaxData = {
                action: 'drop_print_stage_variations',
                nonce: config.nonce,
                product_id: config.productId,
                prospective_variations: newlySelectedProducts
            };
            $.ajax({
                url: config.ajaxUrl,
                type: 'POST',
                dataType: 'json',
                data: stageAjaxData,
                success(stageResponse) {
                    if (stageResponse?.success) {
                        if (window.dropPrint.state) {
                            window.dropPrint.state.metaActiveVariations = newlySelectedProducts.map((v) => ({
                                sku: String(v.sku),
                                price: parseFloat(v.price).toFixed(2),
                                title: String(v.title)
                            }));
                            window.dropPrint.state.prospectiveVariations = null;
                            window.dropPrint.state.dropPrintRequiresSave = true;
                            if (typeof window.dropPrint.dropPrintUpdateTabAppearance === 'function') {
                                window.dropPrint.dropPrintUpdateTabAppearance(true);
                            }
                        }
                        if (isNewFileWorkflow && !uploadAlreadyDone) {
                            if (!state.dropzoneInstance) {
                                alert('DropPrint Error: Upload component not ready. Please reset.');
                                dropPrintUpdateOverlay('is-error', 'Internal Error', 'Upload component missing. Please reset.', { showActions: true, buttonsToShow: ['ok', 'reset'] });
                                $prepareButton.prop('disabled', false);
                                return;
                            }
                            dropPrintUpdateOverlay('is-loading', 'Starting Upload', 'Please wait...', { showActions: false });
                            try {
                                state.dropzoneInstance.processQueue();
                            } catch (dzError) {
                                dropPrintUpdateOverlay('is-error', 'Upload Error', 'Could not start the main upload process. Please try again.', { showActions: true, buttonsToShow: ['ok', 'reset'] });
                                $prepareButton.prop('disabled', false);
                            }
                        } else {
                            const promptTitle = isNewFileWorkflow ? 'File Uploaded' : 'Product Save Required';
                            const promptMsg = '<p>Product variations will now be updated. However, you need to save the product to see these changes in effect.</p><p>Click <strong>\'Save Now\'</strong> or use the \'Publish\' button above.</p>';
                            dropPrintUpdateOverlay('is-prompt', promptTitle, promptMsg, { showActions: true, iconClass: 'dashicons-yes-alt', buttonsToShow: ['docs', 'save_now', 'ok'], });
                            $prepareButton.prop('disabled', false);
                        }
                    } else {
                        const errorMsg = stageResponse?.data?.message || 'Failed to stage variations.';
                        dropPrintUpdateOverlay('is-error', 'Staging Failed', errorMsg, { showActions: true, buttonsToShow: ['ok', 'reset'], iconClass: 'dashicons-warning' });
                        $prepareButton.prop('disabled', false);
                    }
                },
                error(jqXHR, textStatus, errorThrown) {
                    let errorMsg = 'AJAX error staging variations.';
                    try { errorMsg = JSON.parse(jqXHR.responseText)?.data?.message || errorMsg; } catch (e) {}
                    dropPrintUpdateOverlay('is-error', 'Staging Request Error', errorMsg, { showActions: true, buttonsToShow: ['ok', 'reset'], iconClass: 'dashicons-dismiss' });
                    $prepareButton.prop('disabled', false);
                },
            });
        });
    }
    $(function() {
        const jsonPromise = window.dropPrint?.state?.jsonLoadingPromise;
        if (jsonPromise && typeof jsonPromise.always === 'function') {
            jsonPromise.always(function() {
                if (window.dropPrint?.printSizesData || $('#drop-print-product-list-container .drop-print-product').length > 0 || window.dropPrint?.state?.metaActiveVariations?.length > 0) {
                    dropPrintInitVariationPrep();
                }
            });
        } else {
            if ($('#drop-print-product-list-container .drop-print-product').length > 0 || window.dropPrint?.state?.metaActiveVariations?.length > 0) {
                dropPrintInitVariationPrep();
            }
        }
    });
}(jQuery));