/**
 * Manages product selection interactions within the Drop Print panel (v1.1).
 * This script handles product clicks, checkbox state changes, sell price input,
 * and quality warnings. It includes a debounced mechanism for the "Low Sell Price
 * Warning" to prevent excessive notifications and resolves a TypeError related to
 * currency symbol retrieval for this warning. For the "Image Resize May Be Required"
 * dialog, it ensures the user's chosen resize method is explicitly confirmed via
 * an "OK" button (rendered by the updated overlay system) and then correctly
 * stored for later use by the variation handler.
 */
(function($) {
    'use strict';
    window.dropPrint = window.dropPrint || {};
    window.dropPrint.state = window.dropPrint.state || {};
    if (typeof window.dropPrint.state.dropPrintRequiresSave === 'undefined') {
        window.dropPrint.state.dropPrintRequiresSave = false;
    }
    const MATCH_QUALITY_THRESHOLD_FOR_RESIZE_PROMPT = 95;
    const debouncedLowPriceCheck = (typeof _ !== 'undefined' && typeof _.debounce === 'function') ?
        _.debounce(function($productDiv, priceValue, itemCost) {
            if (typeof window.dropPrint.dropPrintUpdateOverlay !== 'function' || typeof window.dropPrint.config === 'undefined') return;
            const wcSymbol = window.dropPrint.config.wcCurrencySymbol || '$';
            let itemCurrencySymbol = '';
            const itemCurrencyCode = $productDiv.data('currency')?.toUpperCase();
            if (itemCurrencyCode) {
                const symbols = { 'USD': '$', 'GBP': '£', 'EUR': '€', 'AUD': 'A$', 'CAD': 'C$', 'ZAR': 'R' };
                itemCurrencySymbol = symbols[itemCurrencyCode] || itemCurrencyCode;
            }
            if (!itemCurrencySymbol && window.dropPrint?.printSizesData?.currency_symbol) {
                itemCurrencySymbol = window.dropPrint.printSizesData.currency_symbol;
            } else if (!itemCurrencySymbol && itemCurrencyCode) {
                itemCurrencySymbol = itemCurrencyCode;
            }
            if (!isNaN(priceValue) && !isNaN(itemCost) && priceValue > 0 && itemCost > 0 && priceValue <= itemCost) {
                const warningTitle = "Low Sell Price Warning";
                const warningMsg = `<p>The entered sell price (<strong>${wcSymbol}${priceValue.toFixed(2)}</strong>) is less than or equal to the estimated cost price (<strong>${itemCurrencySymbol}${itemCost.toFixed(2)}</strong>).</p><p>Are you sure you want to proceed with this price?</p>`;
                window.dropPrint.dropPrintUpdateOverlay('is-prompt', warningTitle, warningMsg, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok'] });
            }
        }, 750)
        : function($productDiv, priceValue, itemCost) {
            if (typeof window.dropPrint.dropPrintUpdateOverlay !== 'function' || typeof window.dropPrint.config === 'undefined') return;
            const wcSymbol = window.dropPrint.config.wcCurrencySymbol || '$';
            let itemCurrencySymbol = '';
            const itemCurrencyCode = $productDiv.data('currency')?.toUpperCase();
            if (itemCurrencyCode) {
                const symbols = { 'USD': '$', 'GBP': '£', 'EUR': '€', 'AUD': 'A$', 'CAD': 'C$', 'ZAR': 'R' };
                itemCurrencySymbol = symbols[itemCurrencyCode] || itemCurrencyCode;
            }
            if (!itemCurrencySymbol && window.dropPrint?.printSizesData?.currency_symbol) {
                itemCurrencySymbol = window.dropPrint.printSizesData.currency_symbol;
            } else if (!itemCurrencySymbol && itemCurrencyCode) {
                itemCurrencySymbol = itemCurrencyCode;
            }
            if (!isNaN(priceValue) && !isNaN(itemCost) && priceValue > 0 && itemCost > 0 && priceValue <= itemCost) {
                const warningTitle = "Low Sell Price Warning";
                const warningMsg = `<p>The entered sell price (<strong>${wcSymbol}${priceValue.toFixed(2)}</strong>) is less than or equal to the estimated cost price (<strong>${itemCurrencySymbol}${itemCost.toFixed(2)}</strong>).</p><p>Are you sure you want to proceed with this price?</p>`;
                window.dropPrint.dropPrintUpdateOverlay('is-prompt', warningTitle, warningMsg, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok'] });
            }
        };
    function dropPrintBindProductEvents() {
        const productListContainer = window.dropPrint?.elements?.productListContainer?.get(0);
        if (!productListContainer) {
            return;
        }
        productListContainer.removeEventListener('click', handleProductClick);
        productListContainer.addEventListener('click', handleProductClick);
        productListContainer.removeEventListener('change', handleCheckboxChange);
        productListContainer.addEventListener('change', handleCheckboxChange);
        $(productListContainer).off('input.dropPrintSellPrice', '.drop-print-product-sell-price');
        $(productListContainer).on('input.dropPrintSellPrice', '.drop-print-product-sell-price', handlePriceInput);
        $(productListContainer).off('click.dropPrintShippingInfo', '.drop-print-shipping-info-icon');
        $(productListContainer).on('click.dropPrintShippingInfo', '.drop-print-shipping-info-icon', handleShippingInfoIconClick);
    }
    function handleShippingInfoIconClick(e) {
        e.preventDefault();
        e.stopPropagation();
        const { dropPrintUpdateOverlay } = window.dropPrint || {};
        if (typeof dropPrintUpdateOverlay !== 'function') {
            alert('Delivery information system is unavailable.');
            return;
        }
        const title = "Delivery Cost Information";
        const message = "<p>Delivery prices  displayed here are estimates. To view the actual cost of delivery, you should know first where you wish to deliver to. Use the <a href='https://api.drop-print.com/my-account/quote/' target='_blank' rel='noopener noreferrer'>quote page</a> to view an exact cost of delivery to any destination before you set this price based on your own needs.</p>";
        dropPrintUpdateOverlay('is-prompt', title, message, { showActions: true, iconClass: 'dashicons-info-outline', buttonsToShow: ['ok'] });
    }
    function handleProductClick(e) {
        const productDiv = e.target.closest('.drop-print-product');
        if (!productDiv) return;
        const $productDiv = $(productDiv);
        const $checkbox = $productDiv.find('.drop-print-product-checkbox');
        const isSelectable = $productDiv.data('is-selectable') !== false;
        const isCheckboxDisabled = $checkbox.length ? $checkbox.prop('disabled') : true;
        if (!isSelectable || isCheckboxDisabled) {
            e.preventDefault();
            e.stopPropagation();
            const productTitle = $productDiv.find('.drop-print-product-title').text() || $productDiv.data('title') || 'Selected Product';
            const productPixels = $productDiv.data('pixels') || 'Unknown Dimensions';
            let currentFilePixels = 'Unknown';
            if (window.dropPrint?.state?.fileDimensions?.width && window.dropPrint?.state?.fileDimensions?.height) {
                currentFilePixels = `${window.dropPrint.state.fileDimensions.width}x${window.dropPrint.state.fileDimensions.height}`;
            }
            const scoredProductData = window.dropPrint?.state?.scoredProducts?.find(p => String(p.sku) === String($productDiv.data('sku')));
            let reasonMessage = 'This print size is not a suitable match for your image based on the calculated quality score (considering factors like DPI, cropping, and scaling).';
            if (scoredProductData && scoredProductData.finalReasoning) {
                reasonMessage = `This print size is not suitable.\nReason: ${scoredProductData.finalReasoning}.`;
            }
            const message = `<p><strong>${productTitle}</strong> (requires approx. ${productPixels}px)</p><p>${reasonMessage}</p><p>Your image dimensions are ${currentFilePixels}px.</p><p>Selecting this option is not recommended due to potential poor print quality. You can download a template for this size to adjust your artwork, or reset and choose a different image/product.</p>`;
            const { elements: currentElements, dropPrintUpdateOverlay: currentUpdateOverlay } = window.dropPrint || {};
            if (!currentElements?.overlay || typeof currentUpdateOverlay !== 'function') {
                alert(message.replace(/<[^>]*>/g, '\n').replace(/\n+/g, '\n').trim());
                return;
            }
            currentElements.overlay.data('template-dimensions', productPixels);
            currentElements.overlay.data('template-title', productTitle);
            currentUpdateOverlay('is-prompt', 'Unsuitable Print Size', message, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['reset', 'download', 'docs', 'ok'] });
            return;
        }
        if (!e.target.closest('.drop-print-product-checkbox, .drop-print-product-actions, .drop-print-shipping-info-icon')) {
            if ($checkbox.length && !$checkbox.prop('disabled')) {
                $checkbox.prop('checked', !$checkbox.prop('checked'));
                const changeEvent = new Event('change', { bubbles: true });
                $checkbox.get(0).dispatchEvent(changeEvent);
            }
        }
    }
    function handleCheckboxChange(e) {
        if (!e.target.matches('.drop-print-product-checkbox')) return;
        const checkbox = e.target;
        const product = checkbox.closest('.drop-print-product');
        const $productDiv = $(product);
        const config = window.dropPrint?.config;
        const state = window.dropPrint?.state;
        if (product) {
            $productDiv.toggleClass('selected', checkbox.checked);
            if (checkbox.checked && !window.dropPrint.state.dropPrintRequiresSave) {
                window.dropPrint.state.dropPrintRequiresSave = true;
                if (typeof window.dropPrint.dropPrintUpdateTabAppearance === 'function') {
                    window.dropPrint.dropPrintUpdateTabAppearance(true);
                }
            }
            const $priceInput = $productDiv.find('.drop-print-product-sell-price');
            if (checkbox.checked) {
                const itemCost = parseFloat($productDiv.data('cost'));
                const priceMarginPercent = parseFloat(config?.priceMargin);
                if (!isNaN(itemCost) && itemCost >= 0 && !isNaN(priceMarginPercent) && $priceInput.length && ($priceInput.val() === "" || $priceInput.val() === "0" || $priceInput.val() === "0.00")) {
                    const marginMultiplier = 1 + (priceMarginPercent / 100);
                    const calculatedSellPrice = (itemCost * marginMultiplier).toFixed(2);
                    $priceInput.val(calculatedSellPrice).attr('data-price', calculatedSellPrice);
                }
                const itemCurrencyCode = $productDiv.data('currency')?.toUpperCase();
                const storeCurrency = config?.wcCurrency?.toUpperCase();
                if (itemCurrencyCode && storeCurrency && itemCurrencyCode !== storeCurrency) {
                    const costValue = $productDiv.data('cost');
                    if (costValue !== undefined && costValue !== '' && !isNaN(parseFloat(costValue))) {
                        let costSymbol = '';
                        const symbols = { 'USD': '$', 'GBP': '£', 'EUR': '€', 'AUD': 'A$', 'CAD': 'C$', 'ZAR': 'R' };
                        costSymbol = symbols[itemCurrencyCode] || itemCurrencyCode;
                        if (!costSymbol && window.dropPrint?.printSizesData?.currency_symbol) {
                           costSymbol = window.dropPrint.printSizesData.currency_symbol;
                        } else if (!costSymbol && itemCurrencyCode) {
                           costSymbol = itemCurrencyCode;
                        }
                        const warningTitle = 'Currency Mismatch Notice';
                        const warningMsg = `<p><strong>Warning:</strong> The cost currency for this item (<strong>${itemCurrencyCode}</strong>) is different from your WooCommerce store currency (<strong>${storeCurrency}</strong>).</p><p>The estimated cost for this item is <strong>${costSymbol}${parseFloat(costValue).toFixed(2)}</strong>.</p><p>Please ensure your final Sell Price (in ${storeCurrency}) accounts for currency conversion rates and fees.</p><p><small>Click 'OK' to acknowledge this notice.</small></p>`;
                        if (typeof window.dropPrint.dropPrintUpdateOverlay === 'function') {
                            window.dropPrint.dropPrintUpdateOverlay('is-prompt', warningTitle, warningMsg, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok', 'docs'] });
                        } else {
                            alert(warningMsg.replace(/<[^>]*>/g, '\n').replace(/\n+/g, '\n').trim());
                        }
                    }
                }
                const scoredProductData = state?.scoredProducts?.find(p => String(p.sku) === String($productDiv.data('sku')));
                let currentResizeMethod = config?.default_image_resize || 'fillPrintArea';
                if (scoredProductData && scoredProductData.score < MATCH_QUALITY_THRESHOLD_FOR_RESIZE_PROMPT && typeof window.dropPrint.dropPrintUpdateOverlay === 'function') {
                    const resizeOptions = [
                        { key: 'fillPrintArea', name: 'Fill Paper Size' },
                        { key: 'fitPrintArea', name: 'Fit to Paper Size' },
                        { key: 'stretchToPrintArea', name: 'Stretch to Fit Paper Size' },
                    ];
                    let selectHtml = '<select id="drop-print-resize-select">';
                    resizeOptions.forEach(opt => {
                        const selected = (currentResizeMethod === opt.key) ? ' selected' : '';
                        selectHtml += `<option value="${opt.key}"${selected}>${opt.name}</option>`;
                    });
                    selectHtml += '</select>';
                    const resizeTitle = "Image Resize May Be Required";
                    const resizeMsg = `<p>This print size has a match score of <strong>${scoredProductData.score}%</strong>. Your image may need to be slightly resized or cropped to fit optimally.</p><p>Please choose a resize method:</p>${selectHtml}`;
                    window.dropPrint.dropPrintUpdateOverlay('is-prompt', resizeTitle, resizeMsg, {
                        showActions: true,
                        iconClass: 'dashicons-image-rotate',
                        buttonsToShow: [],
                        customButtons: [
                            {
                                id: 'drop-print-overlay-confirm-resize-btn',
                                text: 'OK',
                                class: 'button-primary',
                                action: function() {
                                    const selectedResize = $('#drop-print-resize-select').val();
                                    $productDiv.data('selected-resize-method', selectedResize || currentResizeMethod);
                                    window.dropPrint.dropPrintUpdateOverlay('', '', '');
                                }
                            }
                        ]
                    });
                } else {
                    $productDiv.data('selected-resize-method', currentResizeMethod);
                }
            }
            if (typeof window.dropPrint.dropPrintUpdateButtonState === 'function') {
                window.dropPrint.dropPrintUpdateButtonState();
            }
        }
    }
    function handlePriceInput(e) {
        const $input = $(e.target);
        $input.attr('data-price', $input.val());
        if (window.dropPrint && !window.dropPrint.state.dropPrintRequiresSave) {
            window.dropPrint.state.dropPrintRequiresSave = true;
            if (typeof window.dropPrint.dropPrintUpdateTabAppearance === 'function') {
                window.dropPrint.dropPrintUpdateTabAppearance(true);
            }
        }
        const priceValue = parseFloat($input.val());
        const $productDiv = $input.closest('.drop-print-product');
        const itemCost = parseFloat($productDiv.data('cost'));
        debouncedLowPriceCheck($productDiv, priceValue, itemCost);
        if (!isNaN(priceValue) && priceValue >= 0) {
            const $checkbox = $productDiv.find('.drop-print-product-checkbox');
            if ($checkbox.length && !$checkbox.prop('checked') && !$checkbox.prop('disabled')) {
                $checkbox.prop('checked', true);
                const changeEvent = new Event('change', { bubbles: true });
                $checkbox.get(0).dispatchEvent(changeEvent);
            }
        }
    }
    window.dropPrint.dropPrintUpdateButtonState = function() {
        const { elements, state } = window.dropPrint || {};
        const saveDropPrintChangesButton = elements?.dropPrintAddVariationsButton;
        if (!saveDropPrintChangesButton) {
            return;
        }
        const $button = $(saveDropPrintChangesButton);
        const hasExistingDimensions = state?.metaActiveDimensions?.width > 0 && state?.metaActiveDimensions?.height > 0;
        const hasExistingFileRef = state?.metaFileName || state?.metaPreviewId > 0;
        const inExistingMetaWorkflow = hasExistingDimensions && hasExistingFileRef && !state?.currentFile;
        const checkedAndEnabledBoxes = document.querySelectorAll('#drop-print-product-list-container .drop-print-product-checkbox:checked:not(:disabled)');
        const enableButton = checkedAndEnabledBoxes.length > 0;
        const isDisabled = !enableButton;
        const wasDisabled = $button.prop('disabled');
        $button.prop('disabled', isDisabled);
        $button.text(inExistingMetaWorkflow ? 'Modify Selected Variations' : 'Create Product Variations');
        if (!isDisabled && wasDisabled) {
            $button.addClass('drop-print-pulse');
        } else if (isDisabled) {
            $button.removeClass('drop-print-pulse');
        }
    };
    $(document).on('productsRenderedandMatched.dropPrint', function() {
        dropPrintBindProductEvents();
        if (typeof window.dropPrint.dropPrintUpdateButtonState === 'function') {
            window.dropPrint.dropPrintUpdateButtonState();
        }
    });
}(jQuery));