/**
 * Calculates product match scores for Drop Print (v1.1).
 * This script evaluates how well a user's image fits various print product sizes.
 * The core scoring logic (in 'dropPrintCalculateScore') now uses a two-stage
 * approach:
 * 1. "Good Fit" Check: Determines if the image can fit the product dimensions
 * without cropping more than a 0.20-inch physical limit from the image. This
 * acts as a primary gate.
 * 2. DPI Score: If a "Good Fit", a nuanced DPI score is calculated. It rewards
 * DPI in the 270-349 range as excellent, penalizes very low DPI (<150 DPI
 * heavily, 150-269 DPI on a sharp negative scale), and also applies a slight
 * negative sliding penalty for excessively high DPI (350-600+ DPI).
 * The final score influences UI indicators, product suitability, and sorting.
 */
(function($) {
    'use strict';
    window.dropPrint = window.dropPrint || {};
    const CROP_LIMIT_INCHES = 0.20;
    const DPI_EXCELLENT_MIN = 270;
    const DPI_EXCELLENT_MAX = 349;
    const DPI_HIGH_PENALTY_START = 350;
    const DPI_HIGH_PENALTY_END = 5000;
    const DPI_LOW_PENALTY_MAX = 269;
    const DPI_MIN_ACCEPTABLE = 150;
    const SCORE_EXCELLENT_DPI = 100;
    const SCORE_AT_HIGH_PENALTY_START = 95;
    const SCORE_AT_EXTREME_DPI_LIMIT = 30;
    const SCORE_AT_MIN_ACCEPTABLE_DPI = 5;
    const MIN_ACCEPTABLE_SCORE_THRESHOLD = 30;
    const SELECTABLE_SCORE_THRESHOLD = 40;
    const QUALITY_THRESHOLDS = { excellent: 90, good: 70, fair: 40, poor: 10 };
    function getPixelDimensions(pixelsStr) {
        if (!pixelsStr || typeof pixelsStr !== 'string') return null;
        const dims = String(pixelsStr).split('x').map(Number);
        if (dims.length !== 2 || dims.some(isNaN) || dims.some((d) => d <= 0)) {
            return null;
        }
        return { width: dims[0], height: dims[1], longest: Math.max(dims[0], dims[1]), shortest: Math.min(dims[0], dims[1]) };
    }
    function getPhysicalDimensionsInches(dimensionsInStr) {
        if (!dimensionsInStr || typeof dimensionsInStr !== 'string') return null;
        const cleanedStr = String(dimensionsInStr).toLowerCase().replace(/\s*inches/g, '').replace(/\s/g, '');
        const dims = cleanedStr.split('x').map(Number);
        if (dims.length !== 2 || dims.some(isNaN) || dims.some((d) => d <= 0)) {
            return null;
        }
        return { width: dims[0], height: dims[1], longest: Math.max(dims[0], dims[1]), shortest: Math.min(dims[0], dims[1]) };
    }
    function dropPrintGetProductData() {
        const products = [];
        const container = window.dropPrint?.elements?.productListContainer || $('#drop-print-product-list-container');
        if (!container.length) {
            return products;
        }
        container.find('.drop-print-product[data-pixels][data-sku][data-in]').each(function() {
            const productElement = this;
            const $productElement = $(productElement);
            const pixelsStr = $productElement.data('pixels');
            const inchesStr = $productElement.data('in');
            const sku = $productElement.data('sku');
            const title = $productElement.find('.drop-print-product-title').text().trim();
            const productPxDims = getPixelDimensions(pixelsStr);
            const productInDims = getPhysicalDimensionsInches(inchesStr);
            if (!productPxDims || !productInDims || !sku) {
                return;
            }
            products.push({
                element: productElement,
                sku: String(sku),
                pxWidth: productPxDims.width,
                pxHeight: productPxDims.height,
                pxLongest: productPxDims.longest,
                pxShortest: productPxDims.shortest,
                inWidth: productInDims.width,
                inHeight: productInDims.height,
                inLongest: productInDims.longest,
                inShortest: productInDims.shortest,
                title: title,
            });
        });
        return products;
    }
    function calculateGoodFitScore(clientImgPxW, clientImgPxH, paperInW, paperInH) {
        const imgAspect = clientImgPxW / clientImgPxH;
        const paperAspect = paperInW / paperInH;
        let imageCropInches = 0;
        if (Math.abs(imgAspect - paperAspect) > 0.001) {
            if (imgAspect > paperAspect) {
                const effectiveDPI = clientImgPxH / paperInH;
                const scaledImageWidthInches = clientImgPxW / effectiveDPI;
                imageCropInches = scaledImageWidthInches - paperInW;
            } else {
                const effectiveDPI = clientImgPxW / paperInW;
                const scaledImageHeightInches = clientImgPxH / effectiveDPI;
                imageCropInches = scaledImageHeightInches - paperInH;
            }
        }
        imageCropInches = Math.max(0, imageCropInches);
        const isGoodFit = imageCropInches <= CROP_LIMIT_INCHES;
        return { score: isGoodFit ? 100 : 0, cropAmountInches: imageCropInches, isGoodFit: isGoodFit };
    }
    function calculateDPIWithNuanceScore(actualPrintDPI) {
        let score = 0;
        if (actualPrintDPI < DPI_MIN_ACCEPTABLE) {
            score = 0;
        } else if (actualPrintDPI <= DPI_LOW_PENALTY_MAX) {
            score = SCORE_AT_MIN_ACCEPTABLE_DPI + ((actualPrintDPI - DPI_MIN_ACCEPTABLE) / (DPI_LOW_PENALTY_MAX - DPI_MIN_ACCEPTABLE)) * (SCORE_AT_EXTREME_DPI_LIMIT - SCORE_AT_MIN_ACCEPTABLE_DPI);
        } else if (actualPrintDPI >= DPI_EXCELLENT_MIN && actualPrintDPI <= DPI_EXCELLENT_MAX) {
            score = SCORE_EXCELLENT_DPI;
        } else if (actualPrintDPI >= DPI_HIGH_PENALTY_START && actualPrintDPI <= DPI_HIGH_PENALTY_END) {
            score = SCORE_AT_HIGH_PENALTY_START - (((actualPrintDPI - DPI_HIGH_PENALTY_START) / (DPI_HIGH_PENALTY_END - DPI_HIGH_PENALTY_START)) * (SCORE_AT_HIGH_PENALTY_START - SCORE_AT_EXTREME_DPI_LIMIT));
        } else if (actualPrintDPI > DPI_HIGH_PENALTY_END) {
            score = SCORE_AT_EXTREME_DPI_LIMIT;
        } else {
            score = SCORE_EXCELLENT_DPI;
        }
        return { score: Math.max(0, Math.min(100, Math.round(score))), actualDPI: actualPrintDPI };
    }
    function dropPrintCalculateScore(product, clientImgLongestPx, clientImgShortestPx, clientImgActualWidth, clientImgActualHeight) {
        if (!clientImgActualWidth || clientImgActualWidth <= 0 || !clientImgActualHeight || clientImgActualHeight <= 0 || !product.inWidth || product.inWidth <= 0 || !product.inHeight || product.inHeight <= 0) {
            return { ...product, score: 0, finalReasoning: "Invalid image or product dimensions.", cropAmountInches: 0, actualDPI: 0, isAcceptableAspect: false };
        }
        const fitResult = calculateGoodFitScore(clientImgActualWidth, clientImgActualHeight, product.inWidth, product.inHeight);
        let actualPrintDPI = 0;
        if (fitResult.isGoodFit) {
            const imgAspect = clientImgActualWidth / clientImgActualHeight;
            const paperAspect = product.inWidth / product.inHeight;
            if (Math.abs(imgAspect - paperAspect) < 0.001) {
                actualPrintDPI = clientImgActualWidth / product.inWidth;
            } else if (imgAspect > paperAspect) {
                actualPrintDPI = clientImgActualHeight / product.inHeight;
            } else {
                actualPrintDPI = clientImgActualWidth / product.inWidth;
            }
        }
        const dpiResult = calculateDPIWithNuanceScore(actualPrintDPI);
        let finalScore = 0;
        let finalReasoning = [];
        if (!fitResult.isGoodFit) {
            finalScore = 0;
            finalReasoning.push(`Would require excessive cropping. Your image would require at least a ${fitResult.cropAmountInches.toFixed(2)}" crop).`);
            finalReasoning.push(`Low DPI (${dpiResult.actualDPI.toFixed(0)}) or not evaluated due to fit.`);
        } else {
            finalScore = dpiResult.score;
            finalReasoning.push(`Good fit (Image crop: ${fitResult.cropAmountInches.toFixed(2)}").`);
            if (dpiResult.actualDPI >= DPI_EXCELLENT_MIN && dpiResult.actualDPI <= DPI_EXCELLENT_MAX) finalReasoning.push(`Excellent DPI (${dpiResult.actualDPI.toFixed(0)}).`);
            else if (dpiResult.score >= SCORE_AT_HIGH_PENALTY_START && dpiResult.actualDPI >= DPI_HIGH_PENALTY_START) finalReasoning.push(`High DPI (${dpiResult.actualDPI.toFixed(0)}); slight quality reduction.`);
            else if (dpiResult.score >= SCORE_AT_EXTREME_DPI_LIMIT && dpiResult.actualDPI > DPI_EXCELLENT_MAX) finalReasoning.push(`Very High DPI (${dpiResult.actualDPI.toFixed(0)}); penalized.`);
            else if (dpiResult.actualDPI >= DPI_MIN_ACCEPTABLE) finalReasoning.push(`Acceptable DPI (${dpiResult.actualDPI.toFixed(0)}).`);
            else finalReasoning.push(`Low DPI (${dpiResult.actualDPI.toFixed(0)}).`);
        }
        return {
            ...product,
            score: Math.max(0, Math.min(100, Math.round(finalScore))),
            cropAmountInches: fitResult.cropAmountInches,
            actualDPI: dpiResult.actualDPI,
            isAcceptableAspect: fitResult.isGoodFit,
            finalReasoning: finalReasoning.join(' '),
        };
    }
    function dropPrintUpdateProductDisplay(product) {
        const $productElement = $(product.element);
        const $indicator = $productElement.find('.drop-print-image-quality-indicator');
        if (!$indicator.length) return;
        let qualityClass = 'poor';
        if (product.score >= QUALITY_THRESHOLDS.excellent) { qualityClass = 'excellent';}
        else if (product.score >= QUALITY_THRESHOLDS.good) { qualityClass = 'good';}
        else if (product.score >= QUALITY_THRESHOLDS.fair) { qualityClass = 'fair';}
        else if (product.score >= QUALITY_THRESHOLDS.poor) { qualityClass = 'poor'; }
        else { qualityClass = 'poor';}
        const tooltipText = `Match Score: ${product.score}/100. ${product.finalReasoning || ''}`;
        $indicator.removeClass('poor fair good excellent exact-match').addClass(qualityClass).css('width', product.score + '%').attr('title', tooltipText);
        const isUnsuitableOverall = !product.isAcceptableAspect || product.score < MIN_ACCEPTABLE_SCORE_THRESHOLD;
        $productElement.data('is-unsuitable', isUnsuitableOverall);
        $productElement.data('score', product.score);
        $productElement.toggleClass('unsuitable-product', isUnsuitableOverall);
    }
    function dropPrintHighlightBestMatch(allScoredProducts) {
        const { state, elements } = window.dropPrint || {};
        const container = elements?.productListContainer || $('#drop-print-product-list-container');
        if (!container.length) {
            return;
        }
        container.find('.drop-print-product').removeClass('selected match')
            .find('.drop-print-product-checkbox').prop('disabled', true).prop('checked', false);
        allScoredProducts.forEach(product => {
            const $currentProduct = $(product.element);
            const $checkbox = $currentProduct.find('.drop-print-product-checkbox');
            const isMarkedUnsuitableByDisplayLogic = $currentProduct.data('is-unsuitable') === true;
            const canBeSelected = product.score >= SELECTABLE_SCORE_THRESHOLD && !isMarkedUnsuitableByDisplayLogic;
            if ($checkbox.length) {
                $checkbox.prop('disabled', !canBeSelected);
            }
            $currentProduct.data('is-selectable', canBeSelected);
            if (isMarkedUnsuitableByDisplayLogic) {
                $currentProduct.addClass('drop-print-hidden-non-match');
            } else {
                $currentProduct.removeClass('drop-print-hidden-non-match');
            }
        });
        const bestMatchCandidate = allScoredProducts.length > 0 ? allScoredProducts[0] : null;
        if (bestMatchCandidate && bestMatchCandidate.element && bestMatchCandidate.score >= SELECTABLE_SCORE_THRESHOLD && !$(bestMatchCandidate.element).hasClass('drop-print-hidden-non-match')) {
            $(bestMatchCandidate.element).addClass('match');
        }
        if (typeof window.dropPrint.handleProductSearch === 'function') {
            window.dropPrint.handleProductSearch();
        } else {
            if (elements?.productListContainer && state) {
                 elements.productListContainer.find('.drop-print-product').each(function() {
                    const $prod = $(this);
                    if (state.showingAllProducts) {
                        $prod.show();
                    } else {
                        if ($prod.hasClass('drop-print-hidden-non-match')) {
                            $prod.hide();
                        } else {
                            $prod.show();
                        }
                    }
                });
            }
        }
        if (bestMatchCandidate && bestMatchCandidate.element && $(bestMatchCandidate.element).is(':visible') && container.height() < container[0].scrollHeight) {
            const productTop = $(bestMatchCandidate.element).position().top;
            const containerScrollTop = container.scrollTop();
            const containerHeight = container.height();
            const productHeight = $(bestMatchCandidate.element).outerHeight();
            if (productTop < 0 || (productTop + productHeight) > containerHeight) {
                const targetScrollTop = containerScrollTop + productTop - Math.min(50, containerHeight / 4);
                container.stop().animate({ scrollTop: targetScrollTop }, 300);
            }
        }
    }
    window.dropPrint.dropPrintInitiateMatching = function(imgLongestEdge, imgShortestEdge, imgPixelWidth, imgPixelHeight) {
        const { state, elements, dropPrintUpdateOverlay } = window.dropPrint || {};
        if (typeof dropPrintUpdateOverlay !== 'function') {
            alert('DropPrint Critical Error: UI feedback system unavailable.');
            return;
        }
        const allDomProducts = dropPrintGetProductData();
        if (!allDomProducts || allDomProducts.length === 0) {
            if (elements?.overlay?.hasClass('is-loading')) {
                dropPrintUpdateOverlay('', '', '');
            }
            if (window.dropPrint?.printSizesData) {
                dropPrintUpdateOverlay('is-info', 'No Products Rendered', 'Could not find product elements to perform matching. List might be empty or render failed.', { showActions: true, buttonsToShow: ['ok', 'reset', 'refresh'], iconClass: 'dashicons-info-outline' });
            }
            return;
        }
        const clientImgActualWidth = imgPixelWidth || (imgLongestEdge > imgShortestEdge ? imgLongestEdge : imgShortestEdge);
        const clientImgActualHeight = imgPixelHeight || (imgLongestEdge > imgShortestEdge ? imgShortestEdge : imgLongestEdge);
        const scoredProducts = allDomProducts.map((product) => dropPrintCalculateScore(product, imgLongestEdge, imgShortestEdge, clientImgActualWidth, clientImgActualHeight));
        if (window.dropPrint.state) window.dropPrint.state.scoredProducts = scoredProducts;
        scoredProducts.forEach(dropPrintUpdateProductDisplay);
        scoredProducts.sort((a, b) => b.score - a.score);
        const container = elements?.productListContainer || $('#drop-print-product-list-container');
        if (container.length) {
            const productElements = scoredProducts.map(p => p.element);
            console.log('DEBUG: [Matching] About to re-append sorted product elements. Count:', productElements.length);
            container.append(productElements);
        }
        dropPrintHighlightBestMatch(scoredProducts);
        const bestOverallMatch = scoredProducts.length > 0 ? scoredProducts[0] : null;
        if (bestOverallMatch && bestOverallMatch.score >= SELECTABLE_SCORE_THRESHOLD) {
            if (state) state.currentBestMatch = bestOverallMatch.sku;
        } else {
            if (state) state.currentBestMatch = null;
        }
        const visibleProducts = container.find('.drop-print-product:visible').length;
        if (visibleProducts === 0 && allDomProducts.length > 0 && state && !state.showingAllProducts) {
            dropPrintUpdateOverlay('is-info', 'No Suitable Matches Found', 'No products closely match your image specifications. Try "Show All Options", check image resolution, or adjust your image.', { showActions: true, buttonsToShow: ['ok', 'reset'], iconClass: 'dashicons-info-outline' });
        }
        if (elements?.overlay?.hasClass('is-loading')) {
            dropPrintUpdateOverlay('', '', '');
        }
        console.log('DEBUG: [Matching] About to trigger productsRenderedandMatched.dropPrint');
        $(document).trigger('productsRenderedandMatched.dropPrint');
    };
}(jQuery));