/**
 * Manages the primary feedback overlay panel in the Drop Print UI (v1.1).
 * This script provides functions to control the overlay's state (loading, error,
 * prompt), its content (title, message, icon), and the visibility of action
 * buttons. Version 1.1 enhances this by allowing the 'dropPrintUpdateOverlay'
 * function to dynamically render custom buttons with specific actions, in addition
 * to the pre-defined set. This is particularly used for dialogs like the image
 * resize method prompt, enabling more flexible user interactions.
 */
(function($) {
    'use strict';
    window.dropPrint = window.dropPrint || {};
    window.dropPrint.elements = window.dropPrint.elements || {};
    function dropPrintUpdateOverlay(state, title, message, options = {}) {
        const elements = window.dropPrint?.elements;
        if (!elements?.overlay?.length || !elements.overlayIconContainer?.length || !elements.overlayTitle?.length || !elements.overlayMessage?.length || !elements.overlayActions?.length) {
            if (state === 'is-error' && typeof message === 'string' && message) {
                alert(`DropPrint Error: ${title}\n${message}`);
            }
            return;
        }
        const { showActions = true, buttonsToShow = [], iconClass = '', customButtons = [] } = options;
        const overlayClasses = 'is-loading is-error is-prompt is-blocked is-visible';
        elements.overlay.removeClass(overlayClasses);
        if (state) {
            elements.overlay.addClass(state).addClass('is-visible').css('display', 'flex');
        } else {
            elements.overlay.css('display', 'none');
        }
        const $spinner = elements.overlaySpinner;
        const $dashicon = elements.overlayDashicon;
        if (state === 'is-loading') {
            $spinner?.addClass('is-active').show();
            $dashicon?.hide();
        } else {
            $spinner?.removeClass('is-active').hide();
            if (iconClass && $dashicon?.length) {
                $dashicon.attr('class', `dashicons ${iconClass}`).show();
            } else {
                $dashicon?.hide();
            }
        }
        elements.overlayTitle.text(title || '');
        elements.overlayMessage.html(message || '');
        const buttonMap = {
            reset: elements.dropPrintOverlayResetBtn,
            'confirm-reset': elements.dropPrintOverlayConfirmResetBtn,
            download: elements.dropPrintOverlayDownloadBtn,
            docs: elements.dropPrintOverlayDocsBtn,
            save: elements.dropPrintOverlaySaveBtn,
            save_now: elements.dropPrintOverlaySaveNowBtn,
            refresh: elements.dropPrintOverlayRefreshBtn,
            ok: elements.dropPrintOverlayOkBtn,
        };
        for (const key in buttonMap) {
            if (buttonMap[key] && buttonMap[key].length) {
                buttonMap[key].hide();
            }
        }
        elements.overlayActions.find('.dp-custom-button').remove();
        let hasVisibleButtons = false;
        if (Array.isArray(buttonsToShow) && buttonsToShow.length > 0) {
            buttonsToShow.forEach((buttonKey) => {
                if (buttonMap[buttonKey] && buttonMap[buttonKey].length) {
                    buttonMap[buttonKey].show();
                    hasVisibleButtons = true;
                }
            });
        }
        if (Array.isArray(customButtons) && customButtons.length > 0) {
            customButtons.forEach(customBtnConfig => {
                if (customBtnConfig && customBtnConfig.text && typeof customBtnConfig.action === 'function') {
                    const $button = $('<button></button>')
                        .attr('id', customBtnConfig.id || '')
                        .addClass('button dp-custom-button')
                        .addClass(customBtnConfig.class || 'button-secondary')
                        .text(customBtnConfig.text)
                        .on('click.dpCustomAction', function(e) {
                            e.preventDefault();
                            customBtnConfig.action.call(this, e);
                        });
                    elements.overlayActions.append($button);
                    hasVisibleButtons = true;
                }
            });
        }
        elements.overlayActions.toggle(showActions && hasVisibleButtons);
    }
    function dropPrintShowErrorOverlay(title, message, options = {}) {
        let defaultButtons = ['ok', 'reset', 'docs'];
        if (title && title.toLowerCase().includes('file too large')) {
            defaultButtons = ['reset', 'docs'];
        }
        const buttons = Array.isArray(options.buttonsToShow) ? options.buttonsToShow : defaultButtons;
        dropPrintUpdateOverlay('is-error', title, message, {
            showActions: true,
            iconClass: options.iconClass || 'dashicons-warning',
            ...options,
            buttonsToShow: buttons,
        });
    }
    function dropPrintShowPromptOverlay(title, message, options = {}) {
        const defaultButtons = ['ok', 'reset', 'docs'];
        const buttons = Array.isArray(options.buttonsToShow) ? options.buttonsToShow : defaultButtons;
        dropPrintUpdateOverlay('is-prompt', title, message, {
            showActions: true,
            iconClass: options.iconClass || 'dashicons-info-outline',
            ...options,
            buttonsToShow: buttons,
        });
    }
    window.dropPrint.dropPrintUpdateOverlay = dropPrintUpdateOverlay;
    window.dropPrint.dropPrintShowErrorOverlay = dropPrintShowErrorOverlay;
    window.dropPrint.dropPrintShowPromptOverlay = dropPrintShowPromptOverlay;
}(jQuery));