/**
 * DropPrint Client-Side Image Preview and Placeholder Generation.
 *
 * Provides functions to create either an optimized JPEG preview using Canvas
 * or an SVG placeholder indicating image dimensions.
 *
 * @package DropPrint\JS
 * @version 1.1
 */
( function( $ ) {
	'use strict';

	// Ensure the global namespace exists.
	window.dropPrint = window.dropPrint || {};

	/**
	 * Creates an optimized JPEG preview of an image file using Canvas.
	 * Limits preview dimensions and uses JPEG compression.
	 * Calls the callback with the data URL or null on failure.
	 *
	 * @param {File}     file     The image file object.
	 * @param {function} callback Function to call with the result (dataUrl|null).
	 */
	function dropPrintCreateOptimizedPreview( file, callback ) { // Renamed function
		const img = new Image();
		let objectUrl = null; // Keep track of the object URL for revocation.

		// Get cached preview element reference. Use renamed cache key.
		const previewElement = window.dropPrint?.elements?.dropPrintImagePreview;

		if ( ! previewElement ) {
			console.warn( 'DropPrint Preview: Preview element cache missing (elements.dropPrintImagePreview).' );
			callback( null );
			return;
		}

		// Ensure preview is hidden initially and reset any previous source.
		previewElement.classList.remove( 'visible' );
		previewElement.src = '';

		try {
			objectUrl = URL.createObjectURL( file );

			img.onload = function() {
				const canvas = document.createElement( 'canvas' );
				const ctx = canvas.getContext( '2d' );
				// Use camelCase config key.
				const maxDim = window.dropPrint?.config?.MAX_PREVIEW_DIMENSION || 800;
				const originalWidth = this.naturalWidth || this.width;
				const originalHeight = this.naturalHeight || this.height;

				// Validate image dimensions.
				if ( originalWidth <= 0 || originalHeight <= 0 ) {
					console.error( 'DropPrint Preview Error: Invalid image dimensions detected (<= 0).' );
					callback( null );
					// Revoke URL if created.
					if ( objectUrl ) {
						URL.revokeObjectURL( objectUrl );
						objectUrl = null;
					}
					return;
				}

				// Calculate scaling factor to fit within maxDim.
				const scale = Math.min( 1, maxDim / Math.max( originalWidth, originalHeight ) );
				canvas.width = Math.round( originalWidth * scale );
				canvas.height = Math.round( originalHeight * scale );

				// Draw image onto canvas.
				ctx.drawImage( img, 0, 0, canvas.width, canvas.height );

				// Generate JPEG data URL with moderate quality.
				const dataUrl = canvas.toDataURL( 'image/jpeg', 0.85 );
				callback( dataUrl ); // Pass generated preview URL to callback.

				// Revoke URL after use.
				if ( objectUrl ) {
					URL.revokeObjectURL( objectUrl );
					objectUrl = null;
				}
			}; // End img.onload

			img.onerror = function() {
				console.error( 'DropPrint Preview Error: Failed to load image object for preview generation.' );
				callback( null );
				// Revoke URL if created.
				if ( objectUrl ) {
					URL.revokeObjectURL( objectUrl );
					objectUrl = null;
				}
			}; // End img.onerror

			// Set image source to start loading.
			img.src = objectUrl;

		} catch ( e ) {
			console.error( 'DropPrint Preview Error: Exception during preview generation setup.', e );
			callback( null );
			// Ensure URL revocation on exception.
			if ( objectUrl ) {
				try {
					URL.revokeObjectURL( objectUrl );
				} catch ( revokeError ) { /* Ignore */ }
			}
		}
	} // End dropPrintCreateOptimizedPreview

	/**
	 * Creates an SVG data URL to act as a placeholder, displaying image dimensions.
	 * Used when client-side preview fails or is not attempted.
	 *
	 * @param {number} width  Original image width.
	 * @param {number} height Original image height.
	 */
	function dropPrintCreateDimensionPlaceholder( width, height ) { // Renamed function
		// Use renamed element cache key.
		const previewElement = window.dropPrint?.elements?.dropPrintImagePreview;

		if ( ! previewElement ) {
			console.warn( 'DropPrint Warning: Preview element cache missing, cannot display placeholder.' );
			return;
		}

		// Reset preview state.
		previewElement.classList.remove( 'visible' );
		previewElement.src = '';
		previewElement.alt = 'Preview'; // Reset alt text. i18n needed.

		// Validate dimensions provided for placeholder text.
		if ( ! width || ! height || width <= 0 || height <= 0 ) {
			console.warn( 'DropPrint Warning: Invalid dimensions provided for placeholder, displaying generic message.' );
			// Basic fallback SVG.
			previewElement.src = 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="400" height="300" viewBox="0 0 400 300"><rect width="100%" height="100%" fill="%23f0f0f0"/><text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle" font-family="sans-serif" font-size="14" fill="%23999">Preview not available</text></svg>'; // i18n needed.
			previewElement.style.display = 'block'; // Ensure SVG is displayed.
			previewElement.alt = 'Preview not available'; // i18n needed.
			return;
		}

		// Calculate display dimensions for the SVG itself (max 400px).
		const maxDisplayDim = 400;
		const scale = Math.min( 1, maxDisplayDim / Math.max( width, height ) );
		const displayWidth = Math.round( width * scale );
		const displayHeight = Math.round( height * scale );
		// Calculate font size relative to original dimensions for better scaling.
		const fontSize = Math.max( 10, Math.min( width, height ) * 0.05 ); // Ensure minimum size.
		// Construct SVG markup with dimensions text.
		const svgContent = `
			<svg xmlns="http://www.w3.org/2000/svg"
				 width="${ displayWidth }" height="${ displayHeight }"
				 viewBox="0 0 ${ width } ${ height }"
				 preserveAspectRatio="xMidYMid meet"
				 style="background-color: #f0f0f0; max-width: 100%; height: auto;">
				<rect width="100%" height="100%" fill="#f0f0f0"/>
				<text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle"
					  font-family="sans-serif" font-size="${ fontSize }px" fill="#999">
					${ width } x ${ height } px
				</text>
			</svg>`.replace( /\s\s+/g, ' ' ).replace( /#/g, '%23' ).replace( /</g, '%3C' ).replace( />/g, '%3E' ); // Basic encoding.

		const svgDataUrl = `data:image/svg+xml;utf8,${ svgContent }`;

		// Set SVG as source and make visible.
		previewElement.src = svgDataUrl;
		previewElement.alt = `Placeholder for ${ width }x${ height } image`; // i18n needed.
		previewElement.style.display = 'block'; // Ensure SVG placeholder is displayed.
		// Note: No 'visible' class for SVG placeholder, rely on display:block.

	} // End dropPrintCreateDimensionPlaceholder

	// Expose functions on the global namespace.
	window.dropPrint.dropPrintCreateOptimizedPreview = dropPrintCreateOptimizedPreview; // Renamed function
	window.dropPrint.dropPrintCreateDimensionPlaceholder = dropPrintCreateDimensionPlaceholder; // Renamed function

}( jQuery ) ); // Note: jQuery wrapper isn't strictly necessary here as no jQuery is used, but kept for consistency.