/**
 * DropPrint File Handling After Dimensions Are Known.
 *
 * Defines the logic that runs once image dimensions have been determined.
 * Waits for the pricelist JSON to load, then initiates product matching
 * with detailed image dimensions.
 * @package DropPrint\JS
 * @version 1.3
 */
(function($){
'use strict';
window.dropPrint=window.dropPrint||{};
window.dropPrint.dropPrintHandleFilePostDimensions=function(file){
    // Get dependencies, ensuring checks happen *before* destructuring potentially undefined functions
    const {state,elements,dropPrintUpdateOverlay}=window.dropPrint||{};
    const dropPrintInitiateMatching = window.dropPrint?.dropPrintInitiateMatching; // Get reference separately

    // Check for essential functions needed *by this function*
    if(typeof dropPrintUpdateOverlay!=='function'){
        console.error('DropPrint Error (handleFilePostDimensions): dropPrintUpdateOverlay missing!');
        alert('DropPrint Critical Error: UI feedback system unavailable.');
        return;
    }
    if(typeof dropPrintInitiateMatching!=='function'){
        console.error('DropPrint Error (handleFilePostDimensions): dropPrintInitiateMatching function not defined when needed.');
        dropPrintUpdateOverlay('is-error','Initialization Error','Product matching component failed or not loaded. Please reload.',{showActions:true,iconClass:'dashicons-dismiss',buttonsToShow:['reset']});
        return;
    }

    // Check for valid dimensions passed in
    const hasValidFileDimensions=file&&state?.fileDimensions&&
        typeof state.fileDimensions.width==='number'&&typeof state.fileDimensions.height==='number'&&
        state.fileDimensions.width>0&&state.fileDimensions.height>0;
    const hasValidCachedDimensions=file&&file._cachedDimensions&&
        typeof file._cachedDimensions.longestEdge==='number'&&typeof file._cachedDimensions.shortestEdge==='number'&&
        file._cachedDimensions.longestEdge>0&&file._cachedDimensions.shortestEdge>0;

    if(!hasValidFileDimensions||!hasValidCachedDimensions){
        console.error('DropPrint Error: handleFilePostDimensions called without valid dimensions.',file?._cachedDimensions,state?.fileDimensions);
        dropPrintUpdateOverlay('is-error','Processing Error','Could not retrieve valid image dimensions. Please reset.',{showActions:true,iconClass:'dashicons-warning',buttonsToShow:['ok','reset']});
        return;
    }

    const {longestEdge,shortestEdge}=file._cachedDimensions;
    const {width:imgPixelWidth,height:imgPixelHeight}=state.fileDimensions;

    // Wait for JSON promise
    const jsonPromise=window.dropPrint?.state?.jsonLoadingPromise;
    if(jsonPromise&&typeof jsonPromise.always==='function'){
        if(jsonPromise.state()==='pending'){
            dropPrintUpdateOverlay('is-loading','Loading Products','Please wait...',{showActions:false});
        }
        jsonPromise.always(function(){
            if(window.dropPrint?.printSizesData){
                const $overlayTitle=$('#drop-print-panel-overlay .drop-print-overlay-title');
                if($('#drop-print-panel-overlay').hasClass('is-loading')&&$overlayTitle.length&&$overlayTitle.text()==='Loading Products'){
                    dropPrintUpdateOverlay('','','');
                }
                // Call the matching function (already confirmed to exist)
                dropPrintInitiateMatching(longestEdge,shortestEdge,imgPixelWidth,imgPixelHeight);
            }else{
                // JSON promise finished, but data is missing/invalid.
                if(elements?.overlay&&!elements.overlay.hasClass('is-error')){
                    dropPrintUpdateOverlay('is-error','Data Error','Product data failed to load correctly. Cannot match products.',{showActions:true,iconClass:'dashicons-warning',buttonsToShow:['ok','reset','refresh']});
                }
            }
        });
    }else{
        // JSON loading promise itself was missing or invalid.
        console.error('DropPrint (handleFilePostDimensions): JSON loading promise not found or invalid. Cannot initiate matching.');
        dropPrintUpdateOverlay('is-error','Initialization Error','Product data loading failed earlier or was skipped. Cannot match products.',{showActions:true,iconClass:'dashicons-dismiss',buttonsToShow:['reset']});
    }
};
}(jQuery));