/**
 * DropPrint Dropzone Initialization and Event Handlers (v1.1.2).
 * This script initializes the Dropzone.js instance for file uploads.
 * Version 1.1 enhances user feedback during uploads by integrating an animated
 * progress bar. The 'processing', 'uploadprogress', 'success', and 'error'
 * event handlers have been modified to show, update, and hide a custom
 * progress bar displayed within the main feedback overlay, providing users with
 * clearer visual indication of the upload status for large files. Core Dropzone
 * settings for chunking, URL, and parameters remain, ensuring robust file transfer.
 */
( function( $ ) {
	'use strict';
	window.dropPrint = window.dropPrint || {};
	function dropPrintInitDropzone() {
		const currentDropPrintState = window.dropPrint || {};
		const { elements, state, config, dropPrintUpdateOverlay, dropPrintReset, dropPrintHandleFileAdded, dropPrintShowErrorOverlay } = currentDropPrintState;
		const dropzoneElement = elements?.dropzone;
		if ( typeof Dropzone === 'undefined' ) {
			if ( typeof dropPrintShowErrorOverlay === 'function' ) {
				dropPrintShowErrorOverlay( 'Initialization Error', 'Core upload library (Dropzone) failed to load.' );
			} else {
				alert( 'DropPrint Critical Error: Upload library failed to load.' );
			}
			return;
		}
		if ( !dropzoneElement || !config?.ajaxUrl || !config?.nonce || !config?.productId ||
		     typeof dropPrintUpdateOverlay !== 'function' ||
		     typeof dropPrintReset !== 'function' ||
		     typeof dropPrintHandleFileAdded !== 'function' ||
		     typeof dropPrintShowErrorOverlay !== 'function' ) {
			const errorMsg = 'DropPrint Critical Error: Core upload components failed. Please reload.';
			if ( typeof dropPrintUpdateOverlay === 'function' ) {
				dropPrintUpdateOverlay( 'is-error', 'Initialization Error', 'Core upload components failed. Please reload.', { showActions: true, buttonsToShow: [ 'reset' ] } );
			} else {
				alert( errorMsg );
			}
			return;
		}
		if ( state.dropzoneInstance ) {
			return;
		}
		try {
			state.dropzoneInstance = new Dropzone( dropzoneElement, {
				url: config.ajaxUrl,
				paramName: 'file',
				maxFiles: 1,
				acceptedFiles: 'image/jpeg,image/png,image/tiff',
				maxFilesize: config.maxUploadFileSizeMB || 256,
				addRemoveLinks: false,
				autoProcessQueue: false,
				clickable: dropzoneElement,
				chunking: true,
				forceChunking: true,
				chunkSize: 2 * 1024 * 1024,
				retryChunks: true,
				retryChunksLimit: 3,
				previewsContainer: false,
				previewTemplate: '<div class="dz-preview dz-file-preview"></div>',
				dictDefaultMessage: '',
				params: function( files, xhr, chunk ) {
					const currentConfig = window.dropPrint?.config;
					const currentNonce = currentConfig?.nonce;
					const currentProductId = currentConfig?.productId;
					if ( ! currentNonce || ! currentProductId ) {
						return { action: 'drop_print_process_upload', nonce: 'error_config_missing', product_id: 0, };
					}
					const baseParams = { action: 'drop_print_process_upload', nonce: currentNonce, product_id: currentProductId, };
					if ( chunk ) {
						return {
							...baseParams,
							dzuuid: chunk.file.upload.uuid,
							dzchunkindex: chunk.index,
							dztotalchunkcount: chunk.file.upload.totalChunkCount,
							dzfilename: chunk.file.name,
							dztotalfilesize: chunk.file.size,
							dzchunksize: this.options.chunkSize,
						};
					}
					return baseParams;
				},
				init: function() {
					const dz = this;
					dz.on( 'addedfile', ( file ) => {
						if ( dz.files.length > 1 ) { dz.removeFile( dz.files[ 0 ] ); }
						if ( typeof window.dropPrint.dropPrintHandleFileAdded === 'function' ) {
							window.dropPrint.dropPrintHandleFileAdded( file );
						} else {
							if(typeof window.dropPrint.dropPrintShowErrorOverlay === 'function') window.dropPrint.dropPrintShowErrorOverlay( 'Initialization Error', 'Core file handling script missing.' );
						}
					} );
					dz.on( 'processing', () => {
						if(typeof window.dropPrint.dropPrintUpdateOverlay === 'function') window.dropPrint.dropPrintUpdateOverlay( 'is-loading', 'Uploading File', '' );
						$('#dp-upload-progress-container').show();
						$('#dp-upload-progress-bar-inner').css('width', '0%');
						$('#dp-upload-progress-text').text('Starting upload...');
						if ( elements.dropPrintAddVariationsButton ) {
							$(elements.dropPrintAddVariationsButton).prop( 'disabled', true );
						}
						if ( elements.dropPrintResetButton ) {
							$(elements.dropPrintResetButton).prop( 'disabled', true );
						}
					} );
					dz.on( 'uploadprogress', ( file, progress ) => {
						const percent = Math.round( progress );
						$('#dp-upload-progress-bar-inner').css('width', percent + '%');
						let progressText = `Uploading ${percent}%...`;
						if (percent >= 100) {
							progressText = 'Upload complete. Server processing...';
						}
						$('#dp-upload-progress-text').text(progressText);
					} );
					dz.on( 'success', ( file, response_arg ) => {
						$('#dp-upload-progress-container').hide();
						const serverResponseText = file?.xhr?.responseText;
						let parsedResponse = null;
						try {
							if ( ! serverResponseText ) { throw new Error( 'No response text received from server.' ); }
							parsedResponse = JSON.parse( serverResponseText );
						} catch ( e ) {
							if(typeof window.dropPrint.dropPrintShowErrorOverlay === 'function') window.dropPrint.dropPrintShowErrorOverlay( 'Processing Error', 'Invalid server response format.' );
							if ( dz.files.includes( file ) ) { dz.removeFile( file ); }
							if ( typeof window.dropPrint.dropPrintReset === 'function' ) { window.dropPrint.dropPrintReset(); }
							return;
						}
						try {
							if ( parsedResponse?.success === true && parsedResponse?.data?.thumbnail_id && parsedResponse?.data?.afp_base_filename && parsedResponse?.data?.width && parsedResponse?.data?.height ) {
								const responseData = parsedResponse.data;
								if ( state ) {
									const serverWidth = parseInt( responseData.width, 10 ) || 0;
									const serverHeight = parseInt( responseData.height, 10 ) || 0;
									state.metaPreviewId = responseData.thumbnail_id;
									state.metaFileName = responseData.afp_base_filename;
									state.attachmentUrl = responseData.attachment_url || null;
									state.fileDimensions = { width: serverWidth, height: serverHeight };
									state.dimensionsKnown = true;
									if ( state.currentFile ) {
										state.currentFile._cachedDimensions = { longestEdge: Math.max( serverWidth, serverHeight ), shortestEdge: Math.min( serverWidth, serverHeight ) };
									}
									state.prospectiveVariations = null;
									state.variationsStagedNotSaved = false;
								}
								if ( typeof window.dropPrint.dropPrintUpdateFileInfo === 'function' ) {
									window.dropPrint.dropPrintUpdateFileInfo( state.fileDimensions.width, state.fileDimensions.height );
								}
								const previewElement = document.getElementById('drop-print-image-preview');
								if ( previewElement && state.attachmentUrl ) {
									previewElement.src = state.attachmentUrl;
									previewElement.style.display = 'block';
									previewElement.onload = () => { previewElement.classList.add( 'visible' ); };
									previewElement.onerror = () => {
										previewElement.classList.remove( 'visible' );
										if ( typeof window.dropPrint.createDimensionPlaceholder === 'function' && state.fileDimensions?.width ) {
											window.dropPrint.createDimensionPlaceholder( state.fileDimensions.width, state.fileDimensions.height );
										}
									};
									if ( previewElement.complete ) { previewElement.classList.add( 'visible' ); }
								} else if ( typeof window.dropPrint.createDimensionPlaceholder === 'function' && state.fileDimensions?.width ) {
									window.dropPrint.createDimensionPlaceholder( state.fileDimensions.width, state.fileDimensions.height );
								}
								if ( state.uploadCompleteForDimensions ) {
									if ( typeof window.dropPrint.dropPrintHandleFilePostDimensions === 'function' ) {
										window.dropPrint.dropPrintHandleFilePostDimensions( state.currentFile );
									} else {
										if(typeof window.dropPrint.dropPrintShowErrorOverlay === 'function') window.dropPrint.dropPrintShowErrorOverlay( 'Initialization Error', 'Matching script missing.' );
									}
									if ( $( '#drop-print-panel-overlay' ).hasClass( 'is-loading' ) ) {
										if(typeof window.dropPrint.dropPrintUpdateOverlay === 'function') window.dropPrint.dropPrintUpdateOverlay( '', '', '' );
									}
								} else {
									if(typeof window.dropPrint.dropPrintUpdateOverlay === 'function') window.dropPrint.dropPrintUpdateOverlay(
										'is-prompt', 'Product Save Required',
										'<p>Image file was uploaded securely and successfully.</p><p>Product variations will now be created. However, you need to save the product to see these changes in effect.</p><p>Click <strong>\'Save Now\'</strong> or use the \'Publish\' button above.</p>',
										{ showActions: true, iconClass: 'dashicons-yes-alt', buttonsToShow: [ 'ok', 'reset', 'save_now' ] }
									);
									if ( elements.dropPrintAddVariationsButton ) {
										$(elements.dropPrintAddVariationsButton).prop( 'disabled', false );
									}
								}
							} else {
								const errorMsg = parsedResponse?.data?.message || parsedResponse?.message || 'Server file processing failed.';
								if(typeof window.dropPrint.dropPrintShowErrorOverlay === 'function') window.dropPrint.dropPrintShowErrorOverlay( 'File Processing Failed', errorMsg );
								if ( dz.files.includes( file ) ) { dz.removeFile( file );}
								if ( typeof window.dropPrint.dropPrintReset === 'function' ) {window.dropPrint.dropPrintReset();}
							}
						} catch ( e ) {
							if(typeof window.dropPrint.dropPrintShowErrorOverlay === 'function') window.dropPrint.dropPrintShowErrorOverlay( 'Internal Client Error', 'Error processing server response.' );
							if ( dz.files.includes( file ) ) { dz.removeFile( file );}
							if ( typeof window.dropPrint.dropPrintReset === 'function' ) { window.dropPrint.dropPrintReset(); }
						}
					} );
					dz.on( 'error', ( file, errorMessage, xhr ) => {
						$('#dp-upload-progress-container').hide();
						let displayMessage = 'File upload failed.';
						if ( xhr?.responseText ) {
							try {
								const parsedError = JSON.parse( xhr.responseText );
								displayMessage = parsedError?.data?.message || parsedError?.message || displayMessage;
							} catch ( e ) { }
						} else if ( typeof errorMessage === 'string' && errorMessage.length > 0 && ! errorMessage.toLowerCase().includes( 'error' ) ) {
							displayMessage = errorMessage;
						} else if ( typeof errorMessage === 'object' && errorMessage !== null ) {
							displayMessage = errorMessage.message || JSON.stringify( errorMessage );
						}
						if(typeof window.dropPrint.dropPrintShowErrorOverlay === 'function') window.dropPrint.dropPrintShowErrorOverlay( 'Upload Failed', displayMessage, { buttonsToShow: [ 'reset', 'docs' ], iconClass: 'dashicons-no-alt' } );
						if ( dz.files.includes( file ) ) { dz.removeFile( file ); }
						if ( elements.dropPrintResetButton ) { $(elements.dropPrintResetButton).prop( 'disabled', false ); }
						if ( elements.dropPrintAddVariationsButton ) {
						    if(typeof window.dropPrint.dropPrintUpdateButtonState === 'function') window.dropPrint.dropPrintUpdateButtonState(); else $(elements.dropPrintAddVariationsButton).prop('disabled', false);
                        }
					} );
					dz.on( 'queuecomplete', () => {
						$('#dp-upload-progress-container').hide();
						if ( elements.overlay && ! elements.overlay.hasClass( 'is-loading' ) && ! elements.overlay.hasClass( 'is-error' ) && ! elements.overlay.hasClass( 'is-prompt' ) ) {
							if ( elements.dropPrintResetButton ) { $(elements.dropPrintResetButton).prop( 'disabled', false );}
							if ( typeof window.dropPrint.dropPrintUpdateButtonState === 'function' ) {
								window.dropPrint.dropPrintUpdateButtonState();
							}
						}
					} );
				},
			} );
		} catch ( e ) {
			if ( elements?.dropzoneWrapper ) {
				elements.dropzoneWrapper.innerHTML = '<p class="drop-print-error">Fatal Error: Upload component failed to initialize. Please reload the page.</p>';
			} else {
				alert( 'DropPrint Critical Error: Upload component failed to initialize. Please reload the page.' );
			}
			if ( state ) { state.dropzoneInstance = null; }
		}
	}
	window.dropPrint.dropPrintInitDropzone = dropPrintInitDropzone;
}( jQuery ) );