/**
 * Core UI management for the Drop Print panel (v1.1.2).
 * Initializes UI elements, manages panel state, tab visibility, overlays, product
 * list interactions, and the dynamic "Show All Options" trigger. Key features
 * include intelligent media dropdown behavior (auto-selects based on existing
 * variations), synchronization of prices from the WC Variations tab, display of
 * existing Drop Print thumbnails in the main panel, robust auto-selection of
 * previously saved variations on page load, and refined product search
 * functionality that searches the full catalog when a term is entered,
 * reverting to the toggle's state when the search is cleared.
 */
(function($) {
    'use strict';
    window.dropPrint = window.dropPrint || {};
    window.dropPrint.elements = window.dropPrint.elements || {};
    let showAllTriggerScrollHandler;
    function initializeDropPrintElements() {
        window.dropPrint.elements = {
            ...(window.dropPrint.elements || {}),
            product_data: document.getElementById('woocommerce-product-data'),
            dropPrintPanel: $('#drop-print-product-data-panel'),
            dropPrintTabLink: $('li.drop-print-tab'),
            dropPrintEnableCheckbox: $('input[name="_drop_print_enable"]'),
            dropzoneWrapper: document.getElementById('drop-print-dropzone-wrapper'),
            dropzone: document.getElementById('drop-print-dropzone'),
            prompt: document.getElementById('drop-print-prompt'),
            dropPrintStep: document.getElementById('drop-print-step-1'),
            previewColumn: document.getElementById('drop-print-preview-col'),
            dropPrintImageWrapper: document.getElementById('drop-print-image-wrapper'),
            dropPrintImagePreview: document.getElementById('drop-print-image-preview'),
            dropPrintFileInfo: {
                name: document.getElementById('drop-print-file-name'),
                widthDisplay: document.getElementById('drop-print-preview-dimensions-width'),
                heightDisplay: document.getElementById('drop-print-preview-dimensions-height')
            },
            productsColumn: document.getElementById('drop-print-products-col'),
            mediaSelectContainer: $('#drop-print-media-select-container'),
            mediaSelect: $('#drop-print-media-select'),
            productSearchInput: $('#drop-print-product-search'),
            showAllTrigger: $('#drop-print-show-all-trigger'),
            productListContainer: $('#drop-print-product-list-container'),
            dropPrintResetButton: document.getElementById('drop-print-reset-button'),
            dropPrintAddVariationsButton: document.getElementById('drop-print-add-variations-button'),
            overlay: $('#drop-print-panel-overlay'),
            overlayContent: $('#drop-print-panel-overlay .drop-print-overlay-content'),
            overlayIconContainer: $('#drop-print-panel-overlay .drop-print-overlay-icon'),
            overlaySpinner: $('#drop-print-panel-overlay .spinner'),
            overlayDashicon: $('#drop-print-panel-overlay .dashicons'),
            overlayTitle: $('#drop-print-panel-overlay .drop-print-overlay-title'),
            overlayMessage: $('#drop-print-panel-overlay .drop-print-overlay-message'),
            overlayActions: $('#drop-print-panel-overlay .drop-print-overlay-actions'),
            dropPrintOverlayResetBtn: $('#drop-print-overlay-reset-btn'),
            dropPrintOverlayConfirmResetBtn: $('#drop-print-overlay-confirm-reset-btn'),
            dropPrintOverlayRefreshBtn: $('#drop-print-overlay-refresh-btn'),
            dropPrintOverlayDownloadBtn: $('#drop-print-overlay-download-btn'),
            dropPrintOverlayDocsBtn: $('#drop-print-overlay-docs-btn'),
            dropPrintOverlaySaveBtn: $('#drop-print-overlay-save-btn'),
            dropPrintOverlaySaveNowBtn: $('#drop-print-overlay-save-now-btn'),
            dropPrintOverlayOkBtn: $('#drop-print-overlay-ok-btn'),
        };
    }
    window.dropPrint.initializeDropPrintElements = initializeDropPrintElements;
    function updateShowAllTriggerVisibility() {
        const { elements, state } = window.dropPrint;
        if (!elements.showAllTrigger || !elements.showAllTrigger.length || !elements.productListContainer || !elements.productListContainer.length || !state) {
            return;
        }
        if (elements.showAllTrigger.attr('aria-disabled') === 'true') {
            elements.showAllTrigger.removeClass('visible is-sticky-top').css('opacity', 0.7).css('pointer-events', 'none');
            return;
        }
        if (state.showingAllProducts) {
            elements.showAllTrigger.addClass('is-sticky-top').removeClass('visible');
            elements.showAllTrigger.css('opacity', 1).css('pointer-events', 'auto');
        } else {
            elements.showAllTrigger.removeClass('is-sticky-top');
            const container = elements.productListContainer[0];
            const threshold = 30;
            const isNearBottom = (container.scrollHeight - container.scrollTop - container.clientHeight) < threshold;
            const isNotScrollable = container.scrollHeight <= container.clientHeight;
            if (isNearBottom || isNotScrollable) {
                elements.showAllTrigger.addClass('visible').css('opacity', 1).css('pointer-events', 'auto');
            } else {
                elements.showAllTrigger.removeClass('visible').css('opacity', 0).css('pointer-events', 'none');
            }
        }
    }
    window.dropPrint.updateShowAllTriggerVisibility = updateShowAllTriggerVisibility;
    function dropPrintUpdateFileInfo(width, height) {
        const { elements, state } = window.dropPrint || {};
        const fileInfoElems = elements?.dropPrintFileInfo;
        if (!elements?.dropPrintPanel?.length) return;
        const $fileInfoWrapper = $('#drop-print-file-info-wrapper');
        let displayName = '', displayWidth = width, displayHeight = height;
        if (state?.currentFile?.name) {
            displayName = state.currentFile.name;
            displayWidth = (width > 0) ? width : null;
            displayHeight = (height > 0) ? height : null;
        } else if (state?.metaActiveDimensions?.width && state?.metaActiveDimensions?.height) {
            displayName = state.metaFileName || 'Existing File Dimensions';
            displayWidth = state.metaActiveDimensions.width;
            displayHeight = state.metaActiveDimensions.height;
        }
        if (displayName) {
            $fileInfoWrapper.stop().fadeIn(150);
            if (fileInfoElems?.name) fileInfoElems.name.textContent = displayName;
            if (fileInfoElems?.widthDisplay) fileInfoElems.widthDisplay.textContent = displayWidth > 0 ? `${displayWidth}px` : '';
            if (fileInfoElems?.heightDisplay) fileInfoElems.heightDisplay.textContent = displayHeight > 0 ? `${displayHeight}px` : '';
            $('#drop-print-dimension-width').toggleClass('visible', !!(displayWidth && displayWidth > 0));
            $('#drop-print-dimension-height').toggleClass('visible', !!(displayHeight && displayHeight > 0));
        } else {
            $fileInfoWrapper.stop().fadeOut(150);
            if (fileInfoElems?.name) fileInfoElems.name.textContent = '';
            if (fileInfoElems?.widthDisplay) fileInfoElems.widthDisplay.textContent = '';
            if (fileInfoElems?.heightDisplay) fileInfoElems.heightDisplay.textContent = '';
            $('#drop-print-dimension-width, #drop-print-dimension-height').removeClass('visible');
        }
    }
    window.dropPrint.dropPrintUpdateFileInfo = dropPrintUpdateFileInfo;
    function dropPrintToggleTab(isActive, isInitialLoad = false) {
        const { elements } = window.dropPrint || {};
        if (!elements?.dropPrintPanel?.length || !elements?.dropPrintTabLink?.length || !elements?.dropPrintEnableCheckbox?.length) {
            return;
        }
        elements.dropPrintEnableCheckbox.prop('checked', isActive);
        elements.dropPrintTabLink.toggle(isActive);
        elements.dropPrintPanel.toggle(isActive);
        if (isActive) {
            const $productType = $('#product-type');
            if ($productType.length && $productType.val() !== 'variable') {
                $productType.val('variable').trigger('change');
            }
            if ((!isInitialLoad || !elements.dropPrintTabLink.hasClass('active')) && elements.dropPrintTabLink.is(':visible')) {
                elements.dropPrintTabLink.find('a[href="#drop-print-product-data-panel"]').trigger('click');
            }
        } else if (elements.dropPrintTabLink.hasClass('active')) {
            $('ul.product_data_tabs a[href="#general_product_data"]').trigger('click');
        }
    }
    window.dropPrint.dropPrintToggleTab = dropPrintToggleTab;
    function dropPrintPerformClientReset() {
        const { state, elements, config, dropPrintUpdateOverlay, dropPrintInitDropzone, dropPrintLoadPricelistFromJson, dropPrintUpdateTabAppearance } = window.dropPrint || {};
        if (!state || !elements || !config || typeof dropPrintUpdateOverlay !== 'function' || typeof dropPrintInitDropzone !== 'function' || typeof dropPrintLoadPricelistFromJson !== 'function' || typeof dropPrintUpdateTabAppearance !== 'function') {
            return;
        }
        const previewIdBeforeReset = state.metaPreviewId;
        if (state.currentFile?.preview && typeof URL.revokeObjectURL === 'function') { try { URL.revokeObjectURL(state.currentFile.preview); } catch (err) {} }
        Object.assign(state, { currentFile: null, currentBestMatch: null, dimensionsKnown: false, uploadCompleteForDimensions: false, fileDimensions: null, metaFileName: null, metaPreviewId: null, metaActiveDimensions: null, metaActiveVariations: [], attachmentUrl: null, prospectiveVariations: null, variationsStagedNotSaved: false, dropPrintRequiresSave: false, currentMediaId: config?.defaultMediaId || null, showingAllProducts: false, scoredProducts: null, metaDoingReset: false, printSizesData: null, jsonLoadingPromise: null });
        window.dropPrint.printSizesData = null;
        if (elements.dropPrintStep) $(elements.dropPrintStep).removeClass('active');
        if (elements.dropzoneWrapper) $(elements.dropzoneWrapper).show();
        if (elements.previewColumn) $(elements.previewColumn).hide();
        if (elements.dropPrintImageWrapper) elements.dropPrintImageWrapper.style.display = 'none';
        if (elements.dropPrintImagePreview) { elements.dropPrintImagePreview.src = ''; elements.dropPrintImagePreview.style.display = 'none'; elements.dropPrintImagePreview.classList.remove('visible'); }
        if (elements.productsColumn) $(elements.productsColumn).removeClass('active');
        if (typeof window.dropPrint.dropPrintUpdateFileInfo === 'function') window.dropPrint.dropPrintUpdateFileInfo(null, null);
        if (elements.productListContainer?.length) elements.productListContainer.html('<p class="drop-print-product-list-loading">Please select a media type to see products.</p>').scrollTop(0);
        if (previewIdBeforeReset && typeof $ === 'function') $('#drop-print-preview-container').empty().append('<p>No Drop Print preview image set.</p>');
        if (elements.dropPrintResetButton) $(elements.dropPrintResetButton).prop('disabled', true);
        if (elements.dropPrintAddVariationsButton) $(elements.dropPrintAddVariationsButton).prop('disabled', true).text('Create Product Variations');
        if (elements.showAllTrigger && elements.showAllTrigger.length) {
            elements.showAllTrigger.attr('aria-disabled', 'true').addClass('disabled');
            elements.showAllTrigger.find('.drop-print-show-all-text').text('Show All Options');
            elements.showAllTrigger.find('.drop-print-show-all-indicator').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
            elements.showAllTrigger.attr('aria-expanded', 'false').removeClass('is-sticky-top visible');
        }
        if (elements.productSearchInput) elements.productSearchInput.val('');
        if (elements.mediaSelect) {
            elements.mediaSelect.prop('disabled', false);
            let defaultMediaToSet = config?.defaultMediaId || null;
            if (config?.availableMedia && config.availableMedia.length > 0 && !elements.mediaSelect.find('option[value="' + defaultMediaToSet + '"]').length) {
                defaultMediaToSet = config.availableMedia[0].id;
            }
            elements.mediaSelect.val(defaultMediaToSet || '');
            state.currentMediaId = elements.mediaSelect.val();
        }
        if (state.dropzoneInstance) { try { state.dropzoneInstance.removeAllFiles(true); state.dropzoneInstance.enable(); } catch (err) { try { state.dropzoneInstance.destroy(); } catch (destroyErr) {} state.dropzoneInstance = null; if(typeof dropPrintInitDropzone === 'function') dropPrintInitDropzone(); } } else if ($(elements.dropzoneWrapper).is(':visible') && typeof dropPrintInitDropzone === 'function') { dropPrintInitDropzone(); }
        if(typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('', '', '');
        if(typeof dropPrintUpdateTabAppearance === 'function') dropPrintUpdateTabAppearance(false);
        if (state.currentMediaId) {
            if(typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('is-loading', 'Loading Products', 'Reloading default product list...', { showActions: false });
            state.jsonLoadingPromise = dropPrintLoadPricelistFromJson(state.currentMediaId);
            state.jsonLoadingPromise.always(function() {
                if (elements?.overlay?.hasClass('is-loading') && typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('', '', '');
                if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', String(!window.dropPrint.printSizesData)).toggleClass('disabled', !window.dropPrint.printSizesData);
                if (typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
            });
        } else {
            state.jsonLoadingPromise = $.Deferred().resolve().promise();
            if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', 'true').addClass('disabled');
            if (typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
        }
    }
    window.dropPrint.dropPrintPerformClientReset = dropPrintPerformClientReset;
    function dropPrintShowResetConfirmation(e) {
        if (e) { e.preventDefault(); e.stopPropagation(); }
        const { dropPrintUpdateOverlay } = window.dropPrint || {};
        if (typeof dropPrintUpdateOverlay !== 'function') {
            if (window.confirm('Are you sure you want to reset?\n\nThis will clear the current file selection and any staged variations.\nAssociated server files may also be deleted when you save the product.')) {
            }
            return;
        }
        dropPrintUpdateOverlay('is-prompt', 'Confirm Reset', '<p>Are you sure you want to reset Drop Print for this product?</p><p><small>This will clear the current file selection and any staged variations. Associated server files may also be deleted when you next save the product.</small></p>', { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['confirm-reset', 'ok'] });
    }
    window.dropPrint.dropPrintReset = dropPrintShowResetConfirmation;
    function dropPrintExecuteReset() {
        const { state, config, dropPrintUpdateOverlay, dropPrintShowErrorOverlay, dropPrintPerformClientReset } = window.dropPrint || {};
        if (!state || !config || typeof dropPrintUpdateOverlay !== 'function' || typeof dropPrintPerformClientReset !== 'function' || typeof dropPrintShowErrorOverlay !== 'function') {
            return;
        }
        dropPrintUpdateOverlay('is-loading', 'Resetting', 'Staging reset and removing assets...');
        $.ajax({ url: config.ajaxUrl, type: 'POST', dataType: 'json', data: { action: 'drop_print_delete_assets', nonce: config.nonce, afp_base_filename: state.metaFileName || '', thumbnail_id: state.metaPreviewId || 0, product_id: config.productId } })
            .done(response => {
                if (response?.success) dropPrintPerformClientReset();
                else dropPrintShowErrorOverlay('Reset Failed', `${response?.data?.message || 'Failed to stage reset on server.'} Please try again or check server logs.`, { buttonsToShow: ['ok'] });
            })
            .fail((jqXHR, textStatus, errorThrown) => {
                let errorMsg = 'Could not communicate with server to stage the reset.'; try { errorMsg = JSON.parse(jqXHR.responseText)?.data?.message || errorMsg; } catch (e) {}
                if (jqXHR.status === 500) errorMsg += ' (500 Internal Server Error - Check PHP logs)';
                dropPrintShowErrorOverlay('Reset Request Failed', `${errorMsg} Please try again.`, { buttonsToShow: ['ok'] });
            });
    }
    window.dropPrint.dropPrintExecuteReset = dropPrintExecuteReset;
    function dropPrintStartExistingMetaWorkflow() {
        const { state, elements, config, dropPrintUpdateOverlay, dropPrintLoadPricelistFromJson, dropPrintCreateDimensionPlaceholder, dropPrintInitiateMatching, dropPrintShowErrorOverlay } = window.dropPrint || {};
        if (typeof dropPrintLoadPricelistFromJson !== 'function' || typeof dropPrintCreateDimensionPlaceholder !== 'function' || typeof dropPrintInitiateMatching !== 'function' || typeof dropPrintUpdateOverlay !== 'function' || typeof dropPrintShowErrorOverlay !== 'function' || typeof window.dropPrint.dropPrintUpdateFileInfo !== 'function') {
            return (typeof dropPrintShowErrorOverlay === 'function') ? dropPrintShowErrorOverlay('Initialization Error', 'Core components missing for existing meta workflow. Please reload.') : alert('Init Error for existing meta.');
        }
        if (elements.dropzoneWrapper) $(elements.dropzoneWrapper).hide();
        if (elements.dropPrintStep) $(elements.dropPrintStep).addClass('active');
        if (elements.previewColumn) $(elements.previewColumn).show();
        if (elements.dropPrintImageWrapper) elements.dropPrintImageWrapper.style.display = 'block';
        if (elements.productsColumn) $(elements.productsColumn).addClass('active');
        $(elements.dropPrintResetButton).prop('disabled', false); $(elements.dropPrintAddVariationsButton).prop('disabled', true);
        if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', 'true').addClass('disabled');
        if (state.metaPreviewId && config.metaPreviewImageUrl) {
            elements.dropPrintImagePreview.src = config.metaPreviewImageUrl;
            elements.dropPrintImagePreview.style.display = 'block';
            elements.dropPrintImagePreview.onload = () => { elements.dropPrintImagePreview.classList.add('visible'); };
            elements.dropPrintImagePreview.onerror = () => {
                elements.dropPrintImagePreview.style.display = 'none';
                elements.dropPrintImagePreview.classList.remove('visible');
                if (state.metaActiveDimensions && typeof dropPrintCreateDimensionPlaceholder === 'function') {
                    dropPrintCreateDimensionPlaceholder(state.metaActiveDimensions.width, state.metaActiveDimensions.height);
                }
            };
            if (elements.dropPrintImagePreview.complete && elements.dropPrintImagePreview.naturalWidth > 0) {
                elements.dropPrintImagePreview.classList.add('visible');
            }
        } else if (state.metaActiveDimensions && typeof dropPrintCreateDimensionPlaceholder === 'function') {
            dropPrintCreateDimensionPlaceholder(state.metaActiveDimensions.width, state.metaActiveDimensions.height);
        }
        const width = parseInt(state.metaActiveDimensions?.width, 10), height = parseInt(state.metaActiveDimensions?.height, 10);
        if (isNaN(width) || isNaN(height) || width <= 0 || height <= 0) {
            if (typeof dropPrintShowErrorOverlay === 'function') dropPrintShowErrorOverlay('Data Error', 'Invalid existing dimensions found in product meta. Cannot proceed with matching.');
            return;
        }
        window.dropPrint.dropPrintUpdateFileInfo(width, height);
        dropPrintUpdateOverlay('is-loading', 'Loading Products', 'Please wait...');
        const mediaToLoad = state.currentMediaId || config.defaultMediaId;
        if (!mediaToLoad) {
             if(typeof dropPrintShowErrorOverlay === 'function') dropPrintShowErrorOverlay('Configuration Error', 'No media type specified for loading products.');
            return;
        }
        state.jsonLoadingPromise = dropPrintLoadPricelistFromJson(mediaToLoad);
        state.jsonLoadingPromise.done(() => {
            if (window.dropPrint?.printSizesData) {
                if(typeof dropPrintUpdateOverlay === 'function' && elements.overlay.hasClass('is-loading')) dropPrintUpdateOverlay('', '', '');
                if(typeof dropPrintInitiateMatching === 'function') dropPrintInitiateMatching(Math.max(width, height), Math.min(width, height), width, height);
                if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', 'false').removeClass('disabled');
            } else if(typeof dropPrintShowErrorOverlay === 'function') {
                dropPrintShowErrorOverlay('Data Error', 'Product data (pricelist) failed to load correctly for media ID: ' + mediaToLoad, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['reset', 'refresh'] });
            }
            if (typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
        }).fail(() => {
            if (typeof dropPrintShowErrorOverlay === 'function') dropPrintShowErrorOverlay('Data Error', 'Failed to load pricelist information. Please try refreshing or check settings.', { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['reset', 'refresh'] });
            if (typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
        });
    }
    window.dropPrint.dropPrintStartExistingMetaWorkflow = dropPrintStartExistingMetaWorkflow;
    function dropPrintHandleSaveNow() {
        const $publishButton = $('#publish'), $saveDraftButton = $('#save-post');
        if ($publishButton.length && !$publishButton.is(':disabled') && $publishButton.is(':visible')) $publishButton.trigger('click');
        else if ($saveDraftButton.length && !$saveDraftButton.is(':disabled')) $saveDraftButton.trigger('click');
        else { alert('Could not trigger save. Please use the main WordPress save buttons.'); }
    }
    window.dropPrint.dropPrintHandleSaveNow = dropPrintHandleSaveNow;
    function dropPrintUpdateTabAppearance(isActive) {
        const $tabs = $('li.attribute_tab, li.variations_tab'), interceptedClass = 'drop-print-tab-intercepted';
        if (isActive) $tabs.addClass(interceptedClass); else $tabs.removeClass(interceptedClass);
    }
    window.dropPrint.dropPrintUpdateTabAppearance = dropPrintUpdateTabAppearance;
    function populateMediaDropdown() {
        const { elements, config, state } = window.dropPrint;
        if (!elements?.mediaSelect?.length || !config?.availableMedia || !Array.isArray(config.availableMedia)) {
            if(elements?.mediaSelect?.length) elements.mediaSelect.html('<option value="">Error loading media types</option>');
            return;
        }
        elements.mediaSelect.empty();
        if (config.availableMedia.length === 0) {
            elements.mediaSelect.html('<option value="">No media types available</option>');
            return;
        }
        config.availableMedia.forEach(mediaItem => {
            if (mediaItem && typeof mediaItem.id === 'string' && typeof mediaItem.name === 'string') {
                elements.mediaSelect.append($('<option></option>').val(mediaItem.id).text(mediaItem.name));
            }
        });
        let initialMediaId = state.currentMediaId || config.defaultMediaId;
        const hasExistingVariations = state?.metaActiveVariations && state.metaActiveVariations.length > 0;
        if (hasExistingVariations) {
            const firstSku = state.metaActiveVariations[0].sku;
            if (firstSku && typeof firstSku === 'string') {
                const lastHyphenIndex = firstSku.lastIndexOf('-');
                if (lastHyphenIndex > 0) {
                    const potentialMediaId = firstSku.substring(0, lastHyphenIndex).toLowerCase();
                    if (elements.mediaSelect.find('option[value="' + potentialMediaId + '"]').length > 0) {
                        initialMediaId = potentialMediaId;
                    }
                }
            }
        }
        if (initialMediaId && elements.mediaSelect.find('option[value="' + initialMediaId + '"]').length > 0) {
            elements.mediaSelect.val(initialMediaId);
        } else if (config.availableMedia.length > 0 && config.availableMedia[0].id) {
            elements.mediaSelect.val(config.availableMedia[0].id);
        } else {
            elements.mediaSelect.val('');
        }
        state.currentMediaId = elements.mediaSelect.val();
        if (hasExistingVariations) {
            elements.mediaSelect.prop('disabled', true);
        } else {
            elements.mediaSelect.prop('disabled', false);
        }
    }
    window.dropPrint.populateMediaDropdown = populateMediaDropdown;
    function triggerCurrencyMismatchCheck(pricelistCurrencyCode) {
        const { config, dropPrintUpdateOverlay } = window.dropPrint || {};
        if (!config || typeof dropPrintUpdateOverlay !== 'function') {
            return;
        }
        const wcCurrency = config.wcCurrency?.toUpperCase();
        const plCurrency = pricelistCurrencyCode?.toUpperCase();
        if (wcCurrency && plCurrency && wcCurrency !== plCurrency) {
            const wcSettingsUrl = $('<a>').attr('href', config.ajaxUrl.replace(/admin-ajax\.php.*/, 'admin.php?page=wc-settings&tab=general')).text('change your store currency in WooCommerce settings')[0].outerHTML;
            const dpAccountUrl = $('<a>').attr('href', 'https://drop-print.com/my-account/').attr('target', '_blank').text('create a new Drop Print account')[0].outerHTML;
            const title = "Store Currency Mismatch";
            const message = `<p>Your WooCommerce store currency (<strong>${wcCurrency}</strong>) does not match the billing currency of the currently selected pricelist (<strong>${plCurrency}</strong>).</p><p>For accurate pricing and to avoid confusion, it's recommended that these match. You can either ${wcSettingsUrl} or ${dpAccountUrl} with a billing currency that matches your store.</p><p>You may proceed, but please be mindful of currency differences when setting prices.</p>`;
            dropPrintUpdateOverlay('is-prompt', title, message, { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok', 'docs'] });
        }
    }
    window.dropPrint.triggerCurrencyMismatchCheck = triggerCurrencyMismatchCheck;
    function handleMediaChange() {
        const { state, elements, dropPrintLoadPricelistFromJson, dropPrintInitiateMatching, dropPrintUpdateOverlay } = window.dropPrint;
        const newMediaId = elements.mediaSelect.val();
        if (!newMediaId || newMediaId === state.currentMediaId) return;
        const proceedWithMediaChange = () => {
            state.currentMediaId = newMediaId;
            state.showingAllProducts = false;
            if (elements.showAllTrigger && elements.showAllTrigger.length) {
                elements.showAllTrigger.find('.drop-print-show-all-text').text('Show All Options');
                elements.showAllTrigger.find('.drop-print-show-all-indicator').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                elements.showAllTrigger.attr('aria-expanded', 'false').attr('aria-disabled', 'true').addClass('disabled').removeClass('is-sticky-top visible');
            }
            if (elements.productSearchInput) elements.productSearchInput.val('');
            if(typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('is-loading', 'Loading Products', `Loading ${elements.mediaSelect.find('option:selected').text()} products...`);
            if (elements.productListContainer) elements.productListContainer.html('<p class="drop-print-product-list-loading">Loading...</p>');
            state.jsonLoadingPromise = dropPrintLoadPricelistFromJson(newMediaId);
            state.jsonLoadingPromise.done(() => {
                if(typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('', '', '');
                if (state.fileDimensions?.width > 0 || state.metaActiveDimensions?.width > 0) {
                    const width = state.fileDimensions?.width || state.metaActiveDimensions?.width;
                    const height = state.fileDimensions?.height || state.metaActiveDimensions?.height;
                    if (window.dropPrint?.printSizesData) {
                        if (typeof dropPrintInitiateMatching === 'function') dropPrintInitiateMatching(Math.max(width, height), Math.min(width, height), width, height);
                    }
                }
                if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', String(!window.dropPrint?.printSizesData)).toggleClass('disabled', !window.dropPrint?.printSizesData);
                if(typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
            }).fail(() => { if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', 'true').addClass('disabled'); if(typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility(); });
        };
        const $checkedItems = elements.productListContainer.find('.drop-print-product-checkbox:checked:not(:disabled)');
        if ($checkedItems.length > 0) {
            if (typeof dropPrintUpdateOverlay === 'function') {
                const originalOverlayState = elements.overlay.attr('class');
                const originalTitle = elements.overlayTitle.text();
                const originalMessage = elements.overlayMessage.html();
                const originalOptions = {
                    showActions: elements.overlayActions.is(':visible'),
                    buttonsToShow: elements.overlayActions.find('button:visible:not(.dp-custom-button)').map((i, el) => $(el).attr('id').replace('drop-print-overlay-', '').replace('-btn', '')).get(),
                    customButtons: elements.overlayActions.find('button.dp-custom-button').map((i, el) => { const $btn = $(el); return { id: $btn.attr('id'), text: $btn.text(), class: $btn.attr('class').replace(/button\s*/g, '').replace(/dp-custom-button\s*/g, '').trim(), action: $._data($btn[0], "events")?.click?.[0]?.handler }; }).get(),
                    iconClass: elements.overlayDashicon.attr('class')?.match(/dashicons-[\w-]+/)?.[0]
                };
                const confirmChange = () => {
                    proceedWithMediaChange();
                };
                const cancelChange = () => {
                    elements.mediaSelect.val(state.currentMediaId);
                    dropPrintUpdateOverlay(originalOverlayState, originalTitle, originalMessage, originalOptions);
                };
                dropPrintUpdateOverlay('is-prompt', 'Media Type Change', '<p>You are about to change the media type, but you have items selected from the current media type. You may only select items from one media type per product.</p><p>Continuing will clear your current selections. Do you want to proceed?</p>', { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: [], customButtons: [ {id:'dp-media-change-confirm', text:'Proceed', class:'button-primary', action: confirmChange}, {id:'dp-media-change-cancel', text:'Cancel', class:'button-secondary', action: cancelChange} ] });
            } else {
                if (confirm("You are about to change the media type with items selected. Continuing will clear selections. Proceed?")) {
                    proceedWithMediaChange();
                } else {
                    elements.mediaSelect.val(state.currentMediaId);
                }
            }
        } else {
            proceedWithMediaChange();
        }
    }
    window.dropPrint.handleMediaChange = handleMediaChange;
    function handleProductSearch() {
        const { elements, state } = window.dropPrint;
        if (!elements || !elements.productSearchInput || !elements.productListContainer || !state) {
            return;
        }
        const searchTerm = elements.productSearchInput.val().toLowerCase().trim();
        const isSearchActive = searchTerm !== '';
        const displayModeIsAll = isSearchActive || state.showingAllProducts;
        elements.productListContainer.find('.drop-print-product').each(function() {
            const $product = $(this);
            const title = $product.find('.drop-print-product-title').text().toLowerCase();
            const sku = $product.data('sku')?.toString().toLowerCase() || '';
            let searchCriteriaMet = true;
            if (isSearchActive) {
                searchCriteriaMet = title.includes(searchTerm) || sku.includes(searchTerm);
            }
            let shouldShow = false;
            if (displayModeIsAll) {
                if (searchCriteriaMet) {
                    shouldShow = true;
                }
            } else {
                const isHiddenNonMatch = $product.hasClass('drop-print-hidden-non-match');
                if (searchCriteriaMet && !isHiddenNonMatch) {
                    shouldShow = true;
                }
            }
            $product.toggle(shouldShow);
        });
        if (typeof updateShowAllTriggerVisibility === 'function') {
            updateShowAllTriggerVisibility();
        }
    }
    const debouncedProductSearch = typeof _ !== 'undefined' && typeof _.debounce === 'function' ? _.debounce(handleProductSearch, 300) : handleProductSearch;
    window.dropPrint.handleProductSearch = handleProductSearch;
    function handleShowAllProducts() {
        const { state, elements } = window.dropPrint;
        if (!elements || !elements.showAllTrigger || !elements.showAllTrigger.length || !state ) return;
        if (elements.showAllTrigger.attr('aria-disabled') === 'true' && !state.showingAllProducts && elements.showAllTrigger.hasClass('disabled')) {
            return;
        }
        state.showingAllProducts = !state.showingAllProducts;
        elements.showAllTrigger.find('.drop-print-show-all-text').text(state.showingAllProducts ? 'Show Matching Only' : 'Show All Options');
        elements.showAllTrigger.find('.drop-print-show-all-indicator').toggleClass('dashicons-arrow-down-alt2', !state.showingAllProducts).toggleClass('dashicons-arrow-up-alt2', state.showingAllProducts);
        elements.showAllTrigger.attr('aria-expanded', state.showingAllProducts.toString());
        if (state.showingAllProducts) {
            elements.showAllTrigger.addClass('is-sticky-top').removeClass('visible');
        } else {
            elements.showAllTrigger.removeClass('is-sticky-top');
        }
        if(typeof window.dropPrint.handleProductSearch === 'function') window.dropPrint.handleProductSearch();
        if(typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
    }
    window.dropPrint.handleShowAllProducts = handleShowAllProducts;
    $(function() {
        if(typeof window.dropPrint.initializeDropPrintElements === 'function') window.dropPrint.initializeDropPrintElements();
        const { elements, config, state, dropPrintUpdateOverlay, dropPrintLoadPricelistFromJson, dropPrintShowErrorOverlay, dropPrintInitDropzone, dropPrintStartExistingMetaWorkflow, dropPrintReset, dropPrintToggleTab } = window.dropPrint || {};
        if (!elements || !config || !state || typeof dropPrintUpdateOverlay !== 'function' || typeof dropPrintLoadPricelistFromJson !== 'function' || typeof dropPrintShowErrorOverlay !== 'function') {
            if (typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('is-error', 'Initialization Error', 'Core components failed. Please reload.', { showActions: true, buttonsToShow: ['reset'] });
            else alert('DropPrint Critical Error: Core components failed. Please reload.');
            return;
        }
        if(typeof window.dropPrint.populateMediaDropdown === 'function') window.dropPrint.populateMediaDropdown();
        if (config.defaultPricelist && typeof window.dropPrint.dropPrintIsJsonDataValid === 'function' && window.dropPrint.dropPrintIsJsonDataValid(config.defaultPricelist) && config.defaultPricelist.media_id === state.currentMediaId) {
            state.jsonLoadingPromise = $.Deferred().resolve(config.defaultPricelist).promise();
            window.dropPrint.printSizesData = config.defaultPricelist;
            if (typeof window.dropPrint.dropPrintRenderProductList === 'function') window.dropPrint.dropPrintRenderProductList();
            else { if(elements.productListContainer) elements.productListContainer.html('<p class="drop-print-error">Error: Product rendering function failed.</p>');}
        } else if (state.currentMediaId) {
            state.jsonLoadingPromise = dropPrintLoadPricelistFromJson(state.currentMediaId);
        } else {
            if (elements.productListContainer) elements.productListContainer.html('<p class="drop-print-product-list-loading">Please select a media type to see products.</p>');
            state.jsonLoadingPromise = $.Deferred().resolve().promise();
        }
        state.jsonLoadingPromise.always(() => {
            if (elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', String(!(window.dropPrint?.printSizesData))).toggleClass('disabled', !(window.dropPrint?.printSizesData));
            if (typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
        });
        const isInitiallyBlocked = elements.overlay?.hasClass('is-blocked');
        const hasExistingDims = state?.metaActiveDimensions?.width > 0 && state?.metaActiveDimensions?.height > 0;
        const hasExistingFile = state?.metaFileName || state?.metaPreviewId > 0;
        if (isInitiallyBlocked) {
            const initialTitle = elements.overlayTitle?.text() || 'Action Required';
            const initialMessage = elements.overlayMessage?.html() || 'Please save the product first.';
            const initialIconClass = elements.overlayDashicon?.attr('class')?.replace(/dashicons\s*/, '') || 'dashicons-warning';
            if(typeof dropPrintUpdateOverlay === 'function') dropPrintUpdateOverlay('is-blocked', initialTitle, initialMessage, { showActions: true, iconClass: initialIconClass, buttonsToShow: ['save_now'] });
            if (elements.dropzoneWrapper) $(elements.dropzoneWrapper).show();
            if (elements.dropPrintStep) $(elements.dropPrintStep).removeClass('active');
            if(elements.dropPrintResetButton) $(elements.dropPrintResetButton).prop('disabled', true);
            if(elements.dropPrintAddVariationsButton) $(elements.dropPrintAddVariationsButton).prop('disabled', true);
            if(elements.showAllTrigger && elements.showAllTrigger.length) $(elements.showAllTrigger).attr('aria-disabled', 'true').addClass('disabled');
        } else if (hasExistingDims && hasExistingFile) {
            if (typeof dropPrintStartExistingMetaWorkflow === 'function') dropPrintStartExistingMetaWorkflow();
            else if(typeof dropPrintShowErrorOverlay === 'function') dropPrintShowErrorOverlay("Initialization Error", "Cannot start existing meta workflow.");
        } else {
            if (elements.dropzoneWrapper) $(elements.dropzoneWrapper).show();
            if (elements.dropPrintStep) $(elements.dropPrintStep).removeClass('active');
            if (typeof dropPrintInitDropzone === 'function') dropPrintInitDropzone();
            else { if(typeof dropPrintShowErrorOverlay === 'function') dropPrintShowErrorOverlay('Initialization Error', 'Upload component failed.'); }
            if (typeof window.dropPrint.dropPrintUpdateFileInfo === 'function') window.dropPrint.dropPrintUpdateFileInfo(null, null);
        }
        if (typeof dropPrintToggleTab === 'function') dropPrintToggleTab(config?.metaIsActive === 'yes', true);
        if (elements.dropPrintEnableCheckbox) elements.dropPrintEnableCheckbox.on('change', function() { if (typeof dropPrintToggleTab === 'function') dropPrintToggleTab(this.checked, false); });
        if (elements.dropPrintResetButton) $(elements.dropPrintResetButton).on('click', dropPrintReset);
        if (elements.overlayActions) {
            elements.overlayActions.on('click', '#drop-print-overlay-confirm-reset-btn', e => { e.preventDefault(); if (typeof window.dropPrint.dropPrintExecuteReset === 'function') window.dropPrint.dropPrintExecuteReset(); });
            elements.overlayActions.on('click', '#drop-print-overlay-reset-btn', e => { e.preventDefault(); if (typeof window.dropPrint.dropPrintUpdateOverlay === 'function') window.dropPrint.dropPrintUpdateOverlay('', '', ''); });
        }
        if (elements.mediaSelect) elements.mediaSelect.on('change', () => { if(typeof window.dropPrint.handleMediaChange === 'function') window.dropPrint.handleMediaChange();});
        if (elements.productSearchInput) elements.productSearchInput.on('input', debouncedProductSearch);
        if (elements.showAllTrigger && elements.showAllTrigger.length) elements.showAllTrigger.on('click keypress', function(e) { if (e.type === 'click' || (e.type === 'keypress' && (e.key === 'Enter' || e.key === ' '))) { e.preventDefault(); if(typeof window.dropPrint.handleShowAllProducts === 'function') window.dropPrint.handleShowAllProducts();} });
        if (elements.productListContainer && elements.productListContainer.length) {
            showAllTriggerScrollHandler = (typeof _ !== 'undefined' && typeof _.debounce === 'function') ? _.debounce(updateShowAllTriggerVisibility, 150) : updateShowAllTriggerVisibility;
            elements.productListContainer.on('scroll.dpShowAll', showAllTriggerScrollHandler);
        }
        setTimeout(updateShowAllTriggerVisibility, 250);
        if (elements.dropPrintOverlayDownloadBtn) {
            elements.dropPrintOverlayDownloadBtn.on('click', async function(e) {
                e.preventDefault();
                const { elements: elems, dropPrintUpdateOverlay: updateOverlay, config: currentConfig } = window.dropPrint || {};
                if (!elems?.overlay || typeof updateOverlay !== 'function' ||
                    !currentConfig?.apiBaseUrl ||
                    !currentConfig?.apiAuthUserB64 ||
                    !currentConfig?.apiAuthPassB64) {
                    if(typeof updateOverlay === 'function') {
                        updateOverlay('is-error', 'Download Error', 'Client/API configuration missing for authenticated request.', { showActions: true, buttonsToShow: ['reset', 'ok'] });
                    } else {
                        alert('Download Error: Client/API configuration missing for authenticated request.');
                    }
                    return;
                }
                const dimsStr = elems.overlay.data('template-dimensions');
                const titleStr = elems.overlay.data('template-title') || 'template';
                if (!dimsStr || typeof dimsStr !== 'string' || !dimsStr.includes('x')) {
                    updateOverlay('is-error', 'Download Error', 'Invalid dimensions data for template.', { showActions: true, buttonsToShow: ['reset', 'ok'] });
                    return;
                }
                const dims = dimsStr.split('x').map(Number);
                if (dims.length !== 2 || dims.some(isNaN) || dims.some(d => d <= 0)) {
                    updateOverlay('is-error', 'Download Error', 'Invalid dimensions data for template (must be positive numbers).', { showActions: true, buttonsToShow: ['reset', 'ok'] });
                    return;
                }
                const apiUrl = (currentConfig.apiBaseUrl.endsWith('/') ? currentConfig.apiBaseUrl : currentConfig.apiBaseUrl + '/') + 'template/' + dimsStr;
                const safeSlug = titleStr.toLowerCase().replace(/[^a-z0-9\-\s]/g, '').replace(/\s+/g, '-');
                const defaultDownloadFilename = `${safeSlug}-${dimsStr}px-dp-template.jpg`;
                updateOverlay('is-loading', 'Requesting Template', 'Authenticating and preparing download...', { showActions: false });
                try {
                    const decodedUser = atob(currentConfig.apiAuthUserB64);
                    const decodedPass = atob(currentConfig.apiAuthPassB64);
                    const authHeader = 'Basic ' + btoa(decodedUser + ':' + decodedPass);
                    const response = await fetch(apiUrl, {
                        method: 'GET',
                        headers: { 'Authorization': authHeader },
                        cache: 'no-store'
                    });
                    if (response.ok) {
                        const contentType = response.headers.get("content-type");
                        if (contentType && (contentType.startsWith("image/") || contentType.startsWith("application/octet-stream"))) {
                            updateOverlay('is-loading', 'Download Ready', 'Processing file for download...', { showActions: false });
                            const blob = await response.blob();
                            const downloadUrl = URL.createObjectURL(blob);
                            const anchor = document.createElement('a');
                            let fileName = defaultDownloadFilename;
                            const disposition = response.headers.get('content-disposition');
                            if (disposition && disposition.indexOf('attachment') !== -1) {
                                const filenameRegex = /filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/;
                                const matches = filenameRegex.exec(disposition);
                                if (matches != null && matches[1]) {
                                    fileName = matches[1].replace(/['"]/g, '');
                                }
                            }
                            anchor.href = downloadUrl;
                            anchor.setAttribute('download', fileName);
                            document.body.appendChild(anchor);
                            anchor.click();
                            document.body.removeChild(anchor);
                            URL.revokeObjectURL(downloadUrl);
                            setTimeout(() => {
                                updateOverlay('', '', '');
                                if (elems.overlay) {
                                   elems.overlay.removeData('template-dimensions').removeData('template-title');
                                }
                            }, 500);
                        } else {
                            const responseData = await response.json();
                            let errorMsg = 'Received an unexpected response type from the server.';
                            if (responseData && responseData.message) {
                                 errorMsg = responseData.message;
                            } else if (responseData && typeof responseData === 'string') {
                                errorMsg = responseData;
                            }
                            throw new Error(errorMsg);
                        }
                    } else {
                        let errorFromServer = `Server responded with status ${response.status}.`;
                        try {
                            const errorData = await response.json();
                            if (errorData.message) {
                                errorFromServer = errorData.message;
                            } else if (errorData.data && errorData.data.message) {
                                errorFromServer = errorData.data.message;
                            } else if (typeof errorData === 'string') {
                                errorFromServer = errorData;
                            }
                        } catch (jsonError) {
                            try {
                                const textError = await response.text();
                                if (textError) errorFromServer = textError.substring(0, 200);
                            } catch (textE) { }
                        }
                        throw new Error(errorFromServer);
                    }
                } catch (error) {
                    let displayMsg = (error instanceof Error ? error.message : String(error) || 'Request failed.');
                    if (displayMsg.includes('401') || displayMsg.toLowerCase().includes('permission to access this resource')) displayMsg = 'Authentication failed. Please check your Drop Print connection or server permissions.';
                    else if (displayMsg.includes('403')) displayMsg = 'Permission denied. Ensure your user account on the server has the required capabilities.';
                    else if (displayMsg.includes('404')) displayMsg = 'Template API endpoint not found. Check API Base URL on client or route definition on server.';
                    else if (displayMsg.includes('500') || displayMsg.toLowerCase().includes('imagick') || displayMsg.toLowerCase().includes('generate the image')) displayMsg = 'Server error during image generation. Check server logs.';
                    else if (displayMsg.toLowerCase().includes('fetch failed') || displayMsg.toLowerCase().includes('networkerror')) displayMsg = 'Network error fetching template. Check connection or server status.';
                    else if (displayMsg.includes('atob')) displayMsg = 'Error decoding credentials on client-side. Check API details in settings.';
                    updateOverlay('is-error', 'Download Error', displayMsg, { showActions: true, buttonsToShow: ['reset', 'ok'] });
                }
            });
        }
        if (elements.dropPrintOverlayOkBtn) elements.dropPrintOverlayOkBtn.on('click.dropPrintDismiss', e => { e.preventDefault(); if (typeof window.dropPrint.dropPrintUpdateOverlay === 'function') window.dropPrint.dropPrintUpdateOverlay('', '', ''); });
        const saveHandler = e => { if (typeof window.dropPrint.dropPrintHandleSaveNow === 'function') window.dropPrint.dropPrintHandleSaveNow(e); $(e.currentTarget).prop('disabled', true); };
        if (elements.dropPrintOverlaySaveBtn) elements.dropPrintOverlaySaveBtn.on('click', saveHandler);
        if (elements.dropPrintOverlaySaveNowBtn) elements.dropPrintOverlaySaveNowBtn.on('click', saveHandler);
        const productDataElement = elements.product_data;
        if (productDataElement) {
            productDataElement.addEventListener('click', event => {
                const targetLink = event.target.closest('.wc-tabs a, .product_data_tabs a'); if (!targetLink) return;
                const tabHref = targetLink.getAttribute('href'), isAttributesTab = (tabHref === '#product_attributes'), isVariationsTab = (tabHref === '#variable_product_options');
                const requiresSave = window.dropPrint?.state?.dropPrintRequiresSave === true;
                if ((isAttributesTab || isVariationsTab) && requiresSave) {
                    event.preventDefault(); event.stopPropagation();
                    if (typeof window.dropPrint.dropPrintUpdateOverlay === 'function') { window.dropPrint.dropPrintUpdateOverlay('is-prompt', 'Save Required', '<p>Please <strong>Update</strong> or <strong>Publish</strong> the product first before modifying attributes or variations directly.</p>', { showActions: true, iconClass: 'dashicons-warning', buttonsToShow: ['ok', 'docs'] });
                    if (typeof window.dropPrint.dropPrintUpdateTabAppearance === 'function') window.dropPrint.dropPrintUpdateTabAppearance(true); }
                    else alert("Save Required: Save the product before modifying attributes/variations.");
                } else if (typeof window.dropPrint.dropPrintUpdateTabAppearance === 'function') window.dropPrint.dropPrintUpdateTabAppearance(false);
            }, true);
        }
        $(document).on('productsRenderedandMatched.dropPrint', function() {
            setTimeout(function() {
                const { state: curSt, elements: curEl, config } = window.dropPrint || {};
                const inExMeta = curSt?.metaActiveDimensions?.width && Array.isArray(curSt?.metaActiveVariations) && curSt.metaActiveVariations.length > 0 && !curSt?.currentFile;
                if (inExMeta && curEl?.productListContainer) {
                    const $productsInDOM = curEl.productListContainer.find('.drop-print-product');
                    $productsInDOM.each(function(index) {
                        const $prd = $(this);
                        const pSku = $prd.data('sku');
                        const mVar = curSt.metaActiveVariations.find(v => String(v.sku) === String(pSku) || (v.wc_variation_id && String(v.wc_variation_id) === String($prd.data('wc-variation-id')) ));
                        if (mVar) {
                            const $chk = $prd.find('.drop-print-product-checkbox');
                            const $prc = $prd.find('.drop-print-product-sell-price');
                            if ($chk.length && !$prd.hasClass('drop-print-hidden-non-match') && !$prd.hasClass('unsuitable-product')) {
                                $chk.prop('checked', true).prop('disabled', false);
                                $prd.addClass('selected').data('is-selectable', true);
                            }
                            if ($prc.length && mVar.price !== undefined) {
                                const sPrice = parseFloat(mVar.price).toFixed(2);
                                if (!isNaN(sPrice)) {
                                    $prc.val(sPrice).attr('data-price', sPrice);
                                 }
                            }
                        }
                    });
                    if (typeof window.dropPrint.dropPrintUpdateButtonState === 'function') window.dropPrint.dropPrintUpdateButtonState();
                }
                if (curEl?.showAllTrigger && curEl.showAllTrigger.length) $(curEl.showAllTrigger).attr('aria-disabled', String(!(window.dropPrint?.printSizesData))).toggleClass('disabled', !(window.dropPrint?.printSizesData));
                if (typeof window.dropPrint.triggerCurrencyMismatchCheck === 'function' && window.dropPrint?.printSizesData?.currency_code) {
                    window.dropPrint.triggerCurrencyMismatchCheck(window.dropPrint.printSizesData.currency_code);
                }
                if(typeof updateShowAllTriggerVisibility === 'function') updateShowAllTriggerVisibility();
            }, 100);
        });
        $('#woocommerce-product-data').on('change', '#variable_product_options input.wc_input_price', function() {
            const $wcPriceInput = $(this);
            const $variationRow = $wcPriceInput.closest('.woocommerce_variation');
            const wcVariationId = parseInt($variationRow.find('.remove_variation').attr('rel'), 10);
            const newPrice = $wcPriceInput.val();
            if (!wcVariationId || isNaN(wcVariationId)) return;
            const dpMetaVariations = window.dropPrint?.state?.metaActiveVariations || [];
            let dpSkuToUpdate = null;
            for (let i = 0; i < dpMetaVariations.length; i++) {
                if (parseInt(dpMetaVariations[i].wc_variation_id, 10) === wcVariationId) {
                    dpSkuToUpdate = dpMetaVariations[i].sku;
                    break;
                }
            }
            if (dpSkuToUpdate) {
                const $dpProductRow = $('#drop-print-product-' + dpSkuToUpdate);
                if ($dpProductRow.length) {
                    const $dpPriceInput = $dpProductRow.find('.drop-print-product-sell-price');
                    $dpPriceInput.val(newPrice).attr('data-price', newPrice);
                    const $checkbox = $dpProductRow.find('.drop-print-product-checkbox');
                    if ($checkbox.length && !$checkbox.prop('checked') && !$checkbox.prop('disabled') && newPrice.trim() !== '') {
                        $checkbox.prop('checked', true);
                        const changeEvent = new Event('change', { bubbles: true });
                        $checkbox.get(0).dispatchEvent(changeEvent);
                    } else if (typeof window.dropPrint.dropPrintUpdateButtonState === 'function') {
                        window.dropPrint.dropPrintUpdateButtonState();
                    }
                    if (window.dropPrint && !window.dropPrint.state.dropPrintRequiresSave) {
                        window.dropPrint.state.dropPrintRequiresSave = true;
                        if (typeof window.dropPrint.dropPrintUpdateTabAppearance === 'function') {
                            window.dropPrint.dropPrintUpdateTabAppearance(true);
                        }
                    }
                }
            }
        });
    });
}(jQuery));