<?php

namespace _8webit\Dragonfly\Frontend\Rest;

use _8webit\Dragonfly\Common\_InstanceTrait;
use _8webit\Dragonfly\Common\Options;
use _8webit\Dragonfly\Frontend\Handlers\Search as SearchHandler;


if (!defined('ABSPATH')) {
    exit;
}

class Search
{
    use _InstanceTrait;

    private $errors;

    public function __construct()
    {
        $this->errors = array(
            'invalid_values' => new \WP_Error('400', 'Invalid value', array('status' => 'invalid_value')),
            'no_result' => new \WP_Error('404', 'No Result found', array('status' => 'no_result'))
        );
    }

    public static function init()
    {
        add_action('rest_api_init', array(self::get_instance(), 'register_frontend_route'));

        if (is_admin()) {
            add_action('wp_ajax_dragonfly_admin_search', array(self::get_instance(), 'admin_callback'));
        }
    }

    public function register_frontend_route()
    {
        register_rest_route('8webit/v1', '/dragonfly/(?<value>[\s\S]+)', array(
            'methods' => 'GET',
            'callback' => array(self::get_instance(), 'frontend_callback')
        ));
    }

    public function frontend_callback($data)
    {
        if (!isset($data['value'])) {
            return $this->errors['invalid_values'];
        }

        $needle = urldecode($data['value']);
        $needle = esc_html(sanitize_text_field($needle));

        $needle = preg_replace('!\s+!', ' ', $needle);// removes multiply whitespace
        $needle = trim($needle);

        if (mb_strlen($data['value']) < Options::get_minimal_chars() || mb_strlen($data['value']) > 100) {
            return $this->errors['invalid_values'];
        }

        $options = $this->get_options();

        $result = SearchHandler::run(
            $needle,
            $options
        );


        if (empty($result)) {
            return $this->errors['no_result'];
        }

        return SearchHandler::parse($result);
    }

    function admin_callback()
    {
        check_ajax_referer('dragonfly_admin_nonce');

        $needle = urldecode($_POST['data']['needle']);
        $needle = esc_html(sanitize_text_field($needle));
        $options = $this->options_from_params($this->get_options(), $_POST['data']['options']);

        $result = SearchHandler::run(
            $needle,
            $options
        );


        if (empty($result)) {
            wp_send_json_error($this->errors['no_result'], '404');
        }

        wp_send_json(SearchHandler::parse($result));
        wp_die();
    }

    private function get_options()
    {
        return [
            'search_by' => Options::get_searched_by(),
            'searched_terms' => Options::get_searched_taxs(),
            'post_types' => Options::get_searched_post_types(),
            'suggestion_count' => Options::get_suggestion_count(),
            'cat_lvl' => Options::get_cat_lvl(),
            'cat_count' => Options::get_cat_count()
        ];
    }

    private function options_from_params($default_options, $new_options)
    {
        foreach ($default_options as $key => $value) {
            if (isset($new_options[$key])) {
                $new_val = $new_options[$key];
                if (is_array($new_val)) {
                    foreach ($new_val as $val) {
                        $default_options[$key][] = esc_html(sanitize_text_field($val));
                    }
                } else {
                    $default_options[$key] = esc_html(sanitize_text_field($new_val));
                }
            }
        }
        return $default_options;
    }
}
