<?php
namespace _8webit\Dragonfly\Frontend\Handlers;

use _8webit\Dragonfly\Common\_InstanceTrait;
use _8webit\Dragonfly\Common\Options;


/**
 * Class Search
 * @package _8webit\Dragonfly\Frontend\Handlers
 * @since 1.0.0
 */
class Search{
    use _InstanceTrait;
    
    private static $searched_by;
    private static $searched_terms;
    private static $post_types;
	private static $suggestion_count;
	private static $cat_lvl;
	private static $cat_count;

	/**
	 * gets search result.
	 *
	 * search options structure:
	 *  [
	 *	'search_by' => []               @see Options::get_searched_by()
	 *	'searched_terms' => [],         @see Options::get_searched_taxs()
	 *	'post_types' => [],             @see Options::get_searched_taxs()
	 *	'suggestion_count' => '',       @see Options::get_suggestion_count()
	 *	'cat_lvl' => '',                @see Options::get_cat_lvl()
	 *	'cat_count' => ''               @see Options::get_cat_count()
	 *	];
	 *
	 * @since 1.0.0
	 *
	 * @param string $needle    searched value
	 * @param array $options     search options
	 *
	 * @return array    WP_POST array
	 */
    public static function run($needle, $options){
        $result = [];
        
        self::set_options($options);

        if(isset(self::$searched_by['title'])){
            $args = self::get_base_query_args();
            $args['s'] = $needle;
    
            $result = get_posts($args);
        }

        if(count($result) < self::$suggestion_count && isset(self::$searched_by['sku'])){
            $temp = get_posts(self::get_search_by_sku_args($needle));
            $result = array_unique( array_merge( $result, $temp ), SORT_REGULAR );
        }

        if(count($result) < self::$suggestion_count){
            $temp = get_terms(self::get_tax_args($needle));
            $result = array_unique( array_merge( $result, $temp ), SORT_REGULAR );
        }

        array_splice($result, self::$suggestion_count);

        return $result;
	}

	/**
	 * parses given posts by its types and returns its instance.
	 *
	 * @since 1.0.0
	 *
	 * @param $posts array  WP_POST
	 *
	 * @return array|bool   mixed data of WC_Product, WP_POST or WP_TERM
	 */
	public static function parse($posts){
		if(!$posts || empty($posts)){
			return false;
		}

		$results = [];
		foreach($posts as $post){
			if($result = self::parse_data_by_type($post)){
				$results[] = $result;
			}else{
				break;
			}
		}

		if(empty($results)){
			return false;
		}

		return $results;
	}

	/**
	 * merges given options with defaults and sets class options properties
	 *
	 * $options param is same as @see Search::run()
	 *
	 * @sice 1.0.0
	 *
	 * @param array $options
	 *
	 * @return Search
	 */
    private static function set_options($options){
        $defaults = [
            'search_by' => ['title' => true],
            'searched_terms' => false,
            'post_types' => 'product',
			'suggestion_count' => 3,
			'cat_lvl' => 1,
			'cat_count' => 1
        ];

        $options = array_merge($defaults, $options);

        self::$searched_by      = $options['search_by'];
        self::$searched_terms   = $options['searched_terms'];
        self::$post_types       = $options['post_types'];
        self::$suggestion_count = $options['suggestion_count'];
		self::$cat_lvl          = $options['cat_lvl'];
		self::$cat_count        = $options['cat_count'];

        self::get_instance();
    }

	/**
	 * gets base args for get_post() function
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
    private static function get_base_query_args(){
        return array(
            "post_type" => self::$post_types,
            "order" => 'ASC',
            "orderby" => 'relevance',
            "compare" => 'LIKE',
            'numberposts'   => self::$suggestion_count,
        );
    }

	/**
	 * gets args for search by sku query
	 *
	 * @since  1.0.0
	 *
	 * @param $needle
	 *
	 * @return array
	 */
    private static function get_search_by_sku_args($needle){
        $sku_q_args = self::get_base_query_args();
        $sku_q_args['meta_query'] = array(
            array(
                'key' => '_sku',
                'value' => $needle,
                'compare' => 'LIKE'
            )
        );

        return $sku_q_args;
    }

	/**
	 * get args for get_terms() function
	 *
	 * @since 1.0.0
	 *
	 * @param string $needle    taxonomy name
	 *
	 * @return array
	 */
    private static function get_tax_args($needle){
        return array(
            'taxonomy' => Options::get_searched_taxs(),
            "order" => 'ASC',
            "orderby" => 'relevance',
            "compare" => 'LIKE',
            'number' => self::$suggestion_count,
            'search'    => $needle
        );
    }

	/**
	 * gets instance of WC_PRODUCT, WP_TERM or WP_POST by WP_POST->id
	 *  and extracts only necessary property data
	 *
	 * @since 1.0.0
	 *
	 * @param array $post   WP_POST instance
	 *
	 * @return array|bool
	 */
	private static function parse_data_by_type($post){
		if(property_exists($post, 'term_id')){
			return self::get_term($post);
		}

		if($wc_product = wc_get_product($post->ID)){
			return self::get_wc_product($wc_product);
		}else{
			return self::get_post($post);
		}
	}

	/**
	 * extract term data
	 *
	 * @since 1.0.0
	 *
	 * @param \WP_Post $post
	 *
	 * @return array
	 */
	private static function get_term($post){
		return array(
			'title' => $post->name,
			'url'   => get_term_link( $post->term_id )
		);
	}

	/**
	 * extract post data
	 *
	 * @sice 1.0.0
	 *
	 * @param $post
	 *
	 * @return array
	 */
	private static function get_post($post){
		return array(
			'title' => $post->post_title,
			'url'   => get_permalink( $post->ID ),
			'thumbnail' => get_the_post_thumbnail_url($post->ID, 'thumbnail'),
			'categories' => self::get_terms(wp_get_post_categories($post->ID), 'category')
		);
	}

	/**
	 * extract WC_Product data
	 *
	 * @sice 1.0.0
	 *
	 * @param \WC_Product $wc_product
	 *
	 * @return array|bool
	 */
	private static function get_wc_product($wc_product){
		$stock_status = '';
		$image_size = apply_filters( 'single_product_archive_thumbnail_size', 'thumbnail' );
		$thumbnail_url = get_the_post_thumbnail_url( $wc_product->get_id(), $image_size );
		$category_ids = $wc_product->get_category_ids();

		if(empty($wc_product)){
			return false;
		}

		if( $wc_product->get_stock_status() == 'instock'){
			$stock_status = __('In Stock','dragonfly');
		}else{
			$stock_status = __('Out Of Stock','dragonfly');
		}

		return array(
			'title' => $wc_product->get_title(),
			'url'   => get_permalink( $wc_product->get_id() ),
			'stock_status' => $stock_status,
			'price_html' => $wc_product->get_price_html(),
			'thumbnail' => $thumbnail_url,
			'sku' => $wc_product->get_sku(),
			'categories' => self::get_terms($category_ids, 'product_cat')
		);
	}

	/**
	 * gets terms  by ids
	 *
	 * @sinc 1.0.0
	 *
	 * @param $term_ids
	 * @param $taxonomy
	 *
	 * @return bool|string
	 */
	private static function get_terms($term_ids, $taxonomy){
		$terms = false;

		if(count($term_ids) > 0){
			$last = end($term_ids);

			$count = 0;
			foreach($term_ids as $id){
				if($count == self::$cat_count){
					break;
				}

				$count++;

				if($term = get_term_by( 'id', $id, $taxonomy )){
					if($term->parent == 0 || self::$cat_lvl == 1){
						$terms .= $term->name;
					}if($term->parent > 0 && self::$cat_lvl > 1){
						$parent_id = $term->parent;
						$depth = 1;

						while($parent_id != 0 && $depth < self::$cat_lvl){
							$depth++;

							$pterm = get_term_by( 'id', $parent_id, $taxonomy );
							$terms .= $pterm->name . '/';
							$parent_id = $pterm->parent;
						}

						$terms .= $term->name;
					}

					if($last !== $id && $count < self::$cat_count){
						$terms .= ', ';
					}
				}
			}
		}
		return $terms;
	}
}