import debounce from "lodash/debounce";
import {chunkArray, hasParent} from "./helpers";

/**
 * initiates Dragonfly search field
 *
 * @since 1.0.0
 *
 * @param element
 */
export function initSearchField(element){
    element.addEventListener('keyup',function(e){
        hideSuggestion(e.target.parentNode);
    });

    element.addEventListener('keyup',debounce(function(e){
        onKeyupSearchField(
            encodeURIComponent(e.target.value),
            e.target.parentNode,
            dragonflySettings,
            frontendAjax
        );
    }, 300));
}

/**
 * function called when keyup event fired on field.
 * checks if field value length is correct to fire callback,
 * if isn't hides suggestion.
 *
 * @since 1.0.0
 *
 * @param needle        field value
 * @param parentNode    field element parent
 * @param settings      Dragonfly settings for quering
 * @param callback      callback function
 */
export function onKeyupSearchField(needle, parentNode, settings, callback) {
    if (needle.length >= settings.minLength) {
        callback(needle, parentNode, settings);
    } else {
        hideSuggestion(parentNode);
    }
}

/**
 * sends GET ajax requist and renders values with given template
 *
 * used for frontend
 *
 * @param needle    searched value (search field value)
 * @param el        search field parent(or where suggestion result should appended)
 * @param settings  Dragonfly query settings
 */
function frontendAjax(needle, el, settings){
    var url = settings.restUrl + '8webit/v1/dragonfly/';
    url += needle;

    fetch(url)
        .then(function(response) {
            return response.json();
        }).then(function(json) {
            ajaxResult(el, json, settings);
        });
}

/**
 * sends POST ajax requist with given settings from admin settings page and renders values with template
 *
 * @since 1.0.0
 *
 * @param needle    searched value (search field value)
 * @param el        search field parent(or where suggestion result should appended)
 * @param settings  Dragonfly query settings
 */
export function adminAjax(needle, el, settings){
    fetch(settings.restUrl,{
            method: 'POST',
            credentials: 'same-origin',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded;'
            },
            body: $.param({
                _ajax_nonce : settings.ajaxNonce,
                action : 'dragonfly_admin_search',
                data : {
                    'needle': needle,
                    'options': settings.searchOptions
                }
            })
        }).then(function(response) {
            return response.json();
        }).then(function(json) {
            ajaxResult(el, json.data ? json.data[0] : json, settings);
        });
}

/**
 * renders ajax suggestions
 *
 * @since 1.0.0
 *
 * @param el
 * @param json
 * @param settings
 */
function ajaxResult(el, json, settings){
    if(json.code){
        switch (json.code){
            case 400:
            case 404:
                renderSuggestion(el, '<p class="fail-message">' + json.message + '</p>');
                break;
            default:
                renderSuggestion(el, '<p class="fail-message">Upss. Something went wrong :(</p>');
        }
    }else{
        var parsedSuggestions = '';

        if(hasParent(el, 'dragonfly-fullscreen')){
            parsedSuggestions = parseSuggestions(json, settings.template, settings.columnCount);
        }else{
            parsedSuggestions = parseSuggestions(json, settings.template);
        }

        renderSuggestion(el, parsedSuggestions);
    }
}

/**
 * renders ajax single suggestion with given template
 *
 * @since 1.0.0
 *
 * @param element
 * @param resultHtml
 */
function renderSuggestion(element,resultHtml){
    var resultEl = element.parentNode.querySelector('.dragonfly-results');

    if(resultEl){
        show(resultEl);
        resultEl.innerHTML = resultHtml;
    }else{
        var html = '<div class="dragonfly-results">' + resultHtml + '</div>';
        element.parentNode.insertAdjacentHTML('beforeend', html);
    }
}

/**
 * hides search suggestion(search result)
 * @param element
 */
export function hideSuggestion(element){
    var results = element.parentNode.querySelector('.dragonfly-results');

    if(results){
        hide(results);
    }
}

/**
 * inserts search result data in html template,
 * if third paramater specified, also inserts rows and columns
 *
 * @since 1.0.0
 *
 * @param json              Dragonfly search result data
 * @param template          Dragonfly html template
 * @param columnCount       number of columns
 * @returns {string}
 */
function parseSuggestions(data, template, columnCount = ''){
    if(!data.length){
        return '';
    }

    var resutlHTML = '';
    var templateEl = document.createElement('a');

    templateEl.setAttribute('class', 'dragonfly-result');
    templateEl.setAttribute('href', '');
    templateEl.innerHTML = template;

    componentsIDtoClass(templateEl);
    if(columnCount){
        resutlHTML += '<div class="dragonfly-result-row">';

        for(var i = 0; i < data.length; i++){
            resutlHTML += '<div class="dragonfly-result-column">';
            resutlHTML += parseSuggestionTemplate(data[i], templateEl.cloneNode(true));
            resutlHTML += '</div>';
        }

        resutlHTML += '</div>';
    }else{
        for(var i = 0; i < data.length; i++){
            resutlHTML += parseSuggestionTemplate(data[i], templateEl.cloneNode(true));
        }
    }
    return resutlHTML;
}

/**
 * changes componnent id to class.
 *
 * @since 1.0.0
 *
 * @param templateNode
 */
function componentsIDtoClass(templateNode){
    var elements = [
        templateNode.querySelector('#dragonfly-suggestion-title'),
        templateNode.querySelector('#dragonfly-suggestion-cat'),
        templateNode.querySelector('#dragonfly-suggestion-price'),
        templateNode.querySelector('#dragonfly-suggestion-sku'),
        templateNode.querySelector('#dragonfly-suggestion-stock-status'),
        templateNode.querySelector('#dragonfly-suggestion-thubmnail')
    ];
    for(var i = 0; i < elements.length; i++){
        if(elements[i]){
            elements[i].classList.add(elements[i].getAttribute('id'));
            elements[i].removeAttribute('id');
            elements[i].innerHTML = '';
        }
    }
}

/**
 * inserts result data inside template
 *
 * @since 1.0.0
 *
 * @param data
 * @param template
 * @returns string  HTML String
 */
function parseSuggestionTemplate(data, template){
    template.setAttribute('href', data.url);

    if(data.title){
        insertTextNode(template.querySelector('.dragonfly-suggestion-title'), data.title);
    }
    if(data.categories){
        insertTextNode(template.querySelector('.dragonfly-suggestion-cat'), data.categories);
    }
    if(data.price_html){
        insertTextNode(template.querySelector('.dragonfly-suggestion-price'), data.price_html);
    }
    if(data.sku){
        insertTextNode(template.querySelector('.dragonfly-suggestion-sku'), data.sku);
    }
    if(data.stock_status){
        insertTextNode(template.querySelector('.dragonfly-suggestion-stock-status'), data.stock_status);
    }

    var thumbnailEl = template.querySelector('.dragonfly-suggestion-thubmnail');
    if(data.thumbnail && thumbnailEl){
        thumbnailEl.innerHTML = '<img src="'  + data.thumbnail + '">';
    }else if(thumbnailEl){
        var newEl = document.createElement('a');
        newEl.setAttribute('class', 'dragonfly-result');
        newEl.setAttribute('href', data.url);
        var newContent = template.querySelector('.dragonfly-row.has-child-row');
        newContent.classList.remove('has-child-row');
        newEl.appendChild(newContent);

        clearEmptyRows(newEl);

        return newEl.outerHTML;
    }

    clearEmptyRows(template);

    return template.outerHTML;
}

/**
 * if node exist inserts text
 *
 * @param node
 * @param data
 */
function insertTextNode(node, data){
    if(node){
        node.innerHTML = data;
    }
}

/**
 * shows element
 *
 * @param element
 */
function show(element){
    element.style.display = 'block';
}

/**
 * hides element
 *
 * @param element
 */
function hide(element){
    element.style.display = 'none';
}

/**
 * removes empty rows from template
 *
 * @param suggetionEl
 */
function clearEmptyRows(suggetionEl){
    var row = suggetionEl.querySelectorAll('.row');

    for(var i = 0; i < row.length; i++){
        var col = row[i].querySelectorAll('.col');
        for(var i = 0; i < col.length; i++){
            if(col[i].querySelector('.component').childNodes.length === 0){
                col[i].parentNode.removeChild(col[i]);
            }
        }
    }
}