(function () {
    const BANNER_STORAGE_KEY = 'dpdpa_consent';

    function init() {
        const consent = localStorage.getItem(BANNER_STORAGE_KEY);
        if (!consent) {
            showBanner();
        } else {
            const consentData = JSON.parse(consent);
            applyConsent(consentData.purposes);
        }
    }

    function showBanner() {
        if (document.querySelector('.dpdpa-banner')) return;

        const banner = document.createElement('div');
        banner.className = 'dpdpa-banner active';
        const heading = typeof dpdpa_banner_vars !== 'undefined' ? dpdpa_banner_vars.heading : 'Cookie Consent (DPDPA)';
        const message = typeof dpdpa_banner_vars !== 'undefined' ? dpdpa_banner_vars.message : 'We use cookies and other tracking technologies to improve your browsing experience on our website...';

        banner.innerHTML = `
            <div>
                <h3>${heading}</h3>
                <p>${message}</p>
            </div>
            <div class="dpdpa-banner-footer">
                <button class="dpdpa-banner-btn dpdpa-banner-btn-secondary" id="dpdpa-reject">Reject All</button>
                <button class="dpdpa-banner-btn dpdpa-banner-btn-secondary" id="dpdpa-manage">Manage</button>
                <button class="dpdpa-banner-btn dpdpa-banner-btn-primary" id="dpdpa-accept">Accept All</button>
            </div>
        `;
        document.body.appendChild(banner);

        document.getElementById('dpdpa-accept').addEventListener('click', () => saveConsent(['analytics', 'marketing', 'personalization', 'essential'], 'accept'));
        document.getElementById('dpdpa-reject').addEventListener('click', () => saveConsent(['essential'], 'reject'));
        document.getElementById('dpdpa-manage').addEventListener('click', showPreferences);
    }

    function showPreferences() {
        const modal = document.createElement('div');
        modal.className = 'dpdpa-modal-overlay';

        const currentPurposes = window.DPDPAConsent ? window.DPDPAConsent.purposes : ['essential'];

        modal.innerHTML = `
            <div class="dpdpa-modal">
                <h2>Privacy Preferences</h2>
                <p>Manage how your data is processed. Essential cookies are required for the site to function.</p>
                
                <div class="dpdpa-purpose-list">
                    <div class="dpdpa-purpose-item">
                        <div class="dpdpa-purpose-info">
                            <label>Essential</label>
                            <span>Necessary for the website to function properly. Cannot be disabled.</span>
                        </div>
                        <label class="dpdpa-switch">
                            <input type="checkbox" checked disabled>
                            <span class="dpdpa-slider"></span>
                        </label>
                    </div>

                    <div class="dpdpa-purpose-item">
                        <div class="dpdpa-purpose-info">
                            <label>Analytics</label>
                            <span>Helps us understand how visitors interact with the website.</span>
                        </div>
                        <label class="dpdpa-switch">
                            <input type="checkbox" id="pref-analytics" ${currentPurposes.includes('analytics') ? 'checked' : ''}>
                            <span class="dpdpa-slider"></span>
                        </label>
                    </div>

                    <div class="dpdpa-purpose-item">
                        <div class="dpdpa-purpose-info">
                            <label>Marketing</label>
                            <span>Used to deliver relevant advertisements and track ad performance.</span>
                        </div>
                        <label class="dpdpa-switch">
                            <input type="checkbox" id="pref-marketing" ${currentPurposes.includes('marketing') ? 'checked' : ''}>
                            <span class="dpdpa-slider"></span>
                        </label>
                    </div>

                    <div class="dpdpa-purpose-item">
                        <div class="dpdpa-purpose-info">
                            <label>Personalization</label>
                            <span>Allows the site to remember choices you make and provide enhanced features.</span>
                        </div>
                        <label class="dpdpa-switch">
                            <input type="checkbox" id="pref-personalization" ${currentPurposes.includes('personalization') ? 'checked' : ''}>
                            <span class="dpdpa-slider"></span>
                        </label>
                    </div>
                </div>

                <div class="dpdpa-banner-footer">
                    <button class="dpdpa-banner-btn dpdpa-banner-btn-secondary" id="dpdpa-cancel-pref">Cancel</button>
                    <button class="dpdpa-banner-btn dpdpa-banner-btn-primary" id="dpdpa-save-pref">Save Preferences</button>
                </div>
            </div>
        `;

        document.body.appendChild(modal);

        document.getElementById('dpdpa-cancel-pref').addEventListener('click', () => modal.remove());
        document.getElementById('dpdpa-save-pref').addEventListener('click', () => {
            const selected = ['essential'];
            if (document.getElementById('pref-analytics').checked) selected.push('analytics');
            if (document.getElementById('pref-marketing').checked) selected.push('marketing');
            if (document.getElementById('pref-personalization').checked) selected.push('personalization');

            const isAll = selected.length === 4;
            const isNone = selected.length === 1;
            const decision = isAll ? 'accept' : (isNone ? 'reject' : 'partial');

            saveConsent(selected, decision);
            modal.remove();
        });
    }

    function saveConsent(purposes, decision) {
        const consentData = {
            timestamp: new Date().getTime(),
            purposes: purposes,
            decision: decision
        };
        localStorage.setItem(BANNER_STORAGE_KEY, JSON.stringify(consentData));
        document.querySelector('.dpdpa-banner').classList.remove('active');
        applyConsent(purposes);

        // Log consent to server
        logConsent(purposes, decision);
    }

    function applyConsent(purposes) {
        window.DPDPAConsent = {
            purposes: purposes,
            has: (p) => purposes.includes(p)
        };

        // Trigger event
        document.dispatchEvent(new CustomEvent('dpdpa_consent_updated', { detail: purposes }));

        // Enable scripts that were blocked
        document.querySelectorAll('script[data-dpdpa-purpose]').forEach(script => {
            const purpose = script.getAttribute('data-dpdpa-purpose');
            if (purposes.includes(purpose)) {
                if (script.type === 'text/plain') {
                    const newScript = document.createElement('script');
                    Array.from(script.attributes).forEach(attr => {
                        if (attr.name !== 'type' && attr.name !== 'data-dpdpa-purpose') {
                            newScript.setAttribute(attr.name, attr.value);
                        }
                    });
                    newScript.innerHTML = script.innerHTML;
                    script.parentNode.replaceChild(newScript, script);
                }
            }
        });
    }

    function logConsent(purposes, decision) {
        if (typeof dpdpa_vars === 'undefined') return;

        fetch(dpdpa_vars.ajaxurl, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'dpdpa_log_consent',
                decision: decision,
                purposes: JSON.stringify(purposes),
                nonce: dpdpa_vars.nonce
            })
        });
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
