/**
 * Version: 2.3.0
 */
var allps, pss;
let count = 0;
let colors = [];
var wpdm_pass_target = '#pps_z';
String.prototype.wpdm_shuffle = function () {
    var a = this.split(""),
        n = a.length;

    for (var i = n - 1; i > 0; i--) {
        var j = Math.floor(Math.random() * (i + 1));
        var tmp = a[i];
        a[i] = a[j];
        a[j] = tmp;
    }
    return a.join("");
}
String.prototype.wpdm_hash = function () {
    if (Array.prototype.reduce) {
        return this.split("").reduce(function (a, b) {
            a = ((a << 5) - a) + b.charCodeAt(0);
            return a & a
        }, 0);
    }
    var hash = 0;
    if (this.length === 0) return hash;
    for (var i = 0; i < this.length; i++) {
        var character = this.charCodeAt(i);
        hash = ((hash << 5) - hash) + character;
        hash = hash & hash;
    }
    return hash;
}

const WPDM = {

    actions: {},

    init: function ($) {

    },

    addAction: function (action, func) {
        if(!WPDM.actions[action]) WPDM.actions[action] = [];
        WPDM.actions[action].push(func);
    },

    doAction: async function (action, ...params) {
        if(typeof WPDM.actions[action] !== 'undefined')
            WPDM.actions[action].forEach(fn => fn(...params));
    },

    copy: function ($id) {
        var copyText = document.getElementById($id);
        copyText.select();
        copyText.setSelectionRange(0, 99999);
        document.execCommand("copy");
        WPDM.notify('<i class="fa fa-check-double"></i> Copied', 'success', 'top-center', 1000);
    },

    copyTxt: function (textToCopy) {
        WPDM.notify('<i class="fa fa-check-double"></i> Copied', 'success', 'top-center', 1000);
        // navigator clipboard api needs a secure context (https)
        if (navigator.clipboard && window.isSecureContext) {
            // navigator clipboard api method'
            return navigator.clipboard.writeText(textToCopy);
        } else {
            // text area method
            let textArea = document.createElement("textarea");
            textArea.value = textToCopy;
            // make the textarea out of viewport
            textArea.style.position = "fixed";
            textArea.style.left = "-999999px";
            textArea.style.top = "-999999px";
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            return new Promise((res, rej) => {
                // here the magic happens
                document.execCommand('copy') ? res() : rej();
                textArea.remove();
            });
        }
    },

    beep: function () {
        if (WPDM.audio == undefined)
            var snd = new Audio("data:audio/wav;base64,//uQRAAAAWMSLwUIYAAsYkXgoQwAEaYLWfkWgAI0wWs/ItAAAGDgYtAgAyN+QWaAAihwMWm4G8QQRDiMcCBcH3Cc+CDv/7xA4Tvh9Rz/y8QADBwMWgQAZG/ILNAARQ4GLTcDeIIIhxGOBAuD7hOfBB3/94gcJ3w+o5/5eIAIAAAVwWgQAVQ2ORaIQwEMAJiDg95G4nQL7mQVWI6GwRcfsZAcsKkJvxgxEjzFUgfHoSQ9Qq7KNwqHwuB13MA4a1q/DmBrHgPcmjiGoh//EwC5nGPEmS4RcfkVKOhJf+WOgoxJclFz3kgn//dBA+ya1GhurNn8zb//9NNutNuhz31f////9vt///z+IdAEAAAK4LQIAKobHItEIYCGAExBwe8jcToF9zIKrEdDYIuP2MgOWFSE34wYiR5iqQPj0JIeoVdlG4VD4XA67mAcNa1fhzA1jwHuTRxDUQ//iYBczjHiTJcIuPyKlHQkv/LHQUYkuSi57yQT//uggfZNajQ3Vmz+Zt//+mm3Wm3Q576v////+32///5/EOgAAADVghQAAAAA//uQZAUAB1WI0PZugAAAAAoQwAAAEk3nRd2qAAAAACiDgAAAAAAABCqEEQRLCgwpBGMlJkIz8jKhGvj4k6jzRnqasNKIeoh5gI7BJaC1A1AoNBjJgbyApVS4IDlZgDU5WUAxEKDNmmALHzZp0Fkz1FMTmGFl1FMEyodIavcCAUHDWrKAIA4aa2oCgILEBupZgHvAhEBcZ6joQBxS76AgccrFlczBvKLC0QI2cBoCFvfTDAo7eoOQInqDPBtvrDEZBNYN5xwNwxQRfw8ZQ5wQVLvO8OYU+mHvFLlDh05Mdg7BT6YrRPpCBznMB2r//xKJjyyOh+cImr2/4doscwD6neZjuZR4AgAABYAAAABy1xcdQtxYBYYZdifkUDgzzXaXn98Z0oi9ILU5mBjFANmRwlVJ3/6jYDAmxaiDG3/6xjQQCCKkRb/6kg/wW+kSJ5//rLobkLSiKmqP/0ikJuDaSaSf/6JiLYLEYnW/+kXg1WRVJL/9EmQ1YZIsv/6Qzwy5qk7/+tEU0nkls3/zIUMPKNX/6yZLf+kFgAfgGyLFAUwY//uQZAUABcd5UiNPVXAAAApAAAAAE0VZQKw9ISAAACgAAAAAVQIygIElVrFkBS+Jhi+EAuu+lKAkYUEIsmEAEoMeDmCETMvfSHTGkF5RWH7kz/ESHWPAq/kcCRhqBtMdokPdM7vil7RG98A2sc7zO6ZvTdM7pmOUAZTnJW+NXxqmd41dqJ6mLTXxrPpnV8avaIf5SvL7pndPvPpndJR9Kuu8fePvuiuhorgWjp7Mf/PRjxcFCPDkW31srioCExivv9lcwKEaHsf/7ow2Fl1T/9RkXgEhYElAoCLFtMArxwivDJJ+bR1HTKJdlEoTELCIqgEwVGSQ+hIm0NbK8WXcTEI0UPoa2NbG4y2K00JEWbZavJXkYaqo9CRHS55FcZTjKEk3NKoCYUnSQ0rWxrZbFKbKIhOKPZe1cJKzZSaQrIyULHDZmV5K4xySsDRKWOruanGtjLJXFEmwaIbDLX0hIPBUQPVFVkQkDoUNfSoDgQGKPekoxeGzA4DUvnn4bxzcZrtJyipKfPNy5w+9lnXwgqsiyHNeSVpemw4bWb9psYeq//uQZBoABQt4yMVxYAIAAAkQoAAAHvYpL5m6AAgAACXDAAAAD59jblTirQe9upFsmZbpMudy7Lz1X1DYsxOOSWpfPqNX2WqktK0DMvuGwlbNj44TleLPQ+Gsfb+GOWOKJoIrWb3cIMeeON6lz2umTqMXV8Mj30yWPpjoSa9ujK8SyeJP5y5mOW1D6hvLepeveEAEDo0mgCRClOEgANv3B9a6fikgUSu/DmAMATrGx7nng5p5iimPNZsfQLYB2sDLIkzRKZOHGAaUyDcpFBSLG9MCQALgAIgQs2YunOszLSAyQYPVC2YdGGeHD2dTdJk1pAHGAWDjnkcLKFymS3RQZTInzySoBwMG0QueC3gMsCEYxUqlrcxK6k1LQQcsmyYeQPdC2YfuGPASCBkcVMQQqpVJshui1tkXQJQV0OXGAZMXSOEEBRirXbVRQW7ugq7IM7rPWSZyDlM3IuNEkxzCOJ0ny2ThNkyRai1b6ev//3dzNGzNb//4uAvHT5sURcZCFcuKLhOFs8mLAAEAt4UWAAIABAAAAAB4qbHo0tIjVkUU//uQZAwABfSFz3ZqQAAAAAngwAAAE1HjMp2qAAAAACZDgAAAD5UkTE1UgZEUExqYynN1qZvqIOREEFmBcJQkwdxiFtw0qEOkGYfRDifBui9MQg4QAHAqWtAWHoCxu1Yf4VfWLPIM2mHDFsbQEVGwyqQoQcwnfHeIkNt9YnkiaS1oizycqJrx4KOQjahZxWbcZgztj2c49nKmkId44S71j0c8eV9yDK6uPRzx5X18eDvjvQ6yKo9ZSS6l//8elePK/Lf//IInrOF/FvDoADYAGBMGb7FtErm5MXMlmPAJQVgWta7Zx2go+8xJ0UiCb8LHHdftWyLJE0QIAIsI+UbXu67dZMjmgDGCGl1H+vpF4NSDckSIkk7Vd+sxEhBQMRU8j/12UIRhzSaUdQ+rQU5kGeFxm+hb1oh6pWWmv3uvmReDl0UnvtapVaIzo1jZbf/pD6ElLqSX+rUmOQNpJFa/r+sa4e/pBlAABoAAAAA3CUgShLdGIxsY7AUABPRrgCABdDuQ5GC7DqPQCgbbJUAoRSUj+NIEig0YfyWUho1VBBBA//uQZB4ABZx5zfMakeAAAAmwAAAAF5F3P0w9GtAAACfAAAAAwLhMDmAYWMgVEG1U0FIGCBgXBXAtfMH10000EEEEEECUBYln03TTTdNBDZopopYvrTTdNa325mImNg3TTPV9q3pmY0xoO6bv3r00y+IDGid/9aaaZTGMuj9mpu9Mpio1dXrr5HERTZSmqU36A3CumzN/9Robv/Xx4v9ijkSRSNLQhAWumap82WRSBUqXStV/YcS+XVLnSS+WLDroqArFkMEsAS+eWmrUzrO0oEmE40RlMZ5+ODIkAyKAGUwZ3mVKmcamcJnMW26MRPgUw6j+LkhyHGVGYjSUUKNpuJUQoOIAyDvEyG8S5yfK6dhZc0Tx1KI/gviKL6qvvFs1+bWtaz58uUNnryq6kt5RzOCkPWlVqVX2a/EEBUdU1KrXLf40GoiiFXK///qpoiDXrOgqDR38JB0bw7SoL+ZB9o1RCkQjQ2CBYZKd/+VJxZRRZlqSkKiws0WFxUyCwsKiMy7hUVFhIaCrNQsKkTIsLivwKKigsj8XYlwt/WKi2N4d//uQRCSAAjURNIHpMZBGYiaQPSYyAAABLAAAAAAAACWAAAAApUF/Mg+0aohSIRobBAsMlO//Kk4soosy1JSFRYWaLC4qZBYWFRGZdwqKiwkNBVmoWFSJkWFxX4FFRQWR+LsS4W/rFRb/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////VEFHAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAU291bmRib3kuZGUAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMjAwNGh0dHA6Ly93d3cuc291bmRib3kuZGUAAAAAAAAAACU=");
        else
            var snd = new Audio(WPDM.audio);
        snd.play();
    },

    hash: function (str) {
        return str.wpdm_hash();
    },

    uniqueID: function (prefix) {
        var uniq = Date.now() + "abcdefghijklmnopqrstuvwxyz_";
        uniq = uniq.wpdm_shuffle();
        uniq = uniq.substring(1, 10);
        uniq = typeof prefix !== 'undefined' ? prefix + uniq : uniq;
        return uniq;
    },

    fileTypeIcon: function (ext) {
        //let colors = color.split('|');
        if(!colors[ext]) {
            let color1 = "58"+ ("95BB".wpdm_shuffle());
            let color2 = "56789A".wpdm_shuffle();
            colors[ext] = [color1, color2];
        }
        count++;
        return `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 40 40" style="margin: 4px">
                <defs>
                    <linearGradient id="gradient${count}" x1="0" y1="0" x2="0" y2="1">
                        <stop stop-color="#${colors[ext][0]}" offset="0"/>
                        <stop stop-color="var(--color-primary)" stop-opacity="0.7" offset="1"/>
                    </linearGradient>
                </defs>
                <g>
                    <rect fill="url(#gradient${count})" x="0" y="0" width="40" height="40" rx="3" ry="3"/>
                    <text x="5" y="19" font-family="Arial, Helvetica, sans-serif" font-size="11px" letter-spacing="1" font-weight="600" text-rendering="geometricPrecision" fill="#FFFFFF">
                        <tspan>${ext}</tspan>
                        <tspan x="6" y="28">_</tspan>
                    </text>
                </g>
            </svg>`;
    },

    popupWindow: function (url, title, w, h, onclose) {
        /* Fixes dual-screen position                         Most browsers      Firefox */
        var dualScreenLeft = typeof window.screenLeft !== 'undefined' ? window.screenLeft : screen.left;
        var dualScreenTop = typeof window.screenTop !== 'undefined' ? window.screenTop : screen.top;

        var width = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
        var height = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;

        var left = ((width / 2) - (w / 2)) + dualScreenLeft;
        var top = ((height / 2) - (h / 2)) + dualScreenTop;
        var newWindow = window.open(url, title, 'scrollbars=yes, width=' + w + ', height=' + h + ', top=' + top + ', left=' + left);

        /* Puts focus on the newWindow */
        if (window.focus) {
            newWindow.focus();
        }
        if (onclose !== undefined)
            newWindow.onbeforeunload = function () {
                onclose.call();
            }
        return false;
    },

    html: function (elment, html, _class, _id) {
        _class = _class !== undefined ? _class : '';
        _id = _id !== undefined ? _id : '';
        return "<" + elment + " class='" + _class + "' id='" + _id + "'>" + html + "</" + elment + ">";
    },

    el: function (element, attrs, innerHTML) {
        var el = document.createElement(element);
        el = jQuery(el);
        jQuery.each(attrs, function (name, val) {
            el.attr(name, val);
        })
        if (typeof innerHTML !== 'undefined' && innerHTML !== '')
            el.html(innerHTML);
        return el[0].outerHTML;
    },

    bootAlert: function (heading, content, width, backdrop, options) {
        let html, url = '';
        let modal_id = '__bootModal_' + this.uniqueID();
        options = options || {};

        // Handle URL-based content loading
        if (typeof content === 'object') {
            return WPDM.dialog.ajax(heading, content.url, options);
        }
        return WPDM.dialog.alert(heading, content, options);
    },

    /**
     * Local push notification
     * @param title
     * @param message
     * @param icon
     * @param image
     * @param url
     */
    pushNotify: function (title, message, icon, image, url) {
        var type = 'info';
        if (title.includes('rror') || message.includes('rror')) type = 'error';

        if (!('Notification' in window) || !('ServiceWorkerRegistration' in window)) {
            WPDM.notify("<strong>" + title + "</strong><br/>" + message, type, 'top-right');
            return;
        }

        Notification.requestPermission(function (result) {
            if (result === 'granted') {
                console.log('Notification: ' + result);
                try {
                    var notification = new Notification(title, {
                        body: message,
                        icon: icon ? icon : 'https://cdn1.iconfinder.com/data/icons/hawcons/32/698558-icon-47-note-important-512.png',
                        image: image ? image : ''
                    });
                    if (url) {
                        notification.onclick = function (e) {
                            e.preventDefault();
                            window.open(url, '_blank');
                        };
                    }
                } catch (err) {
                    WPDM.notify("<strong>" + title + "</strong><br/>" + message, type, 'top-right');
                    console.log('Notification API error: ' + err);
                }
            } else {
                WPDM.notify("<strong>" + title + "</strong><br/>" + message, type, 'top-right');
                console.log('Notification: ' + result);
            }
        });

    },


    /**
     * Shows notification
     * @param message
     * @param type
     * @param position
     */
    notify: function (message, type, position, autoclose) {
        WPDM.toast(message, type, position, autoclose);
    },

    /**
     * Shows notification
     * @param message
     * @param type
     * @param position
     */
    floatify: function (html, position) {
        WPDM.toast(html, "info", position);
    },

    toast: function (message, type, position, autoclose) {
        var $ = jQuery;
        if (!type) type = 'info';
        if (!position) position = 'bottom-right';
        if (autoclose === undefined) autoclose = 4000;
        if (type === 'danger') type = 'error';

        var containerId = 'wpdm-toast-' + position;
        if ($('#' + containerId).length === 0)
            $('body').append("<div class='w3eden' id='" + containerId + "' class='wpdm-toast-container'></div>");

        var icons = {
            success: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><path d="M20 6L9 17l-5-5"/></svg>',
            error: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>',
            warning: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/><line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>',
            info: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/></svg>'
        };

        var icon = icons[type] || icons.info;
        var toast = $("<div class='wpdm-toast wpdm-toast-" + type + "'>" +
            "<span class='wpdm-toast-icon'>" + icon + "</span>" +
            "<span class='wpdm-toast-msg'>" + message + "</span>" +
            "<button class='wpdm-toast-close'>&times;</button>" +
            "</div>");

        $('#' + containerId).append(toast);
        setTimeout(function () { toast.addClass('wpdm-toast-show'); }, 10);

        toast.find('.wpdm-toast-close').on('click', function () { dismiss(); });

        var timer;
        if (autoclose > 0) {
            timer = setTimeout(dismiss, autoclose);
            toast.on('mouseenter', function () { clearTimeout(timer); });
            toast.on('mouseleave', function () { timer = setTimeout(dismiss, 1500); });
        }

        function dismiss() {
            clearTimeout(timer);
            toast.removeClass('wpdm-toast-show');
            setTimeout(function () { toast.remove(); }, 300);
        }

        return toast;
    },

    blockUI: function (element, xhr) {
        jQuery(element).addClass("blockui");
        if (xhr)
            xhr.addEventListener("load", function () {
                jQuery(element).removeClass("blockui");
            });
    },

    unblockUI: function (element) {
        if (typeof element === 'undefined') element = '.blockui';
        jQuery(element).removeClass("blockui");
    },


    confirm: function (heading, content, buttons, width) {
        var html, $ = jQuery;
        var modal_id = '__boot_popup_' + WPDM.uniqueID();
        $("#w3eden__boot_popup").remove();
        if (!width) width = 350;
        var _buttons = '';
        if (buttons) {
            _buttons = '<div class="modal-footer text-center" style="padding: 8px 15px;justify-content: center;">\n';
            $.each(buttons, function (i, button) {
                var btnid = WPDM.uniqueID();
                _buttons += "<button id='" + btnid + "' class='" + button.class + " btn-xs' style='font-size: 12px;padding: 4px 16px;border-radius: 4px'>" + button.label + "</button> ";
                $('body').on('click', '#' + btnid, function () {
                    button.callback.call($("#" + modal_id));
                    return false;
                });
            });
            _buttons += '</div>\n';
        }

        html = '<div class="w3eden" id="w3eden' + modal_id + '"><div id="' + modal_id + '" style="z-index: 9999999 !important;" class="modal fade" tabindex="-1" role="dialog">\n' +
            '  <div class="modal-dialog wpdm-modal-confirm modal-dialog-centered" role="document" style="max-width: 100%;width: ' + width + 'px">\n' +
            '    <div class="modal-content" style="border-radius: 6px;overflow: hidden">\n' +
            '      <div class="modal-header" style="padding: 12px 15px;background: #f5f5f5;">\n' +
            '        <h4 class="modal-title" style="font-size: 12pt;font-weight: 500;padding: 0;margin: 0;font-family:var(--wpdm-font), san-serif;letter-spacing: 0.5px">' + heading + '</h4>\n' +
            '      </div>\n' +
            '      <div class="modal-body text-center" style="font-family:var(--wpdm-font), san-serif;letter-spacing: 0.5px;font-size: 10pt;font-weight: 300;padding: 25px;line-height: 1.5">\n' +
            '        ' + content + '\n' +
            '      </div>\n' + _buttons +
            '    </div>\n' +
            '  </div>\n' +
            '</div></div>';
        $('body').append(html);
        $("#" + modal_id).modal('show');
        return $("#" + modal_id);
    },
    audioUI: function (audio) {
        var $ = jQuery, song_length, song_length_m, song_length_s;

        var player_html = `<div class="w3eden"><div style="display: none" class="wdmapui" id="wdmapui"><div id="wpdm_close_player"><svg style="width: 20px" data-name="Livello 1" viewBox="0 0 151.57 151.57" xmlns="http://www.w3.org/2000/svg"><circle cx="1038.5" cy="467.01" r="72.28" style="fill:#da2244;stroke:#f2f2f2;stroke-linecap:round;stroke-linejoin:round;stroke-width:7px" transform="translate(-988.78 479.89) rotate(-45)"/><line style="fill:#da2244;stroke:#f2f2f2;stroke-linecap:round;stroke-linejoin:round;stroke-width:7px" x1="47.57" x2="103.99" y1="103.99" y2="47.57"/><line style="fill:#da2244;stroke:#f2f2f2;stroke-linecap:round;stroke-linejoin:round;stroke-width:7px" x1="45.8" x2="105.7" y1="45.87" y2="105.77"/></svg></div><div class="card m-2"><div class="card-body text-center"><div class="mb-3 wpdm-audio-control-buttons d-block d-sm-none"><button class="btn btn-primary btn-play" id="wpdm-btn-play"><i class="fa fa-play"></i></button> <button class="btn btn-secondary btn-backward" id="wpdm-btn-backward"><i class="fa fa-backward"></i></button> <button class="btn btn-secondary btn-forward" id="wpdm-btn-forward"><i class="fa fa-forward"></i></button> <button class="btn btn-info btn-volumctrl" id="wpdm-btn-volumctrl"><i class="fa fa-volume-up"></i></button></div><div class="media"><div class="mr-3 wpdm-audio-control-buttons d-none d-sm-block"><button class="btn btn-primary btn-play" id="wpdm-btn-play"><i class="fa fa-play"></i></button> <button class="btn btn-secondary btn-backward" id="wpdm-btn-backward"><i class="fa fa-backward"></i></button> <button class="btn btn-secondary btn-forward" id="wpdm-btn-forward"><i class="fa fa-forward"></i></button></div><div class="media-body"><div class="position-relative"><div id="played">00:00</div><div id="mins">00:00</div></div><div class="progress"><div  id="wpdm-audio-progress" class="progress-bar bg-success" role="progressbar" style="width: 0%;" aria-valuenow="25" aria-valuemin="0" aria-valuemax="100"></div></div></div><div class="ml-3 wpdm-audio-control-buttons  d-none d-sm-block"> <button class="btn btn-info btn-volumctrl" id="wpdm-btn-volumctrl"><i class="fa fa-volume-up"></i></button> <div class="volumctrl"><input type="range" min="0" max="1" step="0.01" value="3" class="p-0" id="wpdm-audio-volume"></div></div></div></div></div></div></div>`;

        if (audio.duration !== Infinity) {
            song_length = parseInt(audio.duration);
            song_length_m = parseInt(song_length / 60);
            song_length_s = song_length - (song_length_m * 60);
            song_length_m = song_length_m > 9 ? song_length_m : "0" + song_length_m;
            song_length_s = song_length_s > 9 ? song_length_s : "0" + song_length_s;
        } else {
            song_length_m = "--";
            song_length_s = "--";
            audio.addEventListener("durationchange", function (e) {
                console.log(this.duration);
                if (this.duration !== Infinity) {
                    song_length = parseInt(this.duration);
                    song_length_m = parseInt(song_length / 60);
                    song_length_s = song_length - (song_length_m * 60);
                    song_length_m = song_length_m > 9 ? song_length_m : "0" + song_length_m;
                    song_length_s = song_length_s > 9 ? song_length_s : "0" + song_length_s;
                    $('#mins').html(song_length_m + ":" + song_length_s);
                    next(song_length);
                }
            }, false);
        }

        if ($('#wdmapui').length === 0) {
            $('body').append(player_html);
            $("#wdmapui").slideDown();
            $('#mins').html(song_length_m + ":" + song_length_s);
            $('body').on('click', '#wdmapui .progress', function (e) {
                let value = e.offsetX * 100 / this.clientWidth;
                var played = parseInt(song_length * value / 100);
                var played_m = parseInt(played / 60);
                var played_s = played - (played_m * 60);
                played_m = played_m > 9 ? played_m : "0" + played_m;
                played_s = played_s > 9 ? played_s : "0" + played_s;
                $('#played').html(played_m + ":" + played_s);
                audio.currentTime = (song_length * value) / 100;
                $(this).find('.progress-bar').css('width', value + "%");
                //video.currentTime = duration * (value / 100);
            });
            $('body').on('click', '.btn-backward', function () {
                let value = (parseInt($('#wdmapui .progress-bar').css('width')) / parseInt($('#wdmapui .progress').css('width'))) * 100 - 10;
                if (value < 0) value = 0;
                var played = parseInt(song_length * value / 100);
                var played_m = parseInt(played / 60);
                var played_s = played - (played_m * 60);
                played_m = played_m > 9 ? played_m : "0" + played_m;
                played_s = played_s > 9 ? played_s : "0" + played_s;
                $('#played').html(played_m + ":" + played_s);
                audio.currentTime = (song_length * value) / 100;
                $('#wdmapui .progress-bar').css('width', value + "%");
            });
            $('body').on('click', '.btn-forward', function () {
                let value = (parseInt($('#wdmapui .progress-bar').css('width')) / parseInt($('#wdmapui .progress').css('width'))) * 100 + 10;
                if (value > 100) value = 100;
                var played = parseInt(song_length * value / 100);
                var played_m = parseInt(played / 60);
                var played_s = played - (played_m * 60);
                played_m = played_m > 9 ? played_m : "0" + played_m;
                played_s = played_s > 9 ? played_s : "0" + played_s;
                $('#played').html(played_m + ":" + played_s);
                audio.currentTime = (song_length * value) / 100;
                $('#wdmapui .progress-bar').css('width', value + "%");
            });
            $('body').on('click', '.btn-volumctrl', function () {
                $(this).next('.volumctrl').toggle();
            });
            $('body').on('click', '.btn-play', function () {
                if ($(this).find('.fa').hasClass('fa-play')) {
                    $(this).find('.fa').addClass('fa-pause').removeClass('fa-play');
                    $(this).data('state', 'playing');
                    audio.play();
                } else {
                    $(this).find('.fa').addClass('fa-play').removeClass('fa-pause');
                    $(this).data('state', 'paused');
                    audio.pause();
                }
            });
            $('body').on('change', '#wpdm-audio-volume', function () {
                audio.volume = this.value;
            });

        }
        $('#mins').html(song_length_m + ":" + song_length_s);
        audio.addEventListener("play", function () {
            $('.btn-play .fa').addClass('fa-pause').removeClass('fa-play');
        });
        audio.addEventListener("pause", function () {
            $('.btn-play .fa').addClass('fa-play').removeClass('fa-pause');
        });
        audio.addEventListener("timeupdate", function (e) {
            var song_length = parseInt(audio.duration);
            var time_now = audio.currentTime;
            var percent = (time_now / song_length) * 100;
            if (percent > 100) percent = 100;
            $('#wpdm-audio-progress').css('width', percent + "%");
            var played = parseInt(time_now);
            var played_m = parseInt(played / 60);
            var played_s = played - (played_m * 60);
            played_m = played_m > 9 ? played_m : "0" + played_m;
            played_s = played_s > 9 ? played_s : "0" + played_s;
            $('#played').html(played_m + ":" + played_s);
        });


    }

};


jQuery(function ($) {

    var $body = $('body');

    const saves_state = localStorage.getItem('page_state_' + wpdm_url.page_code) || null;
    const container = localStorage.getItem('page_state_container_' + wpdm_url.page_code) || null;
    if(saves_state && container) {
        $.get(saves_state, function (res) {
            $(container).html($(res).find(container).html());
            $(container).removeClass('blockui');
        });
    }

    $body.on('click', '.wpdm-notify, .wpdm-floatify', function () {
        $(this).animate({
            opacity: 0
        }, 1000, function () {
            $(this).slideUp();
        });
    });

    $body.on('click', '.dismis-on-click', function () {
        $(this).slideUp();
    });

    $body.on('click', '.wpdm-download-link.wpdm-download-locked', function (e) {
        e.preventDefault();
        hideLockFrame();

        var parentWindow = document.createElement("a");
        parentWindow.href = document.referrer.toString();
        var __sep = '?';
        if (wpdm_url.home.indexOf('?') > 0) __sep = '&';
        let extras = '';
        if ($(this).data('file') !== undefined) extras += '__wpdmfl=' + $(this).data('file');
        extras += '&REFERRER=' + encodeURI(location.href);
        if (parentWindow.hostname === window.location.hostname || 1)
            $(window.parent.document.body).append("<iframe id='wpdm-lock-frame' style='left:0;top:0;width: 100%;height: 100%;z-index: 999999999;position: fixed;background: rgba(255,255,255,0.4) url(" + wpdm_url.home + "wp-content/plugins/download-manager/assets/images/loader.svg) center center no-repeat;background-size: 80px 80px;border: 0;' src='" + wpdm_url.home + __sep + "__wpdmlo=" + $(this).data('package') + "&" + extras + "'></iframe>");
        else
            window.parent.postMessage({
                'task': 'showiframe',
                'iframe': "<iframe id='wpdm-lock-frame' style='left:0;top:0;width: 100%;height: 100%;z-index: 999999999;position: fixed;background: rgba(255,255,255,0.4) url(" + wpdm_url.home + "wp-content/plugins/download-manager/assets/images/loader.svg) center center no-repeat;background-size: 80px 80px;border: 0;' src='" + wpdm_url.home + __sep + "__wpdmlo=" + $(this).data('package') + __sep + "__wpdmfl=" +  + "'></iframe>"
            }, "*");

    });

    $body.on('click', '.wpdm-download-link.download-on-click[data-downloadurl]', function (e) {
        e.preventDefault();
        if (this.target === '_blank')
            window.open($(this).data('downloadurl'));
        else
            window.location.href = $(this).data('downloadurl');
    });

    $body.on('click', '.__wpdm_playvideo', function (e) {
        e.preventDefault();
        $('#__wpdm_videoplayer').children('source').attr('src', $(this).data('video'));
        console.log('loading...');
        var vid = document.getElementById("__wpdm_videoplayer");
        vid.onloadeddata = function () {
            console.log('loaded....');
        };
        $("#__wpdm_videoplayer").get(0).load();

    });

    $body.on('change', '.terms_checkbox', function (e) {
        if ($(this).is(':checked'))
            $('#wpdm-filelist-' + $(this).data('pid') + ' .btn.inddl, #xfilelist .btn.inddl').removeAttr('disabled');
        else
            $('#wpdm-filelist-' + $(this).data('pid') + ' .btn.inddl, #xfilelist .btn.inddl').attr('disabled', 'disabled');
    });

    $body.on('click', '.wpdm-social-lock', function (e) {

        try {

            if ($(this).data('url'))
                _PopupCenter($(this).data('url'), 'Social Lock', 600, 400);

        } catch (e) {
        }

    });

    $body.on('click', '#wdmds a.list-group-item', function (e) {
        location.href = this.href;
    });

    var $input_group_input = $('.input-group input');
    $input_group_input.on('focus', function () {
        $(this).parent('.input-group').find('.input-group-addon').addClass('input-group-addon-active');
    });
    $input_group_input.on('blur', function () {
        $(this).parent().find('.input-group-addon').removeClass('input-group-addon-active');
    });


    $body.on('click', 'button.btn.inddl', function (e) {
        e.preventDefault();
        var tis = this;
        if ($(this).data('dlurl') !== undefined) {
            location.href = $(this).data('dlurl');
            return;
        }
        $.post(wpdm_rest_url('validate-filepass'), {
            wpdmfileid: $(tis).data('pid'),
            wpdmfile: $(tis).data('file'),
            actioninddlpvr: 1,
            filepass: $($(tis).data('pass')).val()
        }, function (res) {
            if (res.success === true) {
                var dlurl = res.downloadurl;
                $(tis).data('dlurl', dlurl);
                wpdm_boot_popup(wpdm_strings.pass_var, "<div style='padding: 50px;'>"+wpdm_strings.pass_var_q+"<br/><br/><a href='" + dlurl + "' class='btn btn-lg btn-success' target='_blank'>"+wpdm_strings.start_dl+"</a></div>",
                    [{
                        label: 'Close',
                        class: 'btn btn-secondary',
                        callback: function () {
                            $('#__boot_popup').modal('hide');
                            return false;
                        }
                    }]
                );
            } else {
                alert(res.msg);
            }
        });
    });

    $body.on('click', '.wpdm-indir', function (e) {
        e.preventDefault();
        WPDM.blockUI('#xfilelist');
        $('#xfilelist').load(location.href, {
            action: 'wpdmfilelistcd',
            pid: $(this).data('pid'),
            cd: $(this).data('dir')
        }, function (res) {
            WPDM.unblockUI('#xfilelist');
        });
    });


    $body.on('click', '.role-tabs a', function (e) {
        $('.role-tabs a').removeClass('active');
        $(this).addClass('active');
    });


    $body.on('click', '.btn-wpdm-a2f', function (e) {
        var a2fbtn = $(this);
        $.post(wpdm_url.ajax, {action: 'wpdm_addtofav', pid: $(this).data('package')}, function (res) {
            if (a2fbtn.hasClass('btn-secondary'))
                a2fbtn.removeClass('btn-secondary').addClass('btn-danger').html(a2fbtn.data('rlabel'));
            else
                a2fbtn.removeClass('btn-danger').addClass('btn-secondary').html(a2fbtn.data('alabel'));
        });
    });

    $body.on('click', '#wpdm_close_player', function (e) {
        $('#wdmapui').slideUp();
    });

    $body.on('click', '.wpdm-btn-play', function (e) {
        e.preventDefault();

        if ($('#wpdm-audio-player').length === 0) {
            var player = document.createElement('audio');
            player.id = 'wpdm-audio-player';
            player.controls = 'controls';
            player.autoplay = 1;
            player.type = 'audio/mpeg';
            $('body').append(player);
        }

        player = $('#wpdm-audio-player');
        var btn = $(this);

        if (btn.data('state') === 'stop' || !btn.data('state')) {
            player.css('display', 'none');
            player.attr('src', $(this).data('song') + "&play=song.mp3");
        }

        if (btn.data('state') === 'playing') {
            btn.data('state', 'paused');
            player.trigger('pause');
            $(this).html("<i class='fa fa-play'></i>");
            return false;
        }

        if (btn.data('state') === 'paused') {
            $(this).data('state', 'playing');
            player.trigger('play');
            $('.wpdm-btn-play').html("<i class='fa fa-play'></i>");
            $(this).html("<i class='fa fa-pause'></i>");
            $('#wdmapui').slideDown();
            return false;
        }


        $('.wpdm-btn-play').data("state", "stop");
        $('.wpdm-btn-play').html("<i class='fa fa-play'></i>");
        btn.html("<i class='fas fa-sun  fa-spin'></i>");
        player.unbind('loadedmetadata');
        player.on('loadedmetadata', function () {
            console.log("Playing " + this.src + ", for: " + this.duration + "seconds.");
            btn.html("<i class='fa fa-pause'></i>");
            btn.data('state', 'playing');
            WPDM.audioUI(this);
        });

        document.getElementById('wpdm-audio-player').onended = function () {
            btn.html("<i class='fa fa-redo'></i>");
            btn.data('state', 'stop');
        }
    });

    $('.wpdm_remove_empty').remove();

    /* Uploading files */
    var file_frame, dfield;

    $body.on('click', '.wpdm-media-upload', function (event) {
        event.preventDefault();
        dfield = $($(this).attr('rel'));

        /* If the media frame already exists, reopen it. */
        if (file_frame) {
            file_frame.open();
            return;
        }

        /* Create the media frame. */
        file_frame = wp.media.frames.file_frame = wp.media({
            title: $(this).data('uploader_title'),
            button: {
                text: $(this).data('uploader_button_text')
            },
            multiple: false  /* Set to true to allow multiple files to be selected */
        });

        /* When an image is selected, run a callback. */
        file_frame.on('select', function () {
            /* We set multiple to false so only get one image from the uploader */
            attachment = file_frame.state().get('selection').first().toJSON();
            dfield.val(attachment.url).trigger('change');
        });

        /* Finally, open the modal */
        file_frame.open();
    });

    $body.on('click', '.btn-image-selector', function (event) {
        event.preventDefault();
        dfield = $($(this).attr('rel'));
        var dfield_h = $($(this).attr('rel') + '_hidden');

        if (file_frame) {
            file_frame.open();
            return;
        }

        file_frame = wp.media.frames.file_frame = wp.media({
            title: $(this).data('uploader_title'),
            button: {
                text: $(this).data('uploader_button_text')
            },
            multiple: false
        });


        file_frame.on('select', function () {

            attachment = file_frame.state().get('selection').first().toJSON();
            dfield.attr('src', attachment.url);
            dfield_h.val(attachment.url);

        });

        file_frame.open();
    });

    $body.on('click', '.pagination.async a, .__wpdm_load_async', function (e) {
        e.preventDefault();
        var _cont = $(this).data('container');
        var href = this.href;
        $(_cont).addClass('blockui');
        localStorage.setItem('page_state_' + wpdm_url.page_code, href);
        localStorage.setItem('page_state_container_' + wpdm_url.page_code, _cont);
        $.get(href, function (res) {
            $(_cont).html($(res).find(_cont).html());
            $(_cont).removeClass('blockui');
        });
    });

    $body.on("keyup", '.wpdm-pack-search-file', function () {
        var value = $(this).val().toLowerCase();
        var filelist_item = $(this).data('filelist') + " tr";
        $(filelist_item).filter(function () {
            $(this).toggle($(this).text().toLowerCase().indexOf(value) > -1)
        });
    });

    /* ========================================
       FILE LIST GRID - Layout Toggle & Search
       ======================================== */

    // Layout toggle (Grid/List view)
    $(document).on('click', '.wpdm-filelist-toggle__btn', function(e) {
        e.preventDefault();
        var $btn = $(this);
        var $toggle = $btn.closest('.wpdm-filelist-toggle');
        var targetSelector = $toggle.data('target');
        var $grid = $(targetSelector);
        var layout = $btn.data('layout');

        // Update active button
        $toggle.find('.wpdm-filelist-toggle__btn').removeClass('active');
        $btn.addClass('active');

        // Toggle layout class
        if (layout === 'list') {
            $grid.addClass('wpdm-filelist-grid--list');
        } else {
            $grid.removeClass('wpdm-filelist-grid--list');
        }

        // Remember preference in localStorage
        try {
            localStorage.setItem('wpdm_filelist_layout', layout);
        } catch (e) {}
    });

    // Restore layout preference on page load
    $(function() {
        try {
            var savedLayout = localStorage.getItem('wpdm_filelist_layout');
            if (savedLayout === 'list') {
                $('.wpdm-filelist-toggle').each(function() {
                    var $toggle = $(this);
                    var targetSelector = $toggle.data('target');
                    var $grid = $(targetSelector);

                    $toggle.find('.wpdm-filelist-toggle__btn').removeClass('active');
                    $toggle.find('[data-layout="list"]').addClass('active');
                    $grid.addClass('wpdm-filelist-grid--list');
                });
            }
        } catch (e) {}
    });

    // Search for new grid-based file list and gallery
    $(document).on('input', '.wpdm-filelist-search__input', function() {
        var value = $(this).val().toLowerCase().trim();
        var targetSelector = $(this).data('filelist');
        var $items = $(targetSelector).find('.wpdm-filelist-item, .wpdm-gallery__item');

        if (value === '') {
            $items.removeClass('wpdm-filelist-hidden');
            return;
        }

        $items.each(function() {
            var $item = $(this);
            var filename = $item.data('filename') || '';
            var titleText = $item.find('.wpdm-filelist-item__title, .wpdm-gallery__title').text().toLowerCase();
            var matches = filename.indexOf(value) > -1 || titleText.indexOf(value) > -1;

            if (matches) {
                $item.removeClass('wpdm-filelist-hidden');
            } else {
                $item.addClass('wpdm-filelist-hidden');
            }
        });
    });

    // Clear search on Escape key
    $body.on('keydown', '.wpdm-filelist-search__input', function(e) {
        if (e.key === 'Escape') {
            $(this).val('').trigger('input');
        }
    });

    /* FILE LIST GRID ENDS */

    $('.__wpdm_submit_async').on('submit', function (e) {
        e.preventDefault();
        var _cont = $(this).data('container');
        $(_cont).addClass('blockui');
        $(this).ajaxSubmit({
            success: function (response) {
                $(_cont).html($(response).find(_cont).html());
                $(_cont).removeClass('blockui');
            }
        })
    });


    var unlocked = [];

    $('.wpdm-filelist-area[data-termlock]').on('mouseover', function () {

        try {
            if (unlocked[$(this).data('packageid')] === 1) return;
        } catch (e) {

        }
        $('#term-panel-' + $(this).data('packageid')).fadeIn('fast');
    });

    $('.terms_checkbox').on('click', function () {
        if ($(this).is(':checked')) {
            unlocked[$(this).data('pid')] = 1;
            $('#term-panel-' + $(this).data('pid')).fadeOut('fast');
            $('.download_footer_' + $(this).data('pid')).slideDown();
        } else
            $('.download_footer_' + $(this).data('pid')).slideUp();
    });


    $formcontrol = $('.input-wrapper input');
    $formcontrol.on('focus', function () {
        $('.input-wrapper').removeClass('input-focused');
        $(this).parent('.input-wrapper').addClass('input-focused');
    });
    $formcontrol.on('change', function () {
        $('.input-wrapper').removeClass('input-focused');
        $(this).parent('.input-wrapper').addClass('input-focused');
        if ($(this).val() !== '')
            $(this).parent('.input-wrapper').addClass('input-withvalue');
        else
            $(this).parent('.input-wrapper').removeClass('input-withvalue');
    });


});


function _PopupCenter(url, title, w, h) {
    /* Fixes dual-screen position                         Most browsers      Firefox */
    var dualScreenLeft = window.screenLeft != undefined ? window.screenLeft : screen.left;
    var dualScreenTop = window.screenTop != undefined ? window.screenTop : screen.top;

    var width = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
    var height = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;

    var left = ((width / 2) - (w / 2)) + dualScreenLeft;
    var top = ((height / 2) - (h / 2)) + dualScreenTop;
    var newWindow = window.open(url, title, 'scrollbars=yes, width=' + w + ', height=' + h + ', top=' + top + ', left=' + left);

    /* Puts focus on the newWindow */
    if (window.focus) {
        newWindow.focus();
    }

    return false;
}

function generatepass(id) {
    wpdm_pass_target = '#' + id;
    jQuery('#generatepass').modal('show');
}

function hideLockFrame() {
    jQuery('#wpdm-lock-frame').remove();
}

function wpdm_bootModal(heading, content, width) {
    return WPDM.bootAlert(heading, content, width);
}

function wpdm_boot_popup(heading, content, buttons) {
    var html, $ = jQuery;
    $("#w3eden__boot_popup").remove();
    var _buttons = '';
    if (buttons) {
        _buttons = '<div class="modal-footer" style="padding: 8px 15px;">\n';
        $.each(buttons, function (i, button) {
            var id = 'btx_' + i;
            _buttons += "<button id='" + id + "' class='" + button.class + " btn-xs' style='font-size: 10px;padding: 3px 20px;'>" + button.label + "</button> ";
        });
        _buttons += '</div>\n';
    }

    html = '<div class="w3eden" id="w3eden__boot_popup"><div id="__boot_popup" style="z-index: 9999999 !important;" class="modal fade" tabindex="-1" role="dialog">\n' +
        '  <div class="modal-dialog" role="document" style="max-width: 96%;width: 350px">\n' +
        '    <div class="modal-content" style="border-radius: 3px;overflow: hidden">\n' +
        '      <div class="modal-header" style="padding: 12px 15px;background: #f5f5f5;">\n' +
        '        <h4 class="modal-title" style="font-size: 9pt;font-weight: 500;padding: 0;margin: 0;letter-spacing: 0.5px">' + heading + '</h4>\n' +
        '      </div>\n' +
        '      <div class="modal-body text-center" style="letter-spacing: 0.5px;font-size: 10pt;font-weight: 300;padding: 25px;line-height: 1.5">\n' +
        '        ' + content + '\n' +
        '      </div>\n' + _buttons +
        '    </div>\n' +
        '  </div>\n' +
        '</div></div>';
    $('body').append(html);
    $("#__boot_popup").modal('show');
    $.each(buttons, function (i, button) {
        var id = 'btx_' + i;
        $('#' + id).unbind('click');
        $('#' + id).bind('click', function () {
            button.callback.call($("#__boot_popup"));
            return false;
        });
    });
    return $("#__boot_popup");
}

/**
 * Open an url in iframe modal
 * @param url
 * @param closebutton
 * @returns {boolean}
 */
function wpdm_iframe_modal(url, closebutton) {
    var iframe, $ = jQuery;
    if (url === 'close') {
        $('#wpdm_iframe_modal').remove();
        $('#ifcb').remove();
        $('body').removeClass('wpdm-iframe-modal-open');
        return false;
    }
    var closebutton_html = "";
    if (closebutton !== undefined && closebutton === true)
        closebutton_html = "<span id='ifcb' class='w3eden'><a href='#' onclick='return wpdm_iframe_modal(\"close\");' style='border-radius: 0;position: fixed;top: 0;right: 0;z-index: 9999999999 !important;width: 40px;line-height: 40px;padding: 0' class='btn btn-danger'><i class='fas fa-times'></i></a></span>";

    iframe = '<iframe src="' + url + '" style="width: 100%;height: 100%;position: fixed;z-index: 999999999 !important;border: 0;left: 0;top: 0;right: 0;bottom: 0;background: rgba(0,0,0,0.2);display: none;" id="wpdm_iframe_modal"></iframe>' + closebutton_html;
    $('body').append(iframe).addClass('wpdm-iframe-modal-open');
    $('#wpdm_iframe_modal').fadeIn();

}
