<?php
/**
 * DocoDoco Contact Form 7 Extension
 * 
 * @package     docodoco-cf7-extension
 * @author      Geolocation Technology, Inc.
 * @copyright   2024 Geolocation Technology, Inc.
 * @license     GPL-2.0-or-later
 * @version     1.0.1
 * 
 * @wordpress-plugin
 * Plugin Name:         DocoDoco Contact Form 7 Extension
 * Plugin URI:          https://www.docodoco.jp/plugin/
 * Description:         Adds an auto input function for organization information, such as company name and address, based on IP address to "Contact Form 7" forms.
 * Version:             1.0.1
 * Requires at least:   6.0
 * Requires PHP:        7.4
 * Author:              Geolocation Technology, Inc.
 * Author URI:          https://www.geolocation.co.jp/
 * Text Domain:         docodoco-cf7-extension
 * License:             GPL v2 or later
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Requires Plugins:    contact-form-7
 */

if (!defined('ABSPATH')) {
    exit;
}

define('DOCODOCO_CF7_EXTENSION_VERSION', '1.0.1');
define('DOCODOCO_CF7_EXTENSION_PLUGIN_DIR', plugin_dir_path(__FILE__)); // プラグインディレクトリの絶対パス。末尾のスラッシュ付き。

require_once DOCODOCO_CF7_EXTENSION_PLUGIN_DIR . 'includes/ConstantsGenerated.php';
use DocodocoCf7Extension\ConstantsGenerated;

if (is_admin()) {
    /*
     * 管理画面メニュー表示部分
     */
    function docodoco_cf7_extension_admin_page_common_settings() {
        wp_enqueue_script('docodoco-cf7-extension-common-settings', plugin_dir_url(__FILE__) . 'admin/js/common-settings.js', array(), DOCODOCO_CF7_EXTENSION_VERSION, true);

        require_once DOCODOCO_CF7_EXTENSION_PLUGIN_DIR . 'admin/common-settings.php';
    }
    /*
     * ヘルプページ表示部分
     */
    function docodoco_cf7_extension_admin_page_how_to_use() {
        wp_enqueue_script('docodoco-cf7-extension-how-to-use', plugin_dir_url(__FILE__) . 'admin/js/how-to-use.js', array(), DOCODOCO_CF7_EXTENSION_VERSION, true);
        wp_enqueue_style('docodoco-cf7-extension-how-to-use', plugins_url('admin/css/how-to-use.css', __FILE__), [], DOCODOCO_CF7_EXTENSION_VERSION);

        require_once DOCODOCO_CF7_EXTENSION_PLUGIN_DIR . 'admin/how-to-use.php';
    }

    function docodoco_cf7_extension_register_admin_menu() {
        add_menu_page(
            'どこどこCF7拡張',
            'どこどこCF7拡張',
            'edit_pages',
            'docodoco-cf7-extension',
            '',
            'data:image/svg+xml;base64,' . base64_encode( file_get_contents( plugin_dir_path( __FILE__ ) . 'admin/images/menu-icon.svg' ) )
        );
        add_submenu_page(
            'docodoco-cf7-extension',
            '基本設定',
            '基本設定',
            'edit_pages',
            'docodoco-cf7-extension-common-settings',
            'docodoco_cf7_extension_admin_page_common_settings',
        );
        add_submenu_page(
            'docodoco-cf7-extension',
            '使い方',
            '使い方',
            'edit_pages',
            'docodoco-cf7-extension-how-to-use',
            'docodoco_cf7_extension_admin_page_how_to_use',
        );

        // メインメニューのページを非表示にする
        remove_submenu_page('docodoco-cf7-extension', 'docodoco-cf7-extension');
    }
    add_action('admin_menu', 'docodoco_cf7_extension_register_admin_menu');
    
} else {
    /*
     * 公開ページアクセス時の処理
     */

    // 設定がない場合は何もしない
    $common_settings = get_option('docodoco_cf7_extension_common_settings');
    if ($common_settings === false || empty($common_settings['docodocojp_apikey'])) {
        return;
    }

    /*
     * プラグインで必要なスクリプトを配置する
     */
    function docodoco_cf7_extension_enqueue_scripts() {
        // 基本設定画面で設定されたAPIキーを取得
        $common_settings = get_option('docodoco_cf7_extension_common_settings');
        if ($common_settings === false || empty($common_settings['docodocojp_apikey'])) {
            return;
        }
        $docodocojp_apikey = $common_settings['docodocojp_apikey'];
        // 「どこどこJPタグをページに追加する」オプションの設定値からスクリプトタグ追加の有無を決定する
        $scriptTagRequired = true;
        if (isset($common_settings['add_script_tag'])) {
            $scriptTagRequired = docodoco_cf7_extension_isScriptTagRequired($common_settings['add_script_tag']);
        }

        // Contact Form 7 のフォーム設定にカスタムタグオプションが含まれている場合のみ追加する
        if (docodoco_cf7_extension_contains_custom_tag_option()) {
            if ($scriptTagRequired) {
                // どこどこJPのタグを追加する
                $script_url = "//api.docodoco.jp/v6/docodoco?key={$docodocojp_apikey}&bake=1&used_by=docodoco-cf7-extension";
                wp_enqueue_script('docodoco-cf7-extension-docodocojp-api-script', $script_url, array(), null, false);
            }
            // 入力補完を行うスクリプトを追加する
            wp_enqueue_script('docodoco-cf7-extension', plugin_dir_url(__FILE__) . '/public/js/docodoco-cf7-extension.js', array(), DOCODOCO_CF7_EXTENSION_VERSION, true);
            docodoco_cf7_extension_add_inline_script();
        }
    }
    add_action('wpcf7_enqueue_scripts', 'docodoco_cf7_extension_enqueue_scripts', 5);

    /*
     * どこどこJPタグにreferrerpolicy属性を追加する
     */
    function docodoco_cf7_extension_add_referrer_policy_to_docodocojp_script($attributes) {
         // 対象のスクリプトハンドル名をチェック
        if ($attributes['id'] === 'docodoco-cf7-extension-docodocojp-api-script-js') {
            $attributes['referrerpolicy'] = 'no-referrer-when-downgrade';
        }
        return $attributes;
    }
    add_filter('wp_script_attributes', 'docodoco_cf7_extension_add_referrer_policy_to_docodocojp_script', 10, 3);
}

/**
 * どこどこJPスクリプトタグを追加するかどうかを返す関数
 * 
 * @param string 「どこどこJPタグをページに追加する」オプションの設定値
 * @return bool trueの場合、タグの設置が必要
 */
function docodoco_cf7_extension_isScriptTagRequired($add_script_tag) {
    if ($add_script_tag === '1') {
        return true;
    } else { 
        return false;
    }
}


function docodoco_cf7_extension_add_inline_script() {
    // ContactForm7のフォームが存在しなければ何もしない
    if (!function_exists('wpcf7_get_current_contact_form')) {
        return;
    }
    $contact_form = wpcf7_get_current_contact_form();
    if (!$contact_form) {
        return;
    }

    // 入力補完のオプション文字列が設定されていればスクリプトを追加する
    // 判定条件は、'docoform:' で始まるタグオプション
    $tags = $contact_form->scan_form_tags();
    foreach ($tags as $tag) {
        if (!$tag->has_option('docoform')){
            continue;
        }
        
        // 'docoform' オプションから値を取得
        $option_type = $tag->get_option('docoform', '', true );
        $name = $tag->name;
        switch ($option_type) {
            case "use_domain_search":
                docodoco_cf7_extension_gen_add_form_tag_docoform_domainsearch($tag);
                break;
            case "bcflag":
                docodoco_cf7_extension_gen_add_form_tag_docoform_bcflag($tag);
                break;
            case "org_name":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'OrgName');
                break;
            case "org_zip_code":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'OrgZipCode');
                break;
            case "org_address":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'OrgAddress');
                break;
            case "org_tel":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'OrgTel');
                break;
            case "org_fax":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'OrgFax');
                break;
            case "org_url":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'OrgUrl');
                break;
            case "client_city":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'CityJName');
                break;
            case "client_pref":
                docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, 'PrefJName');
                break;
            default:
                // 間違ったオプションを渡している場合は default に入る可能性がある
                // この場合は何もしない
                break;
        }
    }
}

// フォームタグ用: BCFlagとドメイン検索以外のフォームタグ用の関数を生成する
function docodoco_cf7_extension_gen_add_form_tag_docoform_common($name, $surfpoint_getter) {
    $inline_script = "document.addEventListener('DOMContentLoaded', function () {
        // ドメイン検索が有効な場合の処理
        if (typeof DocoDocoCF7ExtensionDomainSearch !== 'undefined') {
            // ドメイン検索が実行された時のイベントのリスナー
            document.addEventListener('docodoco_cf7_extension_domain_search', function (event) {
                document.querySelectorAll('[name={$name}]').forEach(function (e) {
                    if (e.value.length !== 0) {
                        return; // フォームが入力済みだったら何もしない
                    }
                    e.value = event.detail.result.{$surfpoint_getter};
                });
            });
            return;
        }

        // ドメイン検索が有効でない場合の処理
        const limit = 10;
        let count = 0;
        const docodocoTagWaiter = setInterval(function () {
            if (count++ >= limit) {
                // limit 回試行してもSURFPOINTが読み込めなかった場合は処理を終了する
                clearInterval(docodocoTagWaiter);
            }

            if (typeof SURFPOINT !== 'undefined' && SURFPOINT['get{$surfpoint_getter}'] !== undefined) {
                clearInterval(docodocoTagWaiter);
                document.querySelectorAll('[name={$name}]').forEach(function (e) {
                    e.value = SURFPOINT['get{$surfpoint_getter}']();
                });
            }
        }, 500);

    });";

    // インラインスクリプトとして追加
    wp_add_inline_script('docodoco-cf7-extension', $inline_script, 'after');
}

function docodoco_cf7_extension_gen_add_form_tag_docoform_bcflag($tag) {
    if (!isset($tag->values) || !isset($tag->values[0]) || !isset($tag->values[1])){
        return '';
    }
    $name = $tag->name;
    $b_val = $tag->values[0];
    $c_val = $tag->values[1];

    $inline_script = "document.addEventListener('DOMContentLoaded', function () {
        // ドメイン検索が有効な場合の処理
        if (typeof DocoDocoCF7ExtensionDomainSearch !== 'undefined') {
            // ドメイン検索が実行された時のイベントのリスナー
            document.addEventListener('docodoco_cf7_extension_domain_search', function (event) {
                document.querySelectorAll('[name={$name}]').forEach(function (e) {
                    if (e.value === '{$b_val}' && event.detail.result.BCFlag === 'b') {
                        e.checked = true;
                    } else if (e.value === '{$c_val}' && event.detail.result.BCFlag === '') {
                        e.checked = true;
                    }
                });
            });
            return;
        }

        // ドメイン検索が有効でない場合の処理
        function fillInputs() {

        }

        // ドメイン検索が有効でない場合の処理
        const limit = 10;
        let count = 0;
        const docodocoTagWaiter = setInterval(function () {
            if (count++ > limit) {
                // limit 回試行してもSURFPOINTが読み込めなかった場合は処理を終了する
                clearInterval(docodocoTagWaiter);
                return;
            }

            if (typeof SURFPOINT !== 'undefined' && SURFPOINT['getBCFlag'] !== undefined) {
                clearInterval(docodocoTagWaiter);
                document.querySelectorAll('[name={$name}]').forEach(function (e) {
                    let bcflag = SURFPOINT['getBCFlag']();
                    if (e.value === '{$b_val}' && bcflag === 'b') {
                        e.checked = true;
                    } else if (e.value === '{$c_val}' && bcflag === '') {
                        e.checked = true;
                    }
                });
                return;
            }
        }, 500);
    });";
    
    // インラインスクリプトとして追加
    wp_add_inline_script('docodoco-cf7-extension', $inline_script, 'after');
}

function docodoco_cf7_extension_gen_add_form_tag_docoform_domainsearch($tag) {
    $common_settings = get_option('docodoco_cf7_extension_common_settings');
    if ($common_settings === false || empty($common_settings['docodocojp_apikey'])) {
        return;
    }
    $docodocojp_apikey = $common_settings['docodocojp_apikey'];

    if (!isset($tag->name)) {
        return '';
    }
    $name = $tag->name;

    // メールアドレスの正規表現は以下をベースにする
    // ref: https://html.spec.whatwg.org/#e-mail-state-(type=email)
    $inline_script = "const DocoDocoCF7ExtensionDomainSearch = true;
    const DocoDocoCF7ExtensionApiEndpoint = 'https://api.docodoco.jp/v6/docodoco_domain?key={$docodocojp_apikey}&format=json&domain=';
    // &domain=example.com でドメイン検索を行う

    function getSURFPOINTData(result) {
        if (typeof SURFPOINT !== 'undefined' && typeof SURFPOINT.getBCFlag !== 'undefined') {
            // SURFPOINT がある場合は SURFPOINT の値を使う, ない場合（JS版エラー、無効なAPIキー）の場合は初期値を返す
            result.BCFlag = SURFPOINT.getBCFlag();
            result.OrgName = SURFPOINT.getOrgName();
            result.OrgZipCode = SURFPOINT.getOrgZipCode();
            result.OrgAddress = SURFPOINT.getOrgAddress();
            result.OrgTel = SURFPOINT.getOrgTel();
            result.OrgFax = SURFPOINT.getOrgFax();
            result.OrgUrl = SURFPOINT.getOrgUrl();
        }
    }

    document.querySelectorAll('[name={$name}]').forEach(function (mailForm) {
        mailForm.addEventListener('change', function (e) {
            // ajax でドメイン検索を行う。完了したら docodoco_cf7_extension_domain_search イベントを発火する
            let emailMatchResult = mailForm.value.trim().match(\"^[a-zA-Z0-9.+_-]+@([a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)+)$\");

            if (emailMatchResult === null || emailMatchResult.length < 2) {
                // メールドメインがおかしい場合は何もしない
                return;
            }

            let domain = emailMatchResult[1];

            const req = new XMLHttpRequest();
            req.onreadystatechange = function () {
                if (req.readyState !== 4) {
                    return;
                }

                // ドメイン検索結果を格納する変数の初期値
                let result = {
                    BCFlag: '',
                    OrgName: '',
                    OrgZipCode: '',
                    OrgAddress: '',
                    OrgTel: '',
                    OrgFax: '',
                    OrgUrl: '',
                };

                if (req.status !== 200) {
                    // フリーメール・ISPメール・ドメイン情報が取れない場合（オプション未加入かAPIキーが無効などのエラーの場合）
                    // IPアドレスベースの組織情報を使う
                    getSURFPOINTData(result);
                    const event = new CustomEvent('docodoco_cf7_extension_domain_search', { detail: { result: result } });
                    document.dispatchEvent(event);
                    return;
                }

                let data = JSON.parse(req.responseText);

                if (data.status !== undefined) {
                    // フリーメール・ISPメール・ドメイン情報が取れない場合（オプション未加入かAPIキーが無効などのエラーの場合）
                    // IPアドレスベースの組織情報を使う
                    getSURFPOINTData(result);
                    const event = new CustomEvent('docodoco_cf7_extension_domain_search', { detail: { result: result } });
                    document.dispatchEvent(event);
                    return;
                }

                if (
                    data.FreeMail_6 === undefined
                    || data.ISPMail_7 === undefined
                    || data.DomainOrg_8 === undefined
                    || data.FreeMail_6.FreeMailFlag !== \"0\"
                    || data.ISPMail_7.ISPMailFlag !== \"0\"
                ) {
                    // フリーメール・ISPメール・ドメイン情報が取れない場合（オプション未加入かAPIキーが無効などのエラーの場合）
                    // IPアドレスベースの組織情報を使う
                    getSURFPOINTData(result);
                    const event = new CustomEvent('docodoco_cf7_extension_domain_search', { detail: { result: result } });
                    document.dispatchEvent(event);
                    return;
                } else {
                    result = {
                        BCFlag: data.DomainOrg_8.OrgName === undefined || Object.keys(data.DomainOrg_8.OrgName).length === 0 ? '' : 'b',
                        OrgName: data.DomainOrg_8.OrgName === undefined || Object.keys(data.DomainOrg_8.OrgName).length === 0 ? '' : data.DomainOrg_8.OrgName,
                        OrgZipCode: data.DomainOrg_8.OrgZipCode === undefined || Object.keys(data.DomainOrg_8.OrgZipCode).length === 0 ? '' : data.DomainOrg_8.OrgZipCode,
                        OrgAddress: data.DomainOrg_8.OrgAddress === undefined || Object.keys(data.DomainOrg_8.OrgAddress).length === 0 ? '' : data.DomainOrg_8.OrgAddress,
                        OrgTel: data.DomainOrg_8.OrgTel === undefined || Object.keys(data.DomainOrg_8.OrgTel).length === 0 ? '' : data.DomainOrg_8.OrgTel,
                        OrgFax: data.DomainOrg_8.OrgFax === undefined || Object.keys(data.DomainOrg_8.OrgFax).length === 0 ? '' : data.DomainOrg_8.OrgFax,
                        OrgUrl: data.DomainOrg_8.OrgUrl === undefined || Object.keys(data.DomainOrg_8.OrgUrl).length === 0 ? '' : data.DomainOrg_8.OrgUrl,
                    };
                    if (result.BCFlag !== 'b') {
                        // ドメイン検索で企業判定にならなかった場合はIPアドレスベースの組織情報を使う
                        getSURFPOINTData(result);
                    }
                }
                const event = new CustomEvent('docodoco_cf7_extension_domain_search', { detail: { result: result } });
                document.dispatchEvent(event);
            };
            req.open('GET', DocoDocoCF7ExtensionApiEndpoint + domain);
            req.send(null);
        });
    });";

    // インラインスクリプトとして追加
    wp_add_inline_script('docodoco-cf7-extension', $inline_script, 'after');
}

add_filter('wpcf7_special_mail_tags', 'docodoco_cf7_extension_add_mail_tag_docomail_tags', 10, 2);

// SURFPOINTの値を取得する関数。
function docodoco_cf7_extension_get_surfpoint(string $surfpoint_getter): string {
    // フォーム送信時にSURFPOINTの値がJSON形式でPOSTされるのでそれを取得する
    if (isset($_POST['_docoform_surfpoint']) && isset($_POST['_docoform_surfpoint'][$surfpoint_getter])) {
        // SURFPOINTのデータはurlencodeされているが、sanitize_text_field() でデコードされるので urldecode() は不要
        return sanitize_text_field(wp_unslash($_POST['_docoform_surfpoint'][$surfpoint_getter]));
    }
    return '';
}

// メールタグをSURFPOINTの値に置き換える関数
function docodoco_cf7_extension_add_mail_tag_docomail_tags($output, $name) {
    switch ($name) {
        case "docomail_org_name":
            return docodoco_cf7_extension_docomail_get_value_direct('OrgName');
        case "docomail_org_zip_code":
            return docodoco_cf7_extension_docomail_get_value_direct('OrgZipCode');
        case "docomail_org_address":
            return docodoco_cf7_extension_docomail_get_value_direct('OrgAddress');
        case "docomail_org_tel":
            return docodoco_cf7_extension_docomail_get_value_direct('OrgTel');
        case "docomail_org_fax":
            return docodoco_cf7_extension_docomail_get_value_direct('OrgFax');
        case "docomail_org_ipo_type":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgIpoType');
        case "docomail_stock_ticker_number":
            return docodoco_cf7_extension_docomail_get_value_direct('StockTickerNumber');
        case "docomail_org_capital":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgCapitalCode');
        case "docomail_org_employees":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgEmployeesCode');
        case "docomail_org_gross":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgGrossCode');
        case "docomail_org_industrial_category_l":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgIndustrialCategoryL');
        case "docomail_org_industrial_category_m":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgIndustrialCategoryM');
        case "docomail_org_industrial_category_s":
            return docodoco_cf7_extension_docomail_get_value_ref_master('OrgIndustrialCategoryS');
        case "docomail_org_url":
            return docodoco_cf7_extension_docomail_get_value_direct('OrgUrl');
        case "docomail_houjin_bangou":
            return docodoco_cf7_extension_docomail_get_value_direct('HoujinBangou');

        // city, pref はドキュメント化しない（実質デバッグ用）
        case "docomail_client_pref":
            return docodoco_cf7_extension_docomail_get_value_direct('PrefJName');
        case "docomail_client_city":
            return docodoco_cf7_extension_docomail_get_value_direct('CityJName');
        default:
            return $output;
    }
}

// メールタグ用: SURFPOINTデータから属性データを取得する関数。
// これはSURFPOINTに入っている値を直接使えばいい時に使用する。
function docodoco_cf7_extension_docomail_get_value_direct($surfpoint_key) {
    return docodoco_cf7_extension_get_surfpoint($surfpoint_key);
}

// メールタグ用: SURFPOINTデータから属性データを取得する関数。
// これはSURFPOINTに入っている値をマスターから取得する時に使用する。
function docodoco_cf7_extension_docomail_get_value_ref_master($surfpoint_key) {
    $surfpoint_data = docodoco_cf7_extension_get_surfpoint($surfpoint_key);
    if ($surfpoint_data === '') {
        return '';
    }

    $master_data = [];
    switch($surfpoint_key) {
        case "OrgIpoType":
            $master_data = ConstantsGenerated::orgIpoType;
            break;
        case "OrgCapitalCode":
            $master_data = ConstantsGenerated::orgCapitalCode;
            break;
        case "OrgEmployeesCode":
            $master_data = ConstantsGenerated::orgEmployeesCode;
            break;
        case "OrgGrossCode":
            $master_data = ConstantsGenerated::orgGrossCode;
            break;
        case "OrgIndustrialCategoryL":
            $master_data = ConstantsGenerated::orgCategoryL;
            break;
        case "OrgIndustrialCategoryM":
            $master_data = ConstantsGenerated::orgCategoryM;
            break;
        case "OrgIndustrialCategoryS":
            $master_data = ConstantsGenerated::orgCategoryS;
            break;
        default:
            return '';
    }

    // 業種などはカンマ区切りで複数の値が入っているので分解して配列にする。
    $keys = explode(',', $surfpoint_data);

    // 判定結果に一致するマスタデータのみを取り出す
    $matchedData = array_intersect_key($master_data, array_flip($keys));

    return implode(", ", array_column($matchedData, 'contents'));
}

/**
 * Contact Form 7 のフォーム設定とメール設定にどこどこJPのカスタムタグが存在するかどうかをチェックする
 */
function docodoco_cf7_extension_contains_custom_tag_option() {
    // ContactForm7のフォームが存在しなければ何もしない
    if (!function_exists('wpcf7_get_current_contact_form')) {
        return false;
    }
    
    // ページにフォームが存在しない場合は何もしない
    $contact_form = wpcf7_get_current_contact_form();
    if (!$contact_form) {
        return false;
    }

    // フォームテンプレートに、本プラグインのカスタムタグが含まれているかチェックする
    // 判定条件は、'docoform_' で始まるフォームタグ
    $tags = $contact_form->scan_form_tags();
    foreach ($tags as $tag) {
        if ($tag->has_option('docoform')){
            return true;
        }
    }

    // メールテンプレートに、本プラグインのカスタムタグが含まれているかチェックする
    $mail_template = $contact_form->prop('mail');
    if (empty($mail_template)) {
        return false; 
    }

    // メール本文に含まれるカスタムフォームタグを取得する
    // 判定条件は、'[docoform_' で始まり、']' 以外の文字が1回以上続き、最後に ']' で終わる文字列
    $body = $mail_template['body'];
    preg_match_all('/\[docomail_([^\]]+)\]/', $body, $matches);
    if (!empty($matches[0])) {
        return true;
    }

    return false;
}
