<?php
/**
 * Plugin Name: Do Not Copy
 * Plugin URI: https://gptblog.jp/do-not-copy
 * Description: 「Do Not Copy」は、WordPressサイト内でのテキストのコピーを防止するシンプルなプラグインです。
 * Version: 1.1.1
 * Author: Toshikazu Hirano
 * License: GPL2
 */
// 設定メニューを追加
function donoco_add_admin_menu() {
    add_options_page('Do Not Copy', 'Do Not Copy', 'manage_options', 'do-not-copy', 'donoco_settings_page');
}
add_action('admin_menu', 'donoco_add_admin_menu');

// 設定ページの内容
function donoco_settings_page() {
    ?>
    <div class="wrap">
        <h2>Do Not Copy 設定</h2>
        <form method="post" action="options.php">
            <?php
            settings_fields('donoco_settings_group');
            do_settings_sections('do-not-copy');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// 設定登録
function donoco_register_settings() {
    register_setting('donoco_settings_group', 'donoco_global_copy_protection', 'sanitize_key');

    add_settings_section('donoco_main_section', '基本設定', null, 'do-not-copy');

    add_settings_field('donoco_global_copy_protection', 'サイト全体のコピー防止', 'donoco_global_copy_field', 'do-not-copy', 'donoco_main_section');
}
add_action('admin_init', 'donoco_register_settings');

// 全ページのコピー防止設定
function donoco_global_copy_field() {
    $value = esc_attr(get_option('donoco_global_copy_protection', '0'));
    echo '<input type="checkbox" name="donoco_global_copy_protection" value="1" ' . checked(1, $value, false) . '> 有効';
}

// 記事ごとのコピー防止設定（メタボックス追加）
function donoco_add_meta_box() {
    add_meta_box('donoco_meta_box', 'コピー防止設定', 'donoco_meta_box_callback', ['post', 'page'], 'side');
}
add_action('add_meta_boxes', 'donoco_add_meta_box');

function donoco_meta_box_callback($post) {
    $value = get_post_meta($post->ID, '_donoco_copy_protection', true);
    wp_nonce_field('donoco_meta_box', 'donoco_meta_box_nonce');
    echo '<label><input type="checkbox" name="donoco_copy_protection" value="1" ' . checked(1, $value, false) . '> このページでコピーを防止する</label>';
}

// 記事のメタデータ保存
function donoco_save_post_meta($post_id) {
    if (!isset($_POST['donoco_meta_box_nonce'])) return;

    // nonce を適切に処理
    $nonce = sanitize_text_field(wp_unslash($_POST['donoco_meta_box_nonce']));
    if (!wp_verify_nonce($nonce, 'donoco_meta_box')) return;

    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $value = isset($_POST['donoco_copy_protection']) ? 1 : 0;
    update_post_meta($post_id, '_donoco_copy_protection', $value);
}
add_action('save_post', 'donoco_save_post_meta');

// CSSとJSを読み込む
function donoco_enqueue_scripts() {
    $global_enabled = esc_attr(get_option('donoco_global_copy_protection', '0'));

    if (is_singular()) {
        global $post;
        $enabled = get_post_meta($post->ID, '_donoco_copy_protection', true);
    } else {
        $enabled = false;
    }

    // グローバル設定もしくは記事個別設定が有効ならコピー防止スクリプトを読み込む
    if ($enabled || $global_enabled == '1') {
        wp_enqueue_style('donoco-style', plugin_dir_url(__FILE__) . 'css/donoco-style.css', [], '1.1.1');
        wp_enqueue_script('donoco-script', plugin_dir_url(__FILE__) . 'js/donoco-script.js', [], '1.1.1', true);
    }
}
add_action('wp_enqueue_scripts', 'donoco_enqueue_scripts');