<?php

namespace App\Repository;

use App\Loader;
use WP_Term_Query;
use WC_Product_Query;

class ProductRepository extends WoocommerceRepository
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get all products with pagination
     *
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function getAllProducts($page = 1, $per_page = 50)
    {

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products",
            array('per_page' => $per_page, 'page' => $page)
        );

        return $this->ensureResponse($data);
    }

    /**
     * Get multiple products from productIds
     *
     * @param  array<int, int> $productIds
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function getProducts($productIds, $page = 1, $per_page = 100)
    {

        if (empty($productIds)) {
            return [];
        }

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products",
            array('include' => $productIds, 'per_page' => $per_page, 'page' => $page)
        );

        return $this->ensureResponse($data);
    }

    /**
     * Search products by name
     *
     * @param  string $productName
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function searchProducts($productName, $page = 1, $per_page = 50)
    {

        if (empty($productName)) {
            return [];
        }

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products",
            array('search' => $productName, 'per_page' => $per_page, 'page' => $page)
        );

        return $this->ensureResponse($data);
    }

    /**
     * Get products from a collection
     *
     * @param  int $collectionId
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function getProductsFromCollection($collectionId, $page = 1, $per_page = 50)
    {

        // Perfect brands plugin is enable
        if (Loader::init()->brands->enable) {

            // If collection id is a brand id lets try to retrieve all products ids from that collection
            // And then we can reuse the products route in order to preserve data response
            $term = get_term($collectionId);
            if ($term->taxonomy == Loader::BRANDS_TAXONOMY) {
                $productIds = $this->getProductsWithTaxonomy(Loader::BRANDS_TAXONOMY, $collectionId);
                return $this->getProducts($productIds, $page, $per_page);
            }
        }

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products",
            array('category' => strval($collectionId), 'per_page' => $per_page, 'page' => $page)
        );

        return $this->ensureResponse($data);
    }

    /**
     * Get products ids from collections
     *
     * @param  array<int, int> $collectionIds
     * @return array<int, int>
     */
    public function getProductIdsFromCollections($collectionIds)
    {
        return array_merge(
            $this->getProductsWithTaxonomy('product_cat', $collectionIds),
            $this->getProductsWithTaxonomy(Loader::BRANDS_TAXONOMY, $collectionIds)
        );
    }

    /**
     * Get product
     *
     * @param  int $productId
     * @return array
     */
    public function getProduct($productId)
    {
        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products/{$productId}",
        );

        return $this->ensureResponse($data);
    }

    /**
     * Get products ids for taxonomy
     *
     * @return array
     */
    private function getProductsWithTaxonomy($taxonomyName, $value)
    {
        $args = array(
            'limit'                 => '-1',
            'tax_query'             => array(
                array(
                    'taxonomy'      => $taxonomyName,
                    'field'         => 'term_id',
                    'terms'         =>  $value,
                    'operator'      => 'IN'
                ),
            ),
            'return' => 'ids',
        );

        $query = new WC_Product_Query($args);
        return $query->get_products();
    }
}
