<?php

namespace App\Repository;

use App\Models\DiscountRule;

class DiscountRulesRepository
{
    /**
     * @var string
     */
    private $discount_rules_table;

    /**
     * @var string
     */
    private $discount_products_table;

    /**
     * @var string
     */
    private $discount_collections_table;

    public function __construct()
    {
        global $wpdb;
        $this->discount_rules_table = $wpdb->prefix . "napps_discount_rules";
        $this->discount_products_table = $wpdb->prefix . "napps_discount_rules_products";
        $this->discount_collections_table = $wpdb->prefix . "napps_discount_rules_collections";
    }
    
    /**
     * Run migration for version
     *
     * @param  string $version
     * @return void
     */
    public function runMigration($version) {

        global $wpdb;
        
        switch($version) {
            case "1.0.6":
                $wpdb->query("
                    ALTER TABLE `$this->discount_rules_table` 
                    ADD COLUMN start_date VARCHAR(255),
                    ADD COLUMN end_date VARCHAR(255)"
                );

                break;
        }
    }

    /**
     * Create tables for discount_rules
     *
     * @return void
     */
    public function createTables()
    {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $discount_struct = "
            CREATE TABLE IF NOT EXISTS `$this->discount_rules_table` (
                `id` INT NOT NULL AUTO_INCREMENT,
                `name` TEXT NOT NULL,
                `discount_type` INT(10) UNSIGNED NOT NULL,
                `status` INT(10) UNSIGNED NOT NULL,
                `amount` INT(10) UNSIGNED NOT NULL,
                `start_date` VARCHAR(255),
                `end_date` VARCHAR(255),
                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY ( id )
            ) $charset_collate";

        $discount_products_struct = "
            CREATE TABLE IF NOT EXISTS `$this->discount_products_table` (
                `discount_rule_id` INT NOT NULL ,
                `product_id` INT NOT NULL ,
                PRIMARY KEY(discount_rule_id, product_id),
                FOREIGN KEY (discount_rule_id) REFERENCES $this->discount_rules_table(ID) ON DELETE CASCADE
            ) $charset_collate";

        $discount_collections_struct = "
            CREATE TABLE IF NOT EXISTS `$this->discount_collections_table` (
                `discount_rule_id` INT NOT NULL ,
                `collection_id` INT NOT NULL ,
                PRIMARY KEY(discount_rule_id, collection_id),
                FOREIGN KEY (discount_rule_id) REFERENCES $this->discount_rules_table(ID) ON DELETE CASCADE
            ) $charset_collate";

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

        dbDelta($discount_struct);
        dbDelta($discount_products_struct);
        dbDelta($discount_collections_struct);
    }

    /**
     * Get all discount rules
     *
     * @return ?array|object
     */
    public function getAll()
    {
        global $wpdb;
        return $wpdb->get_results(
            "SELECT `id`, `name`, `discount_type`, `status`, `amount`, `created_at`, `start_date`, `end_date`, `updated_at` FROM $this->discount_rules_table ORDER BY ID ASC",
        );
    }

    /**
     * Get all discount rules
     *
     * @param  int $id
     * @return ?object
     */
    public function getById($id)
    {
        global $wpdb;
        $discountRule = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT `id`, `name`, `discount_type`, `status`, `amount`, `created_at`, `start_date`, `end_date`, `updated_at` FROM $this->discount_rules_table WHERE id = %d",
                $id
            )
        );

        if(!$discountRule) {
            return null;
        }

        $discountRule->products = $this->getProducts($discountRule->id);
        $discountRule->collections = $this->getCollections($discountRule->id);
        return $discountRule;
    }

    /**
     * Get product ids for discount rule
     *
     * @param  integer $discountRuleId
     * @return array<int, int>
     */
    public function getProducts($discountRuleId)
    {
        global $wpdb;
        $result = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT `product_id` FROM $this->discount_products_table WHERE discount_rule_id = %d",
                $discountRuleId
            )
        );

        if (!$result || !is_array($result)) {
            return [];
        }

        return $result;
    }

    /**
     * Get collection ids for discount rule
     *
     * @param  int $discountRuleId
     * @return array<int, int>
     */
    public function getCollections($discountRuleId)
    {
        global $wpdb;
        $result = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT `collection_id` FROM $this->discount_collections_table WHERE discount_rule_id = %d",
                $discountRuleId
            )
        );

        if (!$result || !is_array($result)) {
            return [];
        }

        return $result;
    }

    /**
     * Delete a discount rule
     *
     * @param  int $discountRuleId
     * @return void
     */
    public function delete($discountRuleId) {

        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $this->discount_rules_table
                WHERE id = %d",
                $discountRuleId
            )
        );

    }

    /**
     * Save discount rule and return with set id
     *
     * @param  DiscountRule $discountRule
     * @return DiscountRule
     */
    public function save($discountRule)
    {
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO $this->discount_rules_table
                ( name, discount_type, amount, status, start_date, end_date )
                VALUES ( %s, %d, %d, %d, %s, %s )",
                $discountRule->name,
                $discountRule->discount_type,
                $discountRule->amount,
                $discountRule->status,
                $discountRule->start_date,
                $discountRule->end_date,
            )
        );

        $insertedId = $wpdb->insert_id;
        if ($insertedId !== false) {
            $discountRule->id = $insertedId;
            $discountRule->created_at = gmdate("Y-m-d H:i:s");
            $discountRule->updated_at = $discountRule->created_at;
        }

        return $discountRule;
    }

    /**
     * Update discount rule
     *
     * @param  DiscountRule $discountRule
     * @return bool
     */
    public function update($discountRule)
    {
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE $this->discount_rules_table
                SET name = %s, discount_type = %d, amount = %d, status = %d, start_date = %s, end_date = %s
                WHERE id = %d",
                $discountRule->name,
                $discountRule->discount_type,
                $discountRule->amount,
                $discountRule->status,
                $discountRule->start_date,
                $discountRule->end_date,
                $discountRule->id
            )
        );

        if (is_array($discountRule->productIds)) {
            $this->updateProducts($discountRule->id, $discountRule->productIds);
        }

        if (is_array($discountRule->collectionIds)) {
            $this->updateCollections($discountRule->id, $discountRule->collectionIds);
        }

        return empty($wpdb->last_error);
    }

    /**
     * Update product ids for a discount rule
     *
     * @param  int $discountRuleId
     * @param  array<int, int> $productIds
     * @return bool
     */
    protected function updateProducts($discountRuleId, $productIds)
    {
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $this->discount_products_table
                WHERE discount_rule_id = %d",
                $discountRuleId
            )
        );

        if(empty($productIds)) {
            return true;
        }

        $data = array();
        $dataBindings = array();
        foreach ($productIds as $productId) {
            $data[] = "(%d,%d)";
            $dataBindings[] = $discountRuleId; 
            $dataBindings[] = intval($productId);
        }

        if ($wpdb->query(
                $wpdb->prepare(
                    "INSERT INTO {$this->discount_products_table} (discount_rule_id, product_id) VALUES " . implode( ", ", $data ),
                    $dataBindings
                )
            )
        ) {
            return true;
        }

        return false;

    }

    /**
     * Update product ids for a discount rule
     *
     * @param  int $discountRuleId
     * @param  array<int, int> $collectionIds
     * @return bool
     */
    protected function updateCollections($discountRuleId, $collectionIds)
    {
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $this->discount_collections_table
                WHERE discount_rule_id = %d",
                $discountRuleId
            )
        );

        if(empty($collectionIds)) {
            return true;
        }

        $data = array();
        $dataBindings = array();
        foreach ($collectionIds as $collectionId) {
            $data[] = "(%d,%d)";
            $dataBindings[] = $discountRuleId; 
            $dataBindings[] = intval($collectionId);
        }

        if ($wpdb->query(
                $wpdb->prepare(
                    "INSERT INTO {$this->discount_collections_table} (discount_rule_id, collection_id) VALUES " . implode( ", ", $data ),
                    $dataBindings
                )
            )
        ) {
            return true;
        }

        return false;
    }
}
