<?php

namespace App\Repository;

use App\Loader;

class CategoryRepository extends WoocommerceRepository
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get all categories with pagination
     *
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function getAllCategories($page = 1, $per_page = 50) {

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products/categories",
            array('per_page' => $per_page, 'page' => $page)
        );

        $brands = [];

        if(Loader::init()->brands->enable) {
            $brands = $this->getApi()->get_endpoint_data(
                "/wc/{$this->getApiVersion()}/brands",
                array('per_page' => $per_page, 'page' => $page)
            );

            $brands = Loader::init()->brands->ensureBrandsResponse($brands);
        }

        return $this->ensureResponse(array_merge($data, $brands));
    }

    /**
     * Search products by name
     *
     * @param  string $collectionName
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function searchCollections($collectionName, $page = 1, $per_page = 50) {

        if(empty($collectionName)) {
            return [];
        }

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products/categories",
            array('search' => $collectionName, 'per_page' => $per_page, 'page' => $page)
        );

        $brands = [];
        if(Loader::init()->brands->enable) {
            $brands = $this->getApi()->get_endpoint_data(
                "/wc/{$this->getApiVersion()}/brands",
                array('search' => $collectionName, 'per_page' => $per_page, 'page' => $page)
            );

            $brands = Loader::init()->brands->ensureBrandsResponse($brands);
        }

        return $this->ensureResponse(array_merge($data, $brands));
    }

    /**
     * Get multiple categories from ids
     *
     * @param  array<int, int> $categoriesIds
     * @param  int $page
     * @param  int $per_page
     * @return array<int, array>
     */
    public function getCategories($categoriesIds, $page = 1, $per_page = 100) {

        if(empty($categoriesIds)) {
            return [];
        }

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products/categories",
            array('include' => $categoriesIds, 'per_page' => $per_page, 'page' => $page)
        );

        $brands = [];
        if(Loader::init()->brands->enable) {
            $brands = $this->getApi()->get_endpoint_data(
                "/wc/{$this->getApiVersion()}/brands",
                array('include' => $categoriesIds, 'per_page' => $per_page, 'page' => $page)
            );

            $brands = Loader::init()->brands->ensureBrandsResponse($brands);
        }

        return $this->ensureResponse(array_merge($data, $brands));
    }

    /**
     * Get category
     *
     * @param  int $categoryId
     * @return array
     */
    public function getCategory($categoryId) {

        // Check if brands plugin is active and if category it taxonomy is brands and try to retrieve it
        if(Loader::init()->brands->enable) {

            $term = get_term( $categoryId );
            if($term->taxonomy == Loader::BRANDS_TAXONOMY) {
                $brand = $this->getApi()->get_endpoint_data(
                    "/wc/{$this->getApiVersion()}/brands/{$categoryId}"
                );
                
                $brands = Loader::init()->brands->ensureBrandsResponse($brand);

                return $this->ensureResponse($brands);
            } 

            
        }

        /** @var array|\WP_Error $data */
        $data = $this->getApi()->get_endpoint_data(
            "/wc/{$this->getApiVersion()}/products/categories/{$categoryId}",
        );

        return $this->ensureResponse($data);
    }
}
