<?php

namespace App\Models;

use App\Events\Jobs;
use App\Events\Schedule;
use App\Events\UpdatePrice;
use App\Repository\ProductRepository;
use WP_Term_Query;
use WC_Product_Query;

class DiscountRule {

    /**
     * @var int
     */
    public $id;

    /**
     * @var string
     */
    public $name;

    /**
     * @var int
     */
    public $discount_type;

    /**
     * @var int
     */
    public $status;

    /**
     * @var int
     */
    public $amount;

    /**
     * @var string
     */
    public $created_at;

    /**
     * @var string
     */
    public $updated_at;

    /**
     * @var array<int, int>
     */
    public $productIds;

    /**
     * @var string
     */
    public $start_date;

    /**
     * @var string|null
     */
    public $end_date;

    /**
     * @var array<int, int>
     */
    public $collectionIds;

    public function __construct($id = 0)
    {
        $this->id = intval($id);
        $this->end_date = null;
    }

    /**
     * Set the value of discount_type
     *
     * @param  int  $discount_type
     *
     * @return  self
     */
    public function setDiscountType($discount_type)
    {
        $this->discount_type = intval($discount_type);

        return $this;
    }

    /**
     * Set the value of name
     *
     * @param  string  $name
     *
     * @return  self
     */
    public function setName(string $name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * Set the value of amount
     *
     * @param  int  $amount
     *
     * @return  self
     */
    public function setAmount($amount)
    {
        $this->amount = intval($amount);

        return $this;
    }

    /**
     * Can save object
     *
     * @return bool
     */
    public function canBeSaved() {
        return isset($this->name) && isset($this->amount) && isset($this->discount_type) && isset($this->status);
    }

    /**
     * Set the value of created_at
     *
     * @param  string  $created_at
     *
     * @return  self
     */
    public function setCreatedAt(string $created_at)
    {
        $this->created_at = $created_at;

        return $this;
    }

    /**
     * Set the value of updated_at
     *
     * @param  string  $updated_at
     *
     * @return  self
     */
    public function setUpdatedAt(string $updated_at)
    {
        $this->updated_at = $updated_at;

        return $this;
    }

    /**
     * Set the value of productIds
     *
     * @param  array<int, int>  $productIds
     *
     * @return  self
     */
    public function setProductIds(array $productIds)
    {
        $this->productIds = array_map(function($value) {
            return intval($value);
        }, $productIds);

        return $this;
    }

    /**
     * Set the value of collectionIds
     *
     * @param  array<int, int>  $collectionIds
     *
     * @return  self
     */
    public function setCollectionIds(array $collectionIds)
    {
        $this->collectionIds = array_map(function($value) {
            return intval($value);
        }, $collectionIds);

        return $this;
    }

    /**
     * Set the value of status
     *
     * @param  int  $status
     *
     * @return  self
     */
    public function setStatus($status)
    {
        $this->status = intval($status);

        return $this;
    }
    
    /**
     * Get all products id's for this discount rule
     *
     * @return array<int, int>
     */
    public function getAllProducts() {
        $productIds = $this->productIds;
        if(!empty($this->collectionIds)) {
            $productRepo = new ProductRepository();
            $productIds = $productRepo->getProductIdsFromCollections($this->collectionIds);
        }

        return $productIds;
    }

    /**
     * Update price for products associated with this discount rule
     * If old products list is present we want to remove the discount price for this products
     *
     * @param  array $oldProductsList old products list array
     * @return void
     */
    public function updatePrice($oldProductsList = []) {

        $productIds = $this->getAllProducts();
        if(count($oldProductsList) > 0) {
            $removeDiscountProducts = array_diff($oldProductsList, $productIds);
            $this->dispatchJobForProducts($removeDiscountProducts, null);
        }
        
        if(empty($productIds)) {
            return;
        }

        $amount = $this->amount;
        if($this->status == DiscountStatus::Inactive) {
            $amount = null;
        }

        $this->dispatchJobForProducts($productIds, $amount);
    }
            
    /**
     * Cancel start schedule job
     *
     * @return void
     */
    public function cancelStartSchedule() {
        WC()->queue()->cancel_all(
            Jobs::START_TIME_SCHEDULE, 
            array(
             'id' => $this->id
            ),
            'napps_discount_rule_start_time_job_' . $this->id
        );
    }

    
    /**
     * Cancel end schedule job
     *
     * @return void
     */
    public function cancelEndSchedule() {
        WC()->queue()->cancel_all(
            Jobs::END_TIME_SCHEDULE, 
            array(
             'id' => $this->id
            ),
            'napps_discount_rule_end_time_job_' . $this->id
        );
    }
    
    /**
     * Schedule end date job
     * Return true if job was schedule or in case of end date in past, removes sale prices
     *
     * @return bool
     */
    public function scheduleEndDate() {
        if(!$this->end_date) {
            return false;
        }

        if($this->isEndDateInPast()) {
            return false;
        }
        
        $endDateUnix = strtotime($this->end_date);

        // Start time invalid
        if(!$endDateUnix || $endDateUnix  - time() < 0) {
            return false;
        }

        // Schedule job
        WC()->queue()->schedule_single(
            $endDateUnix,
            Jobs::END_TIME_SCHEDULE,
            array(
                'id' => $this->id
            ),
            'napps_discount_rule_end_time_job_' . $this->id
        );
    }

    /**
     * Send schedule job is start time is heigher that current date
     * true is job was schedule
     *
     * @return bool
     */
    public function scheduleStartDate() {
        $startDateUnix = strtotime($this->start_date);

        // Start time invalid or past 
        if($this->isStartDateInPast()) {
            return false;
        }

        $this->cancelStartSchedule();

        WC()->queue()->schedule_single(
            $startDateUnix,
            Jobs::START_TIME_SCHEDULE,
            array(
                'id' => $this->id
            ),
            'napps_discount_rule_start_time_job_' . $this->id
        );

        return true;
    }

    /**
     * Dispatch update job for products
     * If it has more that 50 products we need to do it in chunks (queue)
     *
     * @param  array $productIds
     * @param  int|null $amount
     * @return void
     */
    private function dispatchJobForProducts($productIds, $amount) {
        // If we have less then 50 products dont run it on a job
        // Takes longer (background tasks can take up to 60 seconds) and is not necessary
        if(count($productIds) <= 50) {
            $job = new UpdatePrice();
            $job->handle($productIds, $amount, $this->discount_type);
            return;
        }

        // Otherwise make chunks of 50 products and push each chunk to a job
        // Than can run in background
        foreach(array_chunk($productIds, 50) as $chunk) {
            WC()->queue()->add(
                Jobs::UPDATE_PRICE_ON_PRODUCTS,
                array(
                    'productIds' => $chunk,
                    'amount' => $amount,
                    'discountType' => $this->discount_type
                ),
                'napps_discount_rule_job_' . $this->id
            );
        }
    }

    /**
     * Set the value of startDate
     *
     * @param  string  $startDate
     *
     * @return  self
     */ 
    public function setStartDate($startDate)
    {
        $this->start_date = $startDate;

        return $this;
    }

    /**
     * Set the value of endDate
     *
     * @param  string|null  $endDate
     *
     * @return  self
     */ 
    public function setEndDate($endDate)
    {
        $this->end_date = $endDate;

        return $this;
    }
    
    /**
     * Check if end date is in past
     *
     * @return bool
     */
    public function isEndDateInPast() {
        if(!$this->end_date) {
            return false;
        }

        $endDateUnix = strtotime($this->end_date);
        return $endDateUnix && $endDateUnix  - time() < 0;
    }

    /**
     * Check if end date is in past
     *
     * @return bool
     */
    public function isStartDateInPast() {
        if(!$this->start_date) {
            return false;
        }

        $startDateUnix = strtotime($this->start_date);
        return $startDateUnix && $startDateUnix - time() < 0;
    }

    /**
     * Check if end date is in past
     *
     * @return bool
     */
    public function isStartDateInFuture() {
        if(!$this->start_date) {
            return false;
        }

        $startDateUnix = strtotime($this->start_date);
        return $startDateUnix && $startDateUnix - time() > 0;
    }
}
