<?php
namespace App\Events;

use App\Models\DiscountRule;
use App\Models\DiscountTypes;
use WC_Product_Variable;
use WC_Product;

class UpdatePrice {

    /**
     * Handle product update price
     * Set price on product and trigger his webhook
     *
     * @param  array<int, int> $productIds
     * @param  ?int $amount
     * @param  DiscountTypes $discountType
     * @return void
     */
    public function handle($productIds, $amount, $discountType) {

        // If we have a valid amount but is negative set it as 0
        // If amount is null, clear sale price field
        if($amount != null && $amount < 0) {
            $amount = 0;
        }

        foreach($productIds as $productId) {
            $product = wc_get_product( $productId );
            if(!$product) {
                continue;
            }

            if($product->is_type( 'variable' )) { // Variable product
                $this->handleVariableProduct($product, $amount, $discountType);
            } else if($product->is_type( 'simple' )) { // Simple product

                if($amount == null) {
                    $product->set_sale_price("");
                } else {
                    $product->set_sale_price( wc_clean( $this->calculateValue($discountType, $amount, $product->get_regular_price()) ) );
                }

                $product->save();
            }

            wc_delete_product_transients( $product->get_id() );
        }
    }

    /**
     * Handle price for variable product
     *
     * @param  WC_Product $product
     * @param  ?int $amount
     * @param  DiscountTypes $discountType
     * @return void
     */
    private function handleVariableProduct($product, $amount, $discountType) {

        // Get variation ids for this product
        $variations = get_posts(
            array(
                'post_parent'    => $product->get_id(),
                'posts_per_page' => -1,
                'post_type'      => 'product_variation',
                'fields'         => 'ids',
                'post_status'    => array( 'publish', 'private' ),
            )
        );

        foreach ( $variations as $variation_id ) {
            $variation = wc_get_product( $variation_id );
            if(!$variation) {
                continue;
            }

            if($amount == null) {
                $variation->set_sale_price("");
            } else {
                $variation->set_sale_price( wc_clean( $this->calculateValue($discountType, $amount, $variation->get_regular_price()) ) );
            }
            $variation->save();
        }

        WC_Product_Variable::sync( $product );
    }

    /**
     * Calculate final price based on discount type
     *
     * @param  DiscountTypes $discountType
     * @param  int $amountDiscount
     * @param  int $price
     * @return float
     */
    private function calculateValue($discountType, $amountDiscount, $price) {

        $price = floatval($price);
        
        switch($discountType) {
            case DiscountTypes::Fixed:
                $newPrice = $price - $amountDiscount;
                if($newPrice < 0) {
                    return 0;
                }
                return $newPrice;
            case DiscountTypes::Percentage:
                $newPercentageDiscount = round($price - ($price * ($amountDiscount / 100)), 2);
                if($newPercentageDiscount < 0) {
                    return 0;
                }
                return $newPercentageDiscount;
            default:
                return $amountDiscount;
        }
    }
}
