<?php
namespace App\Api;

use WP_Error;
use WP_REST_Server;
use WP_REST_Response;
use WP_REST_Controller;
use App\Models\DiscountRule;
use App\Models\DiscountTypes;
use App\Models\DiscountStatus;
use App\Adapter\DiscountRuleAdapter;
use App\Repository\ProductRepository;
use App\Repository\CategoryRepository;
use App\Repository\DiscountRulesRepository;

/**
 * Discount rule controller
 */
class DiscountRulesController extends WP_REST_Controller {

    /**
     * @var DiscountRulesRepository
     */
    private $discountRulesRepo;

    /**
     * @var DiscountRuleAdapter
     */
    private $discountRuleAdapter;

    /**
     * @var string
     */
    private $post_type = "napps_discount_rules";

    /**
     * @param  DiscountRulesRepository $discountRulesRepo
     * @return void
     */
    public function __construct($discountRulesRepo) {
        $this->namespace = 'napps-dr/v1';
        $this->discountRulesRepo = $discountRulesRepo;
        $this->discountRuleAdapter = new DiscountRuleAdapter();
    }

    /**
     * Register the routes
     *
     * @return void
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace,
            '/discountrules',
            array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_items' ),
                    'permission_callback' => array( $this, 'permissions_check' ),
                    'args'                => $this->get_collection_params(),
                ),
                array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
                'schema' => array( $this, 'get_item_schema' ),
            )
        );

        // Resources for a discount rule (read and update)
        register_rest_route(
            $this->namespace,
            '/discountrules/(?P<id>\d+)',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __( 'Unique identifier for the resource.', 'discount-rules-by-napps' ),
                        'type'        => 'integer',
                    ),
                ),
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_item' ),
                    'permission_callback' => array( $this, 'permissions_check' ),
                ),
                array(
					'methods'             => 'PUT',
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
                array(
					'methods'             => 'DELETE',
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
                'schema' => array( $this, 'get_item_schema' ),
            )
        );

        // Get target information for discount rule
        register_rest_route(
            $this->namespace,
            '/discountrules/(?P<id>\d+)/target',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __( 'Unique identifier for the resource.', 'discount-rules-by-napps' ),
                        'type'        => 'integer',
                    ),
                ),
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_discount_target' ),
                    'permission_callback' => array( $this, 'permissions_check' )
                ),
            )
        );
    }

    /**
     * Retrieves a discount rule by id
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_item ( $request ) {
        if ( empty( $request['id'] ) ) {
            /* translators: %s: post_type */
			return new WP_REST_Response(array('message' => sprintf( __( 'Cannot get existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 400);
		}

        $id = absint($request['id']);

        $item = $this->discountRulesRepo->getById($id);
        if($item === null) {
            /* translators: %s: post_type */
            return new WP_REST_Response(array('message' => sprintf( __( 'Cannot get existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 400);
        }

        return new WP_REST_Response($this->discountRuleAdapter->fromObject($item), 200);
    }

    /**
     * Retrieves a collection of items.
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_items( $request ) {
        $items = $this->discountRulesRepo->getAll();
        if($items === null) {
            /* translators: %s: post_type */
            return new WP_REST_Response(array('message' => sprintf( __( 'Cannot get existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 400);
        }

        $data = array();
        foreach($items as $item) {
            $data[] = $this->discountRuleAdapter->fromObject($item);
        }

        return new WP_REST_Response(array(
            "discount_rules" => $data
        ), 200);
    }

    /**
     * Retrieves targets for discount rule
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_discount_target( $request ) {
        if ( empty( $request['id'] ) ) {
            /* translators: %s: post_type */
            return new WP_REST_Response(array('message' => sprintf( __( 'Cannot get existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 400);
		}

        $id = absint($request['id']);

        // Get collection ids for discount rule
        $collections = $this->discountRulesRepo->getCollections($id);

        // Get products ids for discount rule
        $productIds = $this->discountRulesRepo->getProducts($id);

        $productRepo = new ProductRepository();
        $parsedProducts = $productRepo->getProducts($productIds);

        $categoryRepo = new CategoryRepository();
        $parsedCollections = $categoryRepo->getCategories($collections);

        return new WP_REST_Response(array(
            "products" => $parsedProducts,
            "collections" => $parsedCollections
        ), 200);
    }

    /**
	 * Delete discount rule
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
    public function delete_item ( $request ) {
        if ( empty( $request['id'] ) ) {
            /* translators: %s: post_type */
            return new WP_REST_Response(array('message' => sprintf( __( 'Cannot delete existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 400);
		}

        $id = absint($request['id']);
        $item = $this->discountRulesRepo->getById($id);
        if($item === null) {
            return new WP_REST_Response(array('message' => 'Could not retrieve data'), 404);
        }

        $discountRule = $this->discountRuleAdapter->fromObject($item);

        // If discount rule is active, set it has inactive and dispatch a job to remove discounted price
        if($discountRule->status == DiscountStatus::Active) {
            $discountRule->status = DiscountStatus::Inactive;
            $discountRule->updatePrice();
            $discountRule->cancelStartSchedule();
            $discountRule->cancelEndSchedule();
        }

        $this->discountRulesRepo->delete($discountRule->id);

        return new WP_REST_Response(null, 200);
    }

    /**
	 * Update discount rule
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
    public function update_item( $request ) {
        if ( empty( $request['id'] ) ) {
            /* translators: %s: post_type */
            return new WP_REST_Response(array('message' => sprintf( __( 'Cannot create existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 400);
		}

        if(!isset($request['start_date'])) {
            $request['start_date'] = gmdate("c");
        }

        $startDateUnix = strtotime($request['start_date']);
        if(isset($request['end_date'])) {
            $endDateUnix = strtotime($request['end_date']);

            if($startDateUnix === false || $endDateUnix === false) {
                return new WP_REST_Response(array('message' => 'Invalid start or end date'), 401);
            }

            if($endDateUnix < $startDateUnix) {
                return new WP_REST_Response(array('message' => 'Invalid end date'), 401);
            }
        }


        $id = absint($request['id']);
        $item = $this->discountRulesRepo->getById($id);
        if($item === null) {
            return new WP_REST_Response(array('message' => 'Could not retrieve data'), 404);
        }

        $discountRule = $this->discountRuleAdapter->fromObject($item);
        $existingProducts = $discountRule->getAllProducts();

        if ( isset( $request['name'] ) ) {
			$discountRule->setName( wp_filter_post_kses( $request['name'] ) );
		}

        if ( isset( $request['discount_type'] ) ) {
			$discountRule->setDiscountType( intval( $request['discount_type'] ) );
		}

        if ( isset( $request['amount'] ) ) {
			$discountRule->setAmount( intval( $request['amount'] ) );
		}

        if ( isset( $request['status'] ) ) {
			$discountRule->setStatus( intval( $request['status'] ) );
		}

        if ( isset( $request['products'] ) && is_array($request['products']) ) {
			$discountRule->setProductIds( $request['products'] );
            $discountRule->setCollectionIds([]);
		}

        if ( isset( $request['collections'] ) && is_array($request['collections']) ) {
			$discountRule->setCollectionIds( $request['collections'] );
            $discountRule->setProductIds([]);
		}

        $discountRule->setStartDate($request['start_date']);

        if(isset($request['end_date']) && $request['end_date']) {
            $discountRule->setEndDate($request['end_date']);
        } else {
            $discountRule->setEndDate(null);
        }

        // If end date is in past set status as inactive
        if($discountRule->isEndDateInPast() || $discountRule->isStartDateInFuture()) {
            $discountRule->setStatus(DiscountStatus::Inactive);
        }
        
        /** @var DiscountRule $discountRule */
        $discountRule = apply_filters( "rest_pre_update_{$this->post_type}", $discountRule, $request );

        $result = $this->discountRulesRepo->update($discountRule);
        if(!$result) {
            /* translators: %s: post_type */
            return new WP_REST_Response(array('message' => sprintf( __( 'Cannot update existing %s.', 'discount-rules-by-napps' ), $this->post_type )), 500);
        }

        $discountRule->cancelStartSchedule();
        $discountRule->cancelEndSchedule();

        $discountRule->scheduleStartDate();
        if($discountRule->end_date != null) {
            $discountRule->scheduleEndDate();
        }

        $discountRule->updatePrice($existingProducts);

        return new WP_REST_Response(
            array(
                'discount_rule' => $discountRule,
            ),
            200
        );
    }

    /**
	 * Create a single product.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function create_item( $request ) {

        $discountRule = new DiscountRule();

        if(!isset($request['start_date'])) {
            $request['start_date'] = gmdate("c");
        }

        $startDateUnix = strtotime($request['start_date']);
        if(isset($request['end_date'])) {
            $endDateUnix = strtotime($request['end_date']);

            if($startDateUnix === false || $endDateUnix === false) {
                return new WP_REST_Response(array('message' => 'Invalid start or end date'), 401);
            }

            if($endDateUnix < $startDateUnix) {
                return new WP_REST_Response(array('message' => 'Invalid end date'), 401);
            }
        }

        if ( isset( $request['name'] ) ) {
			$discountRule->setName( wp_filter_post_kses( $request['name'] ) );
		}

        if ( isset( $request['discount_type'] ) ) {
			$discountRule->setDiscountType( intval( $request['discount_type'] ) );
		}

        if ( isset( $request['amount'] ) ) {
			$discountRule->setAmount( intval( $request['amount'] ) );
		}

        if ( isset( $request['status'] ) ) {
			$discountRule->setStatus( intval( $request['status'] ) );
		}

        $discountRule->setStartDate($request['start_date']);

        if(isset($request['end_date']) && $request['end_date']) {
            $discountRule->setEndDate($request['end_date']);
        }

        // If end date is in past set status as inactive
        if($discountRule->isEndDateInPast()) {
            $discountRule->setStatus(DiscountStatus::Inactive);
        }
        
        $discountRule = apply_filters( "rest_pre_insert_{$this->post_type}", $discountRule, $request );

        if(!$discountRule->canBeSaved()) {
            return new WP_REST_Response(array('message' => 'Missing data'), 400);
        }

        $savedDiscountRule = $this->discountRulesRepo->save($discountRule);

        $savedDiscountRule = apply_filters( "rest_post_insert_{$this->post_type}", $savedDiscountRule, $request );

        return new WP_REST_Response($savedDiscountRule, 201);
    }


    /**
     * Checks if a given request has access to read the items.
     *
     * @param  WP_REST_Request $request Full details about the request.
     *
     * @return true|WP_Error True if the request has read access, WP_Error object otherwise.
     */
    public function permissions_check( $request ) {
        $user = get_current_user_id();
        return $user && current_user_can("manage_woocommerce");
    }

    /**
	 * Retrieves the discount rule schema, conforming to JSON Schema.
	 *
	 * @return array Item schema data.
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'discount_rules',
			'type'       => 'object',
			'properties' => array(
				'id'                 => array(
					'description' => __( 'Unique identifier for the resource.', 'discount-rules-by-napps' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'name'           => array(
					'description' => __( 'Discount rule name', 'discount-rules-by-napps' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
                    'required'    => true
				),
                'amount'           => array(
					'description' => __( 'Discount rule amount', 'discount-rules-by-napps' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
                    'required'    => true
				),
				'discount_type'               => array(
					'description' => __( 'Discount type for rule', 'discount-rules-by-napps' ),
					'type'        => 'integer',
                    'default'     => 0,
					'context'     => array( 'view', 'edit' ),
					'enum'        => array_keys( DiscountTypes::getValues() ),
				),
                'status'               => array(
					'description' => __( 'Discount status for rule', 'discount-rules-by-napps' ),
					'type'        => 'integer',
                    'default'     => 0,
					'context'     => array( 'view', 'edit' ),
					'enum'        => array_keys( DiscountStatus::getValues() ),
				),
                'products'         => array(
					'description' => __( 'List of products for discount rule', 'discount-rules-by-napps' ),
					'type'        => 'array',
					'context'     => array( 'edit' ),
					'items'       => array(
                        'type' => 'integer',
                    ),
				),
				'collections'            => array(
					'description' => __( 'List of collections.', 'discount-rules-by-napps' ),
					'type'        => 'array',
					'context'     => array( 'edit' ),
					'items'       => array(
                        'type' => 'integer',
                    ),
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}

    /**
	 * Retrieves the query params for collections.
	 *
	 * @return array Collection parameters.
	 */
	public function get_collection_params() {
		$query_params = parent::get_collection_params();

		$query_params['context']['default'] = 'view';

		$query_params['exclude'] = array(
			'description' => __( 'Ensure result set excludes specific IDs.', 'discount-rules-by-napps' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$query_params['include'] = array(
			'description' => __( 'Limit result set to specific IDs.', 'discount-rules-by-napps' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$query_params['offset'] = array(
			'description' => __( 'Offset the result set by a specific number of items.', 'discount-rules-by-napps' ),
			'type'        => 'integer',
		);

		$query_params['order'] = array(
			'default'     => 'asc',
			'description' => __( 'Order sort attribute ascending or descending.', 'discount-rules-by-napps' ),
			'enum'        => array( 'asc', 'desc' ),
			'type'        => 'string',
		);

		$query_params['orderby'] = array(
			'default'     => 'name',
			'description' => __( 'Sort collection by object attribute.', 'discount-rules-by-napps' ),
			'enum'        => array('id','name',),
			'type'        => 'string',
		);

		$query_params['discount_types'] = array(
			'description' => __( 'Limit result set to discount rules with one or more specific discount types.', 'discount-rules-by-napps' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
		);

		return apply_filters( "rest_{$this->post_type}_collection_params", $query_params );
	}
}
