<?php
namespace App\Api;

use WP_Error;
use WP_REST_Server;
use WP_REST_Response;
use WP_REST_Controller;
use App\Repository\ProductRepository;
use App\Repository\CategoryRepository;

/**
 * Discount rule controller
 */
class CategoryController extends WP_REST_Controller {

    /**
     * @var CategoryRepository
     */
    private $categoryRepo;

    /**
     * @var string
     */
    private $post_type = "napps_discount_rules_categories";

    /**
     * @param  ProductRepository $productRepo
     * @return void
     */
    public function __construct($categoryRepo) {
        $this->namespace = 'napps-dr/v1';
        $this->categoryRepo = $categoryRepo;
    }

    /**
     * Register the routes
     *
     * @return void
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace,
            '/products/categories',
            array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_items' ),
                    'permission_callback' => array( $this, 'permissions_check' ),
                    'args'                => $this->get_collection_params(),
                ),
            )
        );

        // Resources for a discount rule (read and update)
        register_rest_route(
            $this->namespace,
            '/products/categories/(?P<id>\d+)',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __( 'Unique identifier for the resource.' ),
                        'type'        => 'integer',
                    ),
                ),
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_item' ),
                    'permission_callback' => array( $this, 'permissions_check' ),
                ),
            )
        );
    }

    /**
     * Retrieves a product by id
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_item ( $request ) {
        if ( empty( $request['id'] ) ) {
            /* translators: %s: post_type */
			return new WP_Error( "rest_{$this->post_type}_not_exists", sprintf( __( 'Cannot get %s.', 'discount-rules-by-napps' ), $this->post_type ), array( 'status' => 400 ) );
		}

        $id = absint($request['id']);

        $item = $this->categoryRepo->getCategory($id);
        if($item === null) {
            return new WP_Error('rest_retrieve_error', 'Could not retrieve data', array('status' => 500));
        }

        $response = rest_ensure_response($item);

        return $response;
    }

    /**
     * Retrieves a collection of items.
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_items( $request ) {

        // All args are sanatized see get_collection_params for more info
        $page = $request['page'];
        $per_page = $request['per_page'];
        $search = $request['search'];

        $categories = [];
        if(!empty($search)) {
            $categories = $this->categoryRepo->searchCollections($search, $page, $per_page);
        } else {
            $categories = $this->categoryRepo->getAllCategories($page, $per_page);
        }

        $response = rest_ensure_response($categories);

        return $response;
    }

    /**
     * Checks if a given request has access to read the items.
     *
     * @param  WP_REST_Request $request Full details about the request.
     *
     * @return true|WP_Error True if the request has read access, WP_Error object otherwise.
     */
    public function permissions_check( $request ) {
        return get_current_user_id() && current_user_can("manage_woocommerce");
    }

    /**
	 * Retrieves the query params for collections.
	 *
	 * @return array Collection parameters.
	 */
	public function get_collection_params() {
		$query_params = parent::get_collection_params();

		$query_params['context']['default'] = 'view';

        $params['page'] = array(
			'description'        => __( 'Get page for resource', 'discount-rules-by-napps' ),
			'type'               => 'integer',
			'sanitize_callback'  => 'absint',
			'validate_callback'  => 'rest_validate_request_arg',
		);

        $params['per_page'] = array(
			'description'        => __( 'Get per page count for resource', 'discount-rules-by-napps' ),
			'type'               => 'integer',
			'sanitize_callback'  => 'absint',
			'validate_callback'  => 'rest_validate_request_arg',
		);

        $query_params['search'] = array(
			'description'       => __( 'Limit result set to category name.', 'discount-rules-by-napps' ),
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return apply_filters( "rest_{$this->post_type}_collection_params", $query_params );
	}
}
