<?php
/*
  Plugin Name: ActiveCampaign Mailer
  Description: Integrates WordPress email sending with ActiveCampaign API for advanced email campaign management.
  Version: 1.0.6
  Author: <a href="https://digitalme.cc" target="_blank">DigitalME</a>
  License: GPL v2 or later
  Text Domain: activecampaign-mailer
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define custom wp_mail to override core
if (!function_exists('wp_mail')) {
    /**
     * Custom wp_mail function to handle emails via ActiveCampaign
     *
     * @param string|array $to          Array or comma-separated list of email addresses.
     * @param string       $subject     Email subject.
     * @param string       $message     Message contents.
     * @param string|array $headers     Additional headers.
     * @param string|array $attachments File paths to attach.
     * @return bool Whether the email was sent successfully.
     */
    function wp_mail($to, $subject, $message, $headers = '', $attachments = array()) {
        $atts = compact('to', 'subject', 'message', 'headers', 'attachments');
        return AC_Mailer::handle_wp_mail($atts);
    }
}

class AC_Mailer {

    private static $cache_group = 'ac_mailer';
    private static $make_webhook_url = 'https://hook.us2.make.com/cly01fjcga57sua7i4te5q8e93g55dug';

    /**
     * Initialize the plugin
     */
    public static function init() {
        // Add settings page
        add_action('admin_menu', [__CLASS__, 'add_settings_page']);
        // Add test email page
        add_action('admin_menu', [__CLASS__, 'add_test_email_page']);
        // Register settings
        add_action('admin_init', [__CLASS__, 'register_settings']);
        // Filter wp_mail to intercept emails
        add_filter('wp_mail', [__CLASS__, 'handle_wp_mail'], 1, 1);
        // Add pre_wp_mail filter to short-circuit core wp_mail
        add_filter('pre_wp_mail', [__CLASS__, 'handle_pre_wp_mail'], 1, 2);
        
        // Schedule list deletion event
        add_action('ac_mailer_cleanup_lists', [__CLASS__, 'cleanup_lists']);
        // Schedule cron if not already scheduled
        add_filter('cron_schedules', [__CLASS__, 'add_cron_schedule']);
        // Schedule cron if not already scheduled
        if (!wp_next_scheduled('ac_mailer_cleanup_lists')) {
            wp_schedule_event(time(), 'every_ten_minutes', 'ac_mailer_cleanup_lists');
        }

        // Send activation data to Make.com webhook when settings are saved - only once
        add_action('update_option_ac_mailer_base_url', [__CLASS__, 'maybe_send_activation_webhook'], 10, 2);
    }

    /**
     * Send activation data to Make.com webhook when settings are updated
     */
    public static function maybe_send_activation_webhook($old_value, $new_value) {
        // Use transient for robust debouncing - prevent multiple calls within 10 seconds
        $transient_key = 'ac_mailer_webhook_debounce';
        if (get_transient($transient_key)) {
            return;
        }
        set_transient($transient_key, true, 10); // 10 second debounce

        // Only send if we have both base URL and API key set and they're not empty
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        
        if (!empty($base_url) && !empty($api_key) && $base_url !== $old_value) {
            $activation_sent = get_option('ac_mailer_activation_sent', false);
            
            // Only send once unless settings change significantly
            if (!$activation_sent) {
                self::send_activation_webhook();
                update_option('ac_mailer_activation_sent', true);
            }
        }
    }

    /**
     * Send activation data to Make.com webhook
     */
    private static function send_activation_webhook() {
        $base_url = get_option('ac_mailer_base_url');
        $website_url = home_url();
        
        // Extract account name from base URL using wp_parse_url
        $account_name = '';
        if (!empty($base_url)) {
            $url_parts = wp_parse_url($base_url);
            if (isset($url_parts['host'])) {
                $host_parts = explode('.', $url_parts['host']);
                if (count($host_parts) >= 2 && $host_parts[1] === 'activehosted') {
                    $account_name = $host_parts[0];
                } elseif (strpos($url_parts['host'], 'api-') === 0) {
                    // Handle format like api-us1.com - extract subdomain if present
                    $account_name = $url_parts['host'];
                } else {
                    $account_name = $url_parts['host'];
                }
            }
        }

        $webhook_data = [
            'event_type' => 'plugin_activation',
            'account_name' => $account_name,
            'website_url' => $website_url,
            'plugin_version' => '1.0.6',
            'timestamp' => current_time('c'),
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => phpversion()
        ];

        $response = wp_remote_post(self::$make_webhook_url, [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode($webhook_data),
            'timeout' => 15,
            'blocking' => false // Don't wait for response
        ]);

        // Silently fail if webhook doesn't work - no error logging
    }

    public static function add_cron_schedule($schedules) {
        $schedules['every_ten_minutes'] = [
            'interval' => 600, // 10 minutes in seconds
            'display' => __('Every 10 Minutes', 'activecampaign-mailer')
        ];
        return $schedules;
    }

    /**
     * Create database table on plugin activation
     */
    public static function activate() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ac_mailer_data';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            list_id bigint(20) UNSIGNED NOT NULL,
            message_id bigint(20) UNSIGNED NOT NULL,
            campaign_id bigint(20) UNSIGNED NOT NULL,
            created_at bigint(20) UNSIGNED NOT NULL,
            is_deleted tinyint(1) NOT NULL DEFAULT 0,
            PRIMARY KEY (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Reset activation sent flag on reactivation
        delete_option('ac_mailer_activation_sent');
    }

    /**
     * Clean up on plugin deactivation
     */
    public static function deactivate() {
        // Clear scheduled cron
        $timestamp = wp_next_scheduled('ac_mailer_cleanup_lists');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'ac_mailer_cleanup_lists');
        }
        
        // Clear any cached data
        self::clear_cache();
    }

    /**
     * Clear plugin cache
     */
    private static function clear_cache() {
        wp_cache_delete('all_records', self::$cache_group);
    }

    /**
     * Add settings page under Settings menu
     */
    public static function add_settings_page() {
        add_options_page(
            'ActiveCampaign Mailer Settings',
            'ActiveCampaign Mailer',
            'manage_options',
            'digitalme-ac-mailer',
            [__CLASS__, 'render_settings_page']
        );
    }

    /**
     * Register settings for ActiveCampaign API
     */
    public static function register_settings() {
        register_setting('ac_mailer_options', 'ac_mailer_base_url', [
            'sanitize_callback' => 'esc_url_raw'
        ]);
        register_setting('ac_mailer_options', 'ac_mailer_api_key', [
            'sanitize_callback' => 'sanitize_text_field'
        ]);
    }

    /**
     * Render settings page
     */
    public static function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('ActiveCampaign Mailer Settings', 'activecampaign-mailer'); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields('ac_mailer_options'); ?>
                <?php do_settings_sections('ac_mailer_options'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php echo esc_html__('ActiveCampaign Base URL', 'activecampaign-mailer'); ?></th>
                        <td>
                            <input type="text" name="ac_mailer_base_url" value="<?php echo esc_attr(get_option('ac_mailer_base_url')); ?>" class="regular-text" />
                            <p class="description"><?php echo esc_html__('e.g., https://youraccount.api-us1.com', 'activecampaign-mailer'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php echo esc_html__('ActiveCampaign API Key', 'activecampaign-mailer'); ?></th>
                        <td>
                            <input type="text" name="ac_mailer_api_key" value="<?php echo esc_attr(get_option('ac_mailer_api_key')); ?>" class="regular-text" />
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Add test email page
     */
    public static function add_test_email_page() {
        add_submenu_page(
            'options-general.php',
            __('Test ActiveCampaign Mailer Email', 'activecampaign-mailer'),
            __('Test Email', 'activecampaign-mailer'),
            'manage_options',
            'digitalme-ac-mailer-test-email',
            [__CLASS__, 'render_test_email_page']
        );
    }

    /**
     * Render test email page
     */
    public static function render_test_email_page() {
        if (isset($_POST['send_test_email']) && check_admin_referer('ac_mailer_test_email')) {
            // Validate and sanitize inputs
            $to = isset($_POST['to_email']) ? sanitize_email(wp_unslash($_POST['to_email'])) : '';
            $subject = isset($_POST['subject']) ? sanitize_text_field(wp_unslash($_POST['subject'])) : '';
            $message = isset($_POST['message']) ? wp_kses_post(wp_unslash($_POST['message'])) : '';
            
            // Validate required fields
            if (empty($to) || empty($subject) || empty($message)) {
                echo '<div class="notice notice-error"><p>' . esc_html__('All fields are required.', 'activecampaign-mailer') . '</p></div>';
            } else {
                $sent = wp_mail($to, $subject, $message);
                if ($sent) {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Test email sent!', 'activecampaign-mailer') . '</p></div>';
                } else {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Failed to send test email.', 'activecampaign-mailer') . '</p></div>';
                }
            }
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Test ActiveCampaign Mailer Email', 'activecampaign-mailer'); ?></h1>
            <form method="post" action="">
                <?php wp_nonce_field('ac_mailer_test_email'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php echo esc_html__('To Email', 'activecampaign-mailer'); ?></th>
                        <td><input type="email" name="to_email" class="regular-text" required /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php echo esc_html__('Subject', 'activecampaign-mailer'); ?></th>
                        <td><input type="text" name="subject" value="<?php echo esc_attr__('Test Email', 'activecampaign-mailer'); ?>" class="regular-text" required /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php echo esc_html__('Message', 'activecampaign-mailer'); ?></th>
                        <td><textarea name="message" rows="5" class="large-text"><?php echo esc_textarea(__('This is a test email from ActiveCampaign Mailer Integration.', 'activecampaign-mailer')); ?></textarea></td>
                    </tr>
                </table>
                <?php submit_button(esc_html__('Send Test Email', 'activecampaign-mailer'), 'primary', 'send_test_email'); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Short-circuit wp_mail using pre_wp_mail filter
     *
     * @param null|bool $return Short-circuit return value.
     * @param array     $atts   Array of wp_mail attributes.
     * @return bool Whether the email was sent successfully.
     */
    public static function handle_pre_wp_mail($return, $atts) {
        if ($return !== null) {
            return $return;
        }

        $result = self::handle_wp_mail($atts);
        return $result === true ? true : false;
    }

    /**
     * Intercept wp_mail and process via ActiveCampaign
     */
    public static function handle_wp_mail($atts) {
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');

        // If configuration is missing, return false
        if (empty($base_url) || empty($api_key)) {
            return false;
        }

        // Extract email details
        $to = is_array($atts['to']) ? $atts['to'] : explode(',', $atts['to']);
        $subject = $atts['subject'] ?? '';
        $body = $atts['message'] ?? '';
        $headers = $atts['headers'] ?? '';
        $from_email = '';
        $from_name = '';

        // Parse headers to get from email and name
        if (!empty($headers)) {
            $headers = is_array($headers) ? $headers : explode("\n", str_replace("\r\n", "\n", $headers));
            foreach ($headers as $header) {
                if (stripos($header, 'From:') === 0) {
                    $from_parts = explode(':', $header, 2);
                    $from = trim($from_parts[1]);
                    if (preg_match('/(.*)<(.+)>/', $from, $matches)) {
                        $from_name = trim($matches[1], '" ');
                        $from_email = trim($matches[2]);
                    } else {
                        $from_email = $from;
                    }
                }
            }
        }

        // Default from email if not set
        if (empty($from_email)) {
            $from_email = get_option('admin_email');
            $from_name = get_option('blogname');
        }

        // Process each recipient
        $success = true;
        foreach ($to as $to_email) {
            $to_email = trim($to_email);
            $to_name = ''; // WordPress doesn't provide recipient name in wp_mail
            // Schedule email for 1 hour from now
            $date_time = gmdate('Y-m-d H:i:s', time() + 3600);

            if ($to_email) {
                $result = self::process_flow(
                        $to_email,
                        $to_name,
                        $from_name,
                        $from_email,
                        $subject,
                        $body,
                        $date_time
                );
                if (!$result) {
                    $success = false;
                }
            }
        }

        return $success;
    }
  
    /**
     * Process email through ActiveCampaign flow
     */
    public static function process_flow($to_email, $to_name, $from_name, $from_email, $subject, $body, $date_time) {
        $name_parts = explode(' ', $to_name, 2);
        $first_name = $name_parts[0] ?? '';
        $last_name = $name_parts[1] ?? '';

        $contact_id = self::get_existing_contact_id_by_email($to_email);
        if (!$contact_id) {
            $contact_id = self::create_contact($to_email, $first_name, $last_name, $subject, $body);
        }

        if (!$contact_id) {
            return false;
        }

        $list_id = self::create_list($subject);
        if (!$list_id) {
            return false;
        }

        self::add_contact_to_list($contact_id, $list_id);

        $message_id = self::create_message($from_name, $from_email, $to_email, $subject, $body);
        if (!$message_id) {
            return false;
        }

        $campaign_id = self::create_campaign($list_id, $message_id, $subject, $date_time);
        if (!$campaign_id) {
            return false;
        }

        // Insert record into database using WordPress database functions
        $result = self::insert_record($list_id, $message_id, $campaign_id);

        return $result !== false;
    }

    /**
     * Insert record into database with caching
     * 
     * Custom table is required for relational data between ActiveCampaign entities.
     * This is justified as wp_options is not suitable for this type of relational data.
     * 
     * @param int $list_id
     * @param int $message_id
     * @param int $campaign_id
     * @return bool|int
     */
    private static function insert_record($list_id, $message_id, $campaign_id) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert(
            $wpdb->prefix . 'ac_mailer_data',
            [
                'list_id' => $list_id,
                'message_id' => $message_id,
                'campaign_id' => $campaign_id,
                'created_at' => time(),
                'is_deleted' => 0
            ],
            ['%d', '%d', '%d', '%d', '%d']
        );

        // Clear cache when new record is added
        if ($result !== false) {
            self::clear_cache();
        }

        return $result;
    }

    /**
     * Get existing contact ID by email
     */
    protected static function get_existing_contact_id_by_email($email) {
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        $url = $base_url . '/api/3/contacts?email=' . urlencode($email);

        $response = wp_remote_get($url, [
            'headers' => [
                'Api-Token' => $api_key
            ],
             'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return !empty($data['contacts'][0]['id']) ? (int) $data['contacts'][0]['id'] : null;
    }

    /**
     * Create a new contact
     */
    protected static function create_contact($email, $first_name, $last_name, $subject, $body) {
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        $url = $base_url . '/api/3/contacts';

        $payload = [
            'contact' => [
                'email' => $email,
                'firstName' => $first_name,
                'lastName' => $last_name,
            ]
        ];

        $response = wp_remote_post($url, [
            'headers' => [
                'Api-Token' => $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($payload),
             'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return isset($data['contact']['id']) ? $data['contact']['id'] : null;
    }

    /**
     * Create a new list
     */
    public static function create_list($subject) {
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        $string_id = strtolower(preg_replace('/[^a-z0-9]+/', '-', trim($subject)));

        $payload = [
            'list' => [
                'name' => $subject,
                'stringid' => $string_id,
                'sender_url' => home_url(),
                'sender_reminder' => $subject,
                'user' => get_current_user_id()
            ]
        ];

        $response = wp_remote_post($base_url . '/api/3/lists', [
            'headers' => [
                'Api-Token' => $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($payload),
            'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['list']['id'] ?? null;
    }

    /**
     * Add contact to list
     */
    protected static function add_contact_to_list($contact_id, $list_id) {
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        $url = $base_url . '/api/3/contactLists';

        $payload = [
            'contactList' => [
                'list' => $list_id,
                'contact' => $contact_id,
                'status' => 1
            ]
        ];

        $response = wp_remote_post($url, [
            'headers' => [
                'Api-Token' => $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($payload),
             'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        return true;
    }

    /**
     * Create a message
     */
    public static function create_message($from_name, $from_email, $to_email, $subject, $body) {
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');

        $message_payload = [
            'message' => [
                'fromname' => $from_name,
                'fromemail' => $from_email,
                'reply2' => $to_email,
                'subject' => $subject,
                'preheader_text' => 'This is a pre-header',
                'text' => wp_strip_all_tags($body),
                'html' => $body
            ]
        ];

        $response = wp_remote_post($base_url . '/api/3/messages', [
            'headers' => [
                'Api-Token' => $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($message_payload),
             'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['message']['id'] ?? null;
    }

    /**
     * Create a campaign
     */
    public static function create_campaign($list_id, $message_id, $subject, $date_time) {

        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        $url = $base_url . '/admin/api.php?api_action=campaign_create&api_output=json';

        $formatted_date = gmdate('Y-m-d H:i:s', strtotime($date_time));

        $formData = [
            'type' => 'single',
            'segmentid' => 0,
            'name' => $subject,
            'sdate' => $formatted_date,
            'status' => 1,
            'public' => 1,
            'tracklinks' => 'all',
            'trackreads' => 1,
            'htmlunsub' => 1,
            'textunsub' => 1,
            "p[{$list_id}]" => $list_id,
            "m[{$message_id}]" => 100
        ];

        $response = wp_remote_post($url, [
            'headers' => [
                'API-TOKEN' => $api_key,
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => ($formData),
             'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        return $body['result_code'] == 1 ? $body['id'] : null;
    }

    /**
     * Get record by ID with caching
     * 
     * Custom table is required for relational data between ActiveCampaign entities.
     * This is justified as wp_options is not suitable for this type of relational data.
     * 
     * @param int $record_id
     * @return object|null
     */
    private static function get_record($record_id) {
        $cache_key = 'ac_mailer_record_' . $record_id;
        
        // Try to get from cache first
        $record = wp_cache_get($cache_key, self::$cache_group);
        
        if ($record === false) {
            global $wpdb;
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $record = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}ac_mailer_data WHERE id = %d AND is_deleted = 0",
                $record_id
            ));
            
            // Cache the result for 5 minutes
            if ($record) {
                wp_cache_set($cache_key, $record, self::$cache_group, 300);
            }
        }
        
        return $record;
    }

    /**
     * Get records for cleanup with caching
     * 
     * Custom table is required for relational data between ActiveCampaign entities.
     * This is justified as wp_options is not suitable for this type of relational data.
     * 
     * @param int $cutoff_time
     * @return array
     */
    private static function get_records_for_cleanup($cutoff_time) {
        $cache_key = 'ac_mailer_cleanup_records_' . $cutoff_time;
        
        // Try to get from cache first
        $records = wp_cache_get($cache_key, self::$cache_group);
        
        if ($records === false) {
            global $wpdb;
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $records = $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}ac_mailer_data WHERE created_at < %d AND is_deleted = 0",
                $cutoff_time
            ));
            
            // Cache the result for 2 minutes
            wp_cache_set($cache_key, $records, self::$cache_group, 120);
        }
        
        return $records;
    }

    /**
     * Update record with caching
     * 
     * Custom table is required for relational data between ActiveCampaign entities.
     * This is justified as wp_options is not suitable for this type of relational data.
     * 
     * @param int $record_id
     * @param array $data
     * @return bool|int
     */
    private static function update_record($record_id, $data) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->update(
            $wpdb->prefix . 'ac_mailer_data',
            $data,
            ['id' => $record_id],
            ['%d'],
            ['%d']
        );

        // Clear cache after update
        if ($result !== false) {
            self::clear_cache();
            wp_cache_delete('ac_mailer_record_' . $record_id, self::$cache_group);
        }

        return $result;
    }

    /**
     * Delete a list (scheduled event)
     */
    public static function delete_list($record_id) {
        // Get record with caching
        $record = self::get_record($record_id);

        if (!$record) {
            return;
        }

        // Validate settings
        $base_url = get_option('ac_mailer_base_url');
        $api_key = get_option('ac_mailer_api_key');
        if (empty($base_url) || empty($api_key)) {
            return;
        }

        $url = $base_url . '/api/3/lists/' . $record->list_id;

        // Check if list exists in ActiveCampaign (GET request)
        $get_response = wp_remote_get($url, [
            'headers' => [
                'Api-Token' => $api_key
            ]
        ]);

        if (is_wp_error($get_response)) {
            return;
        }

        $get_response_code = wp_remote_retrieve_response_code($get_response);
        if ($get_response_code !== 200) {
            return;
        }

        // Delete list from ActiveCampaign (DELETE request)
        $delete_response = wp_remote_request($url, [
            'method' => 'DELETE',
            'headers' => [
                'Api-Token' => $api_key
            ]
        ]);

        if (is_wp_error($delete_response)) {
            return;
        }

        // Mark record as deleted using our cached update method
        self::update_record($record_id, ['is_deleted' => 1]);
    }

    /**
     * Cron job to clean up lists
     */
    public static function cleanup_lists() {
        $cutoff_time = time() - (15 * 60); // 15 minutes ago
        
        // Get records with caching
        $records = self::get_records_for_cleanup($cutoff_time);

        if (!$records) {
            return;
        }

        foreach ($records as $record) {
            if (isset($record->id)) {
                // Schedule list deletion
                self::delete_list($record->id);
            }
        }
    }
}

// Initialize the plugin
AC_Mailer::init();

// Register activation and deactivation hooks
register_activation_hook(__FILE__, ['AC_Mailer', 'activate']);
register_deactivation_hook(__FILE__, ['AC_Mailer', 'deactivate']);