<?php
/*
Plugin Name: Digital Signature Addon for Contact Form 7
Description: Converts Contact Form 7 into a signable form with a digital signature field for mouse and touchscreen devices.
Version: 1.0.0
Tested up to: 6.8
Requires PHP: 7.0
Requires Plugins: contact-form-7
Author: Akshar Soft Solutions
License: GPL v2 or later
License URI:https://www.gnu.org/licenses/gpl-2.0.html
Author URI: https://aksharsoftsolutions.com/
Text Domain: as-digital-signature
*/

defined('ABSPATH') || exit;

add_action('wp_enqueue_scripts', function () {
    wp_enqueue_style(
        'as-signature-style',
        plugin_dir_url(__FILE__) . 'assets/css/signature-style.css',
        [],
        '1.0.0'
    );

    wp_enqueue_script('as-signature-pad', plugin_dir_url(__FILE__) . 'assets/js/signature_pad.umd.min.js', [], '4.1.7', true);
    wp_enqueue_script('as-signature-field', plugin_dir_url(__FILE__) . 'assets/js/signature-field.js', ['jquery'], '1.0', true);
});

add_action('wpcf7_init', function () {
    wpcf7_add_form_tag(['as_signature', 'as_signature*'], 'as_signature_form_tag_handler', ['name-attr' => true]);
});

function as_signature_form_tag_handler($tag)
{
    $name = $tag->name;
    if (empty($name)) return '';

    $validation_error = wpcf7_get_validation_error($name);

    $class = wpcf7_form_controls_class($tag->type) . ' as-signature-input';
    $width = $tag->get_option('width')[0] ?? 300;
    $height = $tag->get_option('height')[0] ?? 150;
    $color = $tag->get_option('color')[0] ?? '#000000';
    $backcolor = $tag->get_option('backcolor')[0] ?? '#ffffff';

    $canvas_attrs = sprintf(
        'width="%d" height="%d" data-color="%s" data-backcolor="%s"',
        esc_attr($width),
        esc_attr($height),
        esc_attr($color),
        esc_attr($backcolor)
    );

    return sprintf(
        '<div class="as-signature-wrapper">
        <canvas class="as-signature-canvas" name="%1$s" %2$s></canvas>
        <input type="button" class="as-signature-clear" value="Clear" />
        <span class="wpcf7-form-control-wrap %1$s" data-name="%1$s">
            <input type="hidden" name="%1$s" class="%3$s" />
            <span class="wpcf7-not-valid-tip">%4$s</span>
        </span>
    </div>',
        esc_attr($name),
        $canvas_attrs,
        esc_attr($class),
        esc_html($validation_error)
    );
}


add_filter('wpcf7_validate_as_signature', 'as_signature_validation_filter', 10, 2);
add_filter('wpcf7_validate_as_signature*', 'as_signature_validation_filter', 10, 2);
function as_signature_validation_filter($result, $tag)
{
    $name = $tag['name'];

    if (empty($name)) {
        return $result;
    }

    $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';

    if (empty($nonce) || !wp_verify_nonce($nonce, 'as_signature_form')) {
        return $result;
    }


    $value = isset($_POST[$name]) ? sanitize_text_field(wp_unslash($_POST[$name])) : '';

    if ($tag->is_required() && empty(trim($value))) {
        $result->invalidate($tag, __('Please provide your signature.', 'as-digital-signature'));
    }

    return $result;
}



add_filter('wpcf7_posted_data', function ($posted_data) {
    foreach ($posted_data as $key => $data) {
        if (is_string($data) && strpos($data, 'data:image/png;base64') === 0) {
            $image = explode(',', $data)[1] ?? null;
            if (!$image) continue;

            $filename = sanitize_file_name($key . '-' . time() . '.png');
            $upload_dir = wp_upload_dir();
            $signature_dir = trailingslashit($upload_dir['basedir']) . 'as-signature/';
            $signature_url = trailingslashit($upload_dir['baseurl']) . 'as-signature/' . $filename;

            if (!file_exists($signature_dir)) {
                wp_mkdir_p($signature_dir);
                file_put_contents($signature_dir . '.htaccess', "Order deny,allow\nDeny from all\n<FilesMatch \"\.(png)$\">\nAllow from all\n</FilesMatch>");
            }

            file_put_contents($signature_dir . $filename, base64_decode($image));
            $posted_data[$key] = esc_url_raw($signature_url);
        }
    }
    return $posted_data;
});

add_action('wpcf7_admin_init', function () {
    wpcf7_add_tag_generator('as_signature', __('Digital Signature', 'as-digital-signature'), 'wpcf7-tg-pane-as-signature', 'as_signature_tag_generator', ['version' => 2]);
});

function as_signature_tag_generator($contact_form, $args = '')
{
    $args = wp_parse_args($args, array());
    $wpcf7_contact_form = WPCF7_ContactForm::get_current();
    $contact_form_tags = $wpcf7_contact_form->scan_form_tags();
    $type = 'signature';
    $description = __("Generate a form-tag for a signature field.", 'as-digital-signature');
?>
    <header class="description-box">
        <h3>signature form tag generator</h3>
        <p><?php echo sprintf(esc_html($description)); ?></p>
    </header>
    <div class="control-box">
        <fieldset>
            <legend>
                Field type
            </legend>
            <input type="hidden" data-tag-part="basetype" value="as_signature">
            <label>
                <input type="checkbox" data-tag-part="type-suffix" value="*">This is a required field.</label>
        </fieldset>

        <fieldset>
            <legend>Name</legend>
            <input type="text" data-tag-part="name" pattern="[A-Za-z][A-Za-z0-9_\-]*">
        </fieldset>
        <fieldset>
            <legend>Color</legend>
            <input type="color" data-tag-part="option" data-tag-option="color:" value="#000">
        </fieldset>
        <fieldset>
            <legend>Background Color</legend>
            <input type="color" data-tag-part="option" data-tag-option="backcolor:" value="#dddddd">
        </fieldset>
        <fieldset>
            <legend>Width</legend>
            <input type="number" value="300" data-tag-option="width:" data-tag-part="option">
        </fieldset>
        <fieldset>
            <legend>Height</legend>
            <input type="number" value="200" data-tag-option="height:" data-tag-part="option">
        </fieldset>
        <fieldset>
            <legend>Id</legend>
            <input type="text" data-tag-part="option" data-tag-option="id:" value="">
        </fieldset>
        <fieldset>
            <legend>Class</legend>
            <input type="text" data-tag-part="option" data-tag-option="class:" value="" pattern="[A-Za-z0-9_\-\s]*">
        </fieldset>

    </div>
    <div class="insert-box">
        <div class="flex-container">
            <input type="text" class="code" readonly="readonly" onfocus="this.select();" data-tag-part="tag">
            <div class="submitbox">
                <input type="button" class="button button-primary insert-tag" value="<?php echo esc_attr(__('Insert Tag', 'as-digital-signature')); ?>" />
            </div>
        </div />
        <p class="mail-tag-tip">
            <label for="<?php echo esc_attr($args['content'] . '-mailtag'); ?>">
                <?php
                //translators: %s: mail-tag placeholder (e.g., [as-signature])
                echo sprintf(
                    esc_html__(
                        'To use the value input through this field in a mail field, you need to insert the corresponding mail-tag (%s) into the field on the Mail tab.',
                        'as-digital-signature'
                    ),
                    '<strong><span class="mail-tag"></span></strong>'
                );

                ?>
            </label>
        </p>
    </div>
<?php
}
