<?php
// Simple packager: creates digital-product-passport-<version>.zip in ./dist

declare(strict_types=1);

// Ensure run from plugin root
$root = dirname(__DIR__);
chdir($root);

// Try to load WordPress to use core helpers when available (optional).
$wp_load = dirname($root, 3) . DIRECTORY_SEPARATOR . 'wp-load.php';
if (file_exists($wp_load)) {
    /** @psalm-suppress UnresolvableInclude */
    require_once $wp_load; // phpcs:ignore WordPressVIPMinimum.Files.IncludingFile.UsingCustomConstant
}

/**
 * Write a message to STDERR in CLI context.
 *
 * We intentionally use native I/O here. This runs outside WP requests,
 * so WP_Filesystem is not applicable for STDERR.
 *
 * @param string $message Message (include trailing newline).
 * @return void
 */
function dpp_cli_stderr(string $message): void {
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite -- CLI stderr write in build script.
    fwrite(STDERR, $message);
}

$mainFile = $root . DIRECTORY_SEPARATOR . 'digital-product-passport.php';
if (!file_exists($mainFile)) {
    dpp_cli_stderr("Cannot find digital-product-passport.php in repo root.\n");
    exit(1);
}

// Extract Version from plugin header (prefer WP's get_file_data when available)
$version = '';
if (function_exists('get_file_data')) {
    $data = get_file_data($mainFile, [ 'Version' => 'Version' ]);
    $version = isset($data['Version']) ? trim((string) $data['Version']) : '';
} else {
    $contents = file_get_contents($mainFile) ?: '';
    if (preg_match('/^\s*\*\s*Version:\s*([^\r\n]+)/mi', $contents, $m)) {
        $version = trim($m[1]);
    }
}
if ($version === '') {
    dpp_cli_stderr("Cannot read Version header from digital-product-passport.php\n");
    exit(1);
}

// Validate version format (allow x.y.z or semver-ish)
if (!preg_match('/^[0-9]+\.[0-9]+\.[0-9]+(?:[-+][A-Za-z0-9\.-]+)?$/', $version)) {
    dpp_cli_stderr("Version '{$version}' does not look like x.y.z; proceeding anyway.\n");
}

$slug = 'digital-product-passport';
$distDir = $root . DIRECTORY_SEPARATOR . 'dist';
if (!is_dir($distDir)) {
    if (function_exists('wp_mkdir_p')) {
        if (!wp_mkdir_p($distDir)) {
            dpp_cli_stderr("Failed to create dist directory at {$distDir}\n");
            exit(1);
        }
    } else {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Build script fallback when WP is unavailable.
        if (!mkdir($distDir) && !is_dir($distDir)) {
            dpp_cli_stderr("Failed to create dist directory at {$distDir}\n");
            exit(1);
        }
    }
}

$zipPath = $distDir . DIRECTORY_SEPARATOR . sprintf('%s-%s.zip', $slug, $version);

// Build list of files to include
$rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS));
$include = [];

// Exclude dev or irrelevant paths
$excludeDirs = [
    $root . DIRECTORY_SEPARATOR . '.git',
    $root . DIRECTORY_SEPARATOR . '.github',
    $root . DIRECTORY_SEPARATOR . 'node_modules',
    $root . DIRECTORY_SEPARATOR . 'vendor', // no runtime deps here
    $root . DIRECTORY_SEPARATOR . 'scripts', // exclude packager script
    $root . DIRECTORY_SEPARATOR . 'dist',
];
$excludeNames = [
    '.DS_Store',
    'Thumbs.db',
];

foreach ($rii as $file) {
    $path = $file->getPathname();
    $rel  = substr($path, strlen($root) + 1);

    // Skip excluded directories
    $skip = false;
    foreach ($excludeDirs as $ex) {
        if (strpos($path, $ex) === 0) { $skip = true; break; }
    }
    if ($skip) { continue; }

    // Skip excluded names
    if (in_array($file->getFilename(), $excludeNames, true)) { continue; }

    // Only include files within the plugin (not parent)
    if ($file->isFile()) {
        $include[] = $rel;
    }
}

// Create zip
$zip = new ZipArchive();
if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
    dpp_cli_stderr("Failed to create zip at {$zipPath}\n");
    exit(1);
}

// Add files under a top-level folder named after slug
foreach ($include as $rel) {
    $localName = $slug . '/' . $rel;
    $zip->addFile($root . DIRECTORY_SEPARATOR . $rel, $localName);
}

$zip->close();

$created_msg = 'Created: ' . $zipPath;
if (function_exists('esc_html')) {
    echo esc_html($created_msg) . PHP_EOL;
} else {
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CLI output in build script, not rendered in browser.
    echo $created_msg . PHP_EOL;
}

